/** @file tacunit.c
 *  @brief View or set TAC data units.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Get the data units of TAC files, or optionally set or convert the units.",
  " ",
  "Usage: @P [options] filename(s)",
  " ",
  "Options:",
  " -xset=<unit>",
  "     Set x (sample time) unit; x values are however not changed.",
  " -yset=<unit>",
  "     Set y (concentration) unit; y values are however not changed.",
  " -xconv=<unit>",
  "     Conversion of x (sample time) values to specified unit.", 
  " -yconv=<unit>",
  "     Conversion of y (concentration) values to specified unit.", 
  " -x",
  "     Print only the unit of x in stdout.",
  " -y",
  "     Print only the unit of y in stdout.",
  " -density=<Density (g/mL)>",
  "     Density is needed when converting per mass to per volume units.",
  " -units",
  "     List available units and quit.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: convert sample times to minutes and concentrations to kBq/mL",
  "     @P -xconv=min -yconv=kBq/ml iea*.tac",
  " ",
  "Example 2: convert concentrations to the same units as in another file",
  "           (in Linux/UNIX/OS X)",
  "     unit=$( @P -y iea2285.tac )",
  "     @P -yconv=$unit iea2285.bld",
  " ",
  "See also: taclist, tac2suv, taccalc, imgunit, tacframe, dftmax",
  " ",
  "Keywords: TAC, tool, unit, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int list_units=0; // 1=list available units and exit
  int print_x=0, print_y=0; // 1=print unit without anything else
  int newxunit=-1, newyunit=-1;
  int oldxunit=-1, oldyunit=-1;
  int change_x=0; // 0=not changed, 1=also values are changed
  int change_y=0; // 0=not changed, 1=also values are changed
  double density=nan("");
  char tacfile[FILENAME_MAX];
  TAC tac;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "xset=", 5)==0) {
      cptr+=5;
      newxunit=unitIdentify(cptr); if(newxunit!=UNIT_UNKNOWN) continue;
      if(!cptr[0] || strcasecmp(cptr, "UNKNOWN")==0) continue;
    } else if(strncasecmp(cptr, "yset=", 5)==0) {
      cptr+=5;
      newyunit=unitIdentify(cptr); if(newyunit!=UNIT_UNKNOWN) continue;
      if(!cptr[0] || strcasecmp(cptr, "UNKNOWN")==0) continue;
    } else if(strncasecmp(cptr, "xconv=", 6)==0) {
      cptr+=6; change_x=1;
      newxunit=unitIdentify(cptr); if(newxunit!=UNIT_UNKNOWN) continue;
      if(!cptr[0] || strcasecmp(cptr, "UNKNOWN")==0) continue;
    } else if(strncasecmp(cptr, "yconv=", 6)==0) {
      cptr+=6; change_y=1;
      newyunit=unitIdentify(cptr); if(newyunit!=UNIT_UNKNOWN) continue;
      if(!cptr[0] || strcasecmp(cptr, "UNKNOWN")==0) continue;
    } else if(strcasecmp(cptr, "x")==0) {
      print_x=1; continue;
    } else if(strcasecmp(cptr, "y")==0) {
      print_y=1; continue;
    } else if(strcasecmp(cptr, "UNITS")==0) {
      list_units=1; continue;
    } else if(strncasecmp(cptr, "DENSITY=", 8)==0) {
      if(atofCheck(cptr+8, &density)==0 && density>0.0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("list_units := %d\n", list_units);
    printf("print_x := %d\n", print_x);
    printf("print_y := %d\n", print_y);
    printf("change_x := %d\n", change_x);
    printf("change_y := %d\n", change_y);
    if(!isnan(density)) printf("density := %g\n", density);
    if(newxunit>=0) printf("newxunit := %s\n", unitName(newxunit));
    if(newyunit>=0) printf("newyunit := %s\n", unitName(newyunit));
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* If units are to be listed, then do that and exit */
  if(list_units) {
    for(int u=0; u<UNIT_LAST; u++) printf("%s\n", unitName(u));
    return 0;
  }

  /* Process other arguments, starting from the first non-option */
  int n=0, nr_of_changes, err_nr=0;
  for(; ai<argc; ai++) {
    /* It should be the TAC filename */
    strcpy(tacfile, argv[ai]);
    /* Read TAC file */
    if(verbose>1) printf("reading %s\n", tacfile);
    ret=tacRead(&tac, tacfile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); return(2);
    }
    n++;
    /* Save original units */
    oldxunit=tac.tunit;
    oldyunit=tac.cunit;
    /* If clean printing was required, then do just that */
    if(print_x || print_y) {
      if(print_x) fprintf(stdout, "%s\n", unitName(tac.tunit));
      if(print_y) fprintf(stdout, "%s\n", unitName(tac.cunit));
      fflush(stdout); tacFree(&tac); continue;
    }
    /* If new units were not set then just print the units */
    if(newxunit==-1 && newyunit==-1) {
      fprintf(stdout, "# filename := %s\n", tacfile);
      fprintf(stdout, "unit := %s\n", unitName(tac.cunit));
      fprintf(stdout, "timeunit := %s\n", unitName(tac.tunit));
      /* do not save the data, since it is not modified */
      fflush(stdout); tacFree(&tac); continue;
    }
    nr_of_changes=0;
    /* X unit */
    if(newxunit>=0 && newxunit!=oldxunit) {
      if(change_x==0) {
        tac.tunit=newxunit;
        if(verbose>0) fprintf(stdout, "%s: units '%s' replaced by '%s'\n",
                              tacfile, unitName(oldxunit), unitName(newxunit));
        nr_of_changes++;
      } else {
        ret=tacXUnitConvert(&tac, newxunit, &status);
        if(ret==TPCERROR_OK) {
          if(verbose>0) fprintf(stdout, "%s: conversion of units '%s' to '%s'\n",
                                tacfile, unitName(oldxunit), unitName(newxunit));
          nr_of_changes++;
        } else {
          err_nr++;
          fprintf(stderr, "%s:\n", tacfile);
          fprintf(stderr, "Error: conversion from '%s' to '%s' not supported.\n",
                  unitName(oldxunit), unitName(newxunit));
        } 
      }
      fflush(stdout);
    } 
    /* Y unit */
    if(newyunit>=0 && newyunit!=oldyunit) {
      if(change_y==0) {
        tac.cunit=newyunit;
        if(verbose>0) fprintf(stdout, "%s: units '%s' replaced by '%s'\n",
                              tacfile, unitName(oldyunit), unitName(newyunit));
        nr_of_changes++;
      } else {
        ret=tacYUnitConvert(&tac, newyunit, &status);
        if(ret==TPCERROR_OK) {
          if(verbose>0) fprintf(stdout, "%s: conversion of units '%s' to '%s'\n",
                                tacfile, unitName(oldyunit), unitName(newyunit));
          nr_of_changes++;
        } else if(!isnan(density)) {
          /* Failed, but check if converting gram to mL or vice versa helps */
          if(unitDividerHasVolume(oldyunit) && unitDividerHasMass(newyunit)) {
            ret=tacYUnitVolume2Mass(&tac, density, &status);
          } else if(unitDividerHasMass(oldyunit) && unitDividerHasVolume(newyunit)) {
            ret=tacYUnitMass2Volume(&tac, density, &status);
          }
          if(verbose>1 && ret==TPCERROR_OK)
            fprintf(stdout, "%s: initial conversion of units '%s' to '%s'\n",
                                tacfile, unitName(oldyunit), unitName(tac.cunit));
          /* Try conversion again */
          ret=tacYUnitConvert(&tac, newyunit, &status);
          if(ret==TPCERROR_OK) {
            if(verbose>0) fprintf(stdout, "%s: conversion of units '%s' to '%s'\n",
                                  tacfile, unitName(oldyunit), unitName(newyunit));
            nr_of_changes++;
          }
        }
        if(ret!=TPCERROR_OK) {
          err_nr++;
          fprintf(stderr, "%s:\n", tacfile);
          fprintf(stderr, "Error: conversion from '%s' to '%s' not supported.\n",
                  unitName(oldyunit), unitName(newyunit));
        }
      }
      fflush(stdout);
    }
    /* No reason to save data, if nothing was changed */
    if(nr_of_changes==0) {tacFree(&tac); continue;}
    /* Save modified data */
    if(verbose>1) printf("writing %s\n", tacfile);
    FILE *fp; fp=fopen(tacfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
      tacFree(&tac); return(5);
    }
    ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(6);
    }
    tacFree(&tac);
  } // next file
  tacFree(&tac);

  if(n==0) fprintf(stderr, "Error: no TAC(s) were found.\n");
  if(n==0 || err_nr>0) return(2);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
