/** @file ana_lhdr.c
 *  @brief List the information in Analyze 7.5 header file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the information in Analyze 7.5 image header.",
  " ",
  "Usage: @P headerfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "   @P a8765.hdr",
  " ",
  "See also: ana2ecat, ecat2ana, ana_ehdr, anabyteo, nii_lhdr",
  " ",
  "Keywords: image, Analyze, header, IFT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int         ai, help=0, version=0, verbose=1;
  int         ret;
  char        hdrfile[FILENAME_MAX], *cptr;
  ANALYZE_DSR dsr;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  hdrfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!hdrfile[0]) {strcpy(hdrfile, argv[ai]); continue;}
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!hdrfile[0]) {
    fprintf(stderr, "Error: Analyze header was not given.\n"); 
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("hdrfile := %s\n", hdrfile);
  }
  if(verbose>2) ANALYZE_TEST=verbose-2; else ANALYZE_TEST=0;



  /*
   *  Read Analyze header file
   */
  /* Check that filename contains .hdr extension */
  cptr=strrchr(hdrfile, '.');
  if(cptr==NULL || strcasecmp(cptr, ".hdr")!=0) strcat(hdrfile, ".hdr");
  /* Read header file */
  if(verbose>1) printf("reading %s\n", hdrfile);
  ret=anaReadHeader(hdrfile, &dsr);
  if(ret) {
    fprintf(stderr, "Error: cannot read %s.\n", hdrfile);
    if(verbose>1) fprintf(stderr, "  ret := %d\n", ret);
    return(2);
  }

  /*
   *  Print Analyze header
   */
  anaPrintHeader(&dsr, stdout);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
