/** @file e63mdel.c
 *  @brief Delete specified matrix (plane and frame) from ECAT 6.3 image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Delete specified image matrix (frames and plane) in ECAT 6.3 file.",
  "Note! Missing matrices usually prevent the usage of the image in most software.",
  " ",
  "Usage: @P [Options] imgfile frame plane outputfile ",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P a2345dy1.img 20 10 a2345dy1_partial.img ",
  " ",
  "See also: e63mreg, esplit, imgdelfr, lmlist, eframe, efixplnr, imgadd",
  " ",
  "Keywords: ECAT, matrixlist, cropping, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 ret;
  int                 plane=0, frame=0;
  char                ecatfile[FILENAME_MAX], outfile[FILENAME_MAX];
  ECAT63_mainheader   mainheader63;
  MATRIXLIST          mlist63;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=outfile[0]=(char)0;
  ecat63InitMatlist(&mlist63);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(ecatfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    if(atoi_with_check(argv[ai++], &frame) || frame<1) {
      fprintf(stderr, "Error: invalid frame number.\n"); return(1);}
  }
  if(ai<argc) {
    if(atoi_with_check(argv[ai++], &plane) || plane<1) {
      fprintf(stderr, "Error: invalid plane number.\n"); return(1);}
  }
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: extra argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }

  /* Check output filename */
  if(strcasecmp(ecatfile, outfile)==0) {
    fprintf(stderr, "Error: same name for input and output file.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
    printf("outputfile := %s\n", outfile);
    printf("frame := %d\n", frame);
    printf("plane := %d\n", plane);
  }


  /*
   *  Read main header and matrix list
   */
  if(verbose>1) {fprintf(stdout, "reading %s\n", ecatfile); fflush(stdout);}

  /* Open file for read */
  FILE *fp;
  if((fp=fopen(ecatfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot read file %s\n", ecatfile);
    return(2);
  }

  /* Try to read ECAT 6.3 main header */
  if(verbose>2) {fprintf(stdout, "reading main header\n"); fflush(stdout);}
  ret=ecat63ReadMainheader(fp, &mainheader63);
  if(ret) {
    fprintf(stderr, "Error: cannot read main header.\n");
    if(verbose>1) printf("  ret := %d\n", ret);
    fclose(fp); return(2);
  }

  /* Read matrix list */
  ret=ecat63ReadMatlist(fp, &mlist63, verbose-1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot read matrix list.\n");
    if(verbose>1) printf("  ret := %d\n", ret);
    fclose(fp); return(2);
  }
  if(mlist63.matrixNr<1) {
    fprintf(stderr, "Error: matrix list is empty.\n");
    fclose(fp); return(2);
  }
  ret=ecat63CheckMatlist(&mlist63);
  if(ret!=0) {
    fprintf(stderr, "Warning: matrix list fails testing (%d).\n", ret);
    if(verbose>2) ecat63PrintMatlist(&mlist63);
  } else {
    if(verbose>100) ecat63PrintMatlist(&mlist63);
  }


  /*
   *  Mark listed matrix as deleted
   */
  int delNr=0, planeNr=0, frameNr=0;
  Matval matval63;
  for(int m=0; m<mlist63.matrixNr; m++) {
    mat_numdoc(mlist63.matdir[m].matnum, &matval63);
    if(matval63.plane==plane && matval63.frame==frame) {
      mlist63.matdir[m].matstat=-1;
      delNr++;
      continue;
    }
    if(matval63.plane>planeNr) planeNr=matval63.plane;
    if(matval63.frame>frameNr) frameNr=matval63.frame;
  }
  if(delNr==0) {
    fprintf(stderr, "Error: specified matrix was not found.\n");
    ecat63EmptyMatlist(&mlist63);
    fclose(fp); return(3);
  }
  if(delNr==mlist63.matrixNr) {
    fprintf(stderr, "Error: no matrices left after deleting this matrix.\n");
    ecat63EmptyMatlist(&mlist63);
    fclose(fp); return(4);
  }
  /* Fix main header contents accordingly */
  if(mainheader63.num_planes>planeNr) mainheader63.num_planes=planeNr;
  if(mainheader63.num_frames>frameNr) mainheader63.num_frames=frameNr;


  /*
   *  Write output file
   */
  /* Check if output file exists; rename to backup file, if necessary */
  //backupExistingFile(outfile, NULL, tmp);

  /* Open output file */
  FILE *fp2;
  fp2=ecat63Create(outfile, &mainheader63);
  if(fp2==NULL) {
    fprintf(stderr, "Error: cannot write ECAT file.\n");
    ecat63EmptyMatlist(&mlist63); fclose(fp); return(11);
  }
  /* Copy the matrices */
  int blkNr, nxtblk;
  char buf[MatBLKSIZE];
  for(int m=0; m<mlist63.matrixNr; m++) if(mlist63.matdir[m].matstat==1) {
    blkNr=1+mlist63.matdir[m].endblk-mlist63.matdir[m].strtblk;
    /* Get block number for matrix header and data */
    nxtblk=ecat63Matenter(fp2, mlist63.matdir[m].matnum, blkNr-1);
    if(nxtblk<1) {
      fprintf(stderr, "Error: cannot write ECAT matrix.\n");
      ecat63EmptyMatlist(&mlist63); fclose(fp); fclose(fp2); remove(outfile); return(13);
    }
    if(verbose>3) printf("  m=%d blkNr=%d nxtblk=%d\n", m, blkNr, nxtblk);
    /* Copy each block */
    for(long long bi=mlist63.matdir[m].strtblk; bi<=mlist63.matdir[m].endblk; bi++) {
      /* Read block */
      fseeko(fp, (bi-1)*MatBLKSIZE, SEEK_SET);
      if(ftello(fp)!=(bi-1)*MatBLKSIZE) {
        fprintf(stderr, "Error: cannot find matrix block %lld.\n", bi);
        ecat63EmptyMatlist(&mlist63); fclose(fp); fclose(fp2); remove(outfile); return(8);
      }
      if(fread(buf, MatBLKSIZE, 1, fp)<1) {
        fprintf(stderr, "Error: cannot read matrix block %lld.\n", bi);
        ecat63EmptyMatlist(&mlist63); fclose(fp); fclose(fp2); remove(outfile); return(9);
      }
      /* Write block */
      fseeko(fp2, (long long)(nxtblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftello(fp2)!=(long long)(nxtblk-1)*MatBLKSIZE) {
        fprintf(stderr, "Error: cannot find matrix block %d for write.\n", nxtblk);
        ecat63EmptyMatlist(&mlist63); fclose(fp); fclose(fp2); remove(outfile); return(15);
      }
      if(fwrite(buf, 1, MatBLKSIZE, fp2)<1) {
        fprintf(stderr, "Error: cannot write matrix block %d.\n", nxtblk);
        ecat63EmptyMatlist(&mlist63); fclose(fp); fclose(fp2); remove(outfile); return(16);
      }
      nxtblk++;
    }
  }

  /* Close files */
  fclose(fp); fclose(fp2);
  ecat63EmptyMatlist(&mlist63);

  if(verbose>0) fprintf(stdout, "done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
