/** @file e63mreg.c
 *  @brief Fill missing matrices in ECAT 6.3 image with zero images.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Regularize ECAT 6.3 image with missing matrices by adding planes/frames",
  "with zero pixel values.",
  " ",
  "Usage: @P [Options] imgfile outputfile ",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P a2345dy1.img a2345dy1_fixed.img ",
  " ",
  "See also: esplit, imgdelfr, lmlist, eframe, efixplnr, e63mdel",
  " ",
  "Keywords: ECAT, matrixlist, header, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 ret;
  char                ecatfile[FILENAME_MAX], outfile[FILENAME_MAX];
  ECAT63_mainheader   mainheader63;
  MATRIXLIST          mlist63;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=outfile[0]=(char)0;
  ecat63InitMatlist(&mlist63);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(ecatfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: extra argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }

  /* Check output filename */
  if(strcasecmp(ecatfile, outfile)==0) {
    fprintf(stderr, "Error: same name for input and output file.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
    printf("outputfile := %s\n", outfile);
    fflush(stdout);
  }


  /*
   *  Read main header and matrix list
   */
  if(verbose>1) {fprintf(stdout, "reading %s\n", ecatfile); fflush(stdout);}

  /* Open file for read */
  FILE *fp;
  if((fp=fopen(ecatfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot read file %s\n", ecatfile);
    return(2);
  }

  /* Try to read ECAT 6.3 main header */
  if(verbose>2) {fprintf(stdout, "reading main header\n"); fflush(stdout);}
  ret=ecat63ReadMainheader(fp, &mainheader63);
  if(ret) {
    fprintf(stderr, "Error: cannot read main header.\n");
    if(verbose>1) printf("  ret := %d\n", ret);
    fclose(fp); return(2);
  }


  /* Read matrix list */
  ret=ecat63ReadMatlist(fp, &mlist63, verbose-5);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot read matrix list.\n");
    if(verbose>1) printf("  ret := %d\n", ret);
    fclose(fp); return(2);
  }
  if(mlist63.matrixNr<1) {
    fprintf(stderr, "Error: matrix list is empty.\n");
    fclose(fp); return(2);
  }
  if(verbose>1) {printf("matrixNr := %d\n", mlist63.matrixNr); fflush(stdout);}
  ret=ecat63CheckMatlist(&mlist63);
  if(ret!=0) {
    fprintf(stderr, "Warning: matrix list fails testing (%d).\n", ret); fflush(stderr);
    if(verbose>2) ecat63PrintMatlist(&mlist63);
  } else {
    if(verbose>100) ecat63PrintMatlist(&mlist63);
  }
  /* Trim extra frames */
  if(mainheader63.num_frames>0) {
    int del_nr=ecat63DeleteLateFrames(&mlist63, mainheader63.num_frames);
    if(verbose>1 && del_nr>0)
      printf("  %d entries in matrix list will not be used.\n", del_nr);
  }

  /* Get matrix values */
  Matval *matval;
  /* Allocate memory for matrix values */
  matval=(Matval*)calloc(mlist63.matrixNr, sizeof(Matval));
  if(matval==NULL) {
    fprintf(stderr, "Error: cannot allocate memory for matrix values.\n");
    fclose(fp); ecat63EmptyMatlist(&mlist63);
    return(3);
  }
  /* ... and get the matrix values from valid matrices */
  ECAT63_imageheader image_header;
  char errmsg[128];
  int validMatrixNr=0;
  for(int i=0; i<mlist63.matrixNr; i++) {
    /* check matrix status */
    if(mlist63.matdir[i].matstat!=1) continue;
    /* check that sub header can be read */
    if(ecat63ReadImageheader(fp, mlist63.matdir[i].strtblk, &image_header, verbose-10, errmsg)) {
      if(verbose>1) {
        printf("  cannot read subheader for matrix %d: %s\n", mlist63.matdir[i].matnum, errmsg);
        fflush(stdout);
      }
      mlist63.matdir[i].matstat=0; // mark it as failed
      continue;
    }
    mat_numdoc(mlist63.matdir[i].matnum, matval+validMatrixNr);
    validMatrixNr++;
  }
  if(verbose>1) {printf("validMatrixNr := %d\n", validMatrixNr); fflush(stdout);}
  if(validMatrixNr<4) {
    if(validMatrixNr==0) fprintf(stderr, "Error: no valid content in any matrix.\n");
    else fprintf(stderr, "Error: valid content in only %d matrices; not processed.\n", validMatrixNr);
    fclose(fp); ecat63EmptyMatlist(&mlist63); free(matval);
    return(4);
  }

  /* Check that there is only one gate and bed */
  int gate, bed, matdata;
  {
    if(verbose>2) {fprintf(stdout, "checking for gates and bed positions\n"); fflush(stdout);}
    gate=matval[0].gate;
    for(int i=1; i<validMatrixNr; i++) 
      if(matval[i].gate!=gate) {
        fprintf(stderr, "Error: more than one gate.\n");
        fclose(fp); ecat63EmptyMatlist(&mlist63); free(matval);
        return(5);
      }
    bed=matval[0].bed;
    for(int i=1; i<validMatrixNr; i++) 
      if(matval[i].bed!=bed) {
        fprintf(stderr, "Error: more than one bed position.\n");
        fclose(fp); ecat63EmptyMatlist(&mlist63); free(matval);
        return(5);
      }
    matdata=matval[0].data;
    for(int i=1; i<validMatrixNr; i++) 
      if(matval[i].data!=matdata) {
        fprintf(stderr, "Error: more than one data.\n");
        fclose(fp); ecat63EmptyMatlist(&mlist63); free(matval);
        return(5);
      }
  }

  /* Make list of existing plane numbers */
  INTEGER_LIST planes; integerListInit(&planes);
  for(int i=0; i<validMatrixNr; i++) integerListAdd(&planes, matval[i].plane, 1);
  integerListSort(&planes);
  if(verbose>3 && planes.nr>0) {
    printf("planes: %d", planes.list[0]);
    for(int i=1; i<planes.nr; i++) printf(", %d", planes.list[i]);
    printf("\n"); fflush(stdout);
  }
  if(planes.nr<1) {
    fprintf(stderr, "Error: invalid plane numbers.\n");
    fclose(fp); ecat63EmptyMatlist(&mlist63); free(matval); integerListEmpty(&planes);
    return(6);
  }

  /* Make list of frames for each plane */
  if(verbose>2) {fprintf(stdout, "listing each plane for its frames\n"); fflush(stdout);}
  INTEGER_LIST *plframes=(INTEGER_LIST*)malloc(planes.nr*sizeof(INTEGER_LIST));
  if(plframes==NULL) {
    fprintf(stderr, "Error: cannot allocate memory for frame numbers.\n");
    fclose(fp); ecat63EmptyMatlist(&mlist63); free(matval); integerListEmpty(&planes);
    return(6);
  }
  for(int pi=0; pi<planes.nr; pi++) integerListInit(plframes+pi);
  for(int pi=0; pi<planes.nr; pi++) {
    for(int mi=0; mi<validMatrixNr; mi++) {
      if(matval[mi].plane!=planes.list[pi]) continue;
      integerListAdd(plframes+pi, matval[mi].frame, 1);
    }
    integerListSort(plframes+pi);
  }
  if(verbose>1) {
    printf("\nPlane\tFrames\n");
    for(int pi=0; pi<planes.nr; pi++) {
      printf("%d:", planes.list[pi]);
      for(int fi=0; fi<plframes[pi].nr; fi++)
        printf(" %d", plframes[pi].list[fi]);
      printf("\n");
    }
  }

  /* Make list of planes and frames that we should have */
  INTEGER_LIST keepplanes; integerListInit(&keepplanes);
  int maxfrnr=0, fullestplanei=0;
  {
    /* How many frames there are on planes on average? */
    int fnr=0;
    for(int pi=0; pi<planes.nr; pi++) {
      fnr+=plframes[pi].nr;
      if(plframes[pi].nr>maxfrnr) {maxfrnr=plframes[pi].nr; fullestplanei=pi;}
    }
    fnr/=planes.nr;
    if(verbose>2) printf("%d frames in each plane on average.\n", fnr);
    for(int pi=0; pi<planes.nr; pi++) 
      if(plframes[pi].nr>2*fnr/3)
        integerListAdd(&keepplanes, planes.list[pi], 0);
    if(verbose>3) {
      printf("keeping planes:");
      for(int i=0; i<keepplanes.nr; i++) printf(" %d", keepplanes.list[i]);
      printf("\n"); fflush(stdout);
    }
  }
  INTEGER_LIST keepframes; integerListInit(&keepframes);
  {
    int f, fnr;
    /* Keep frames that are found on most planes */
    for(int fi=0; fi<plframes[fullestplanei].nr; fi++) {
      f=plframes[fullestplanei].list[fi];
      /* on how many planes this frame exists? */
      fnr=0;
      for(int pi=0; pi<planes.nr; pi++) {
        for(int fj=0; fj<plframes[pi].nr; fj++)
          if(f==plframes[pi].list[fi]) {fnr++; break;}
      }
      if(fnr>3*planes.nr/4)
        integerListAdd(&keepframes, f, 0);
    }
    if(verbose>3) {
      printf("keeping frames:");
      for(int i=0; i<keepframes.nr; i++) printf(" %d", keepframes.list[i]);
      printf("\n"); fflush(stdout);
    }
  }

  /* No need for original lists */
  for(int pi=0; pi<planes.nr; pi++) integerListEmpty(plframes+pi);
  free(plframes);
  integerListEmpty(&planes);


  /*
   *  Write output file
   */
  if(verbose>1) {fprintf(stdout, "writing %s\n", outfile); fflush(stdout);}
  /* edit the matrix numbers in the main header */
  {
    int i, m=0;
    for(i=0; i<keepplanes.nr; i++) if(keepplanes.list[i]>m) m=keepplanes.list[i];
    mainheader63.num_planes=m;
    m=0;
    for(i=0; i<keepframes.nr; i++) if(keepframes.list[i]>m) m=keepframes.list[i];
    mainheader63.num_frames=m;
  }
  /* Open output file */
  FILE *fp2;
  fp2=ecat63Create(outfile, &mainheader63);
  if(fp2==NULL) {
    fprintf(stderr, "Error: cannot write ECAT file.\n");
    ecat63EmptyMatlist(&mlist63); free(matval); fclose(fp); 
    integerListEmpty(&keepframes); integerListEmpty(&keepplanes);
    return(11);
  }
  /* Read and save each frame and plane that we want to keep */
  for(int fi=0; fi<keepframes.nr; fi++) {
    int frame=keepframes.list[fi];
    for(int pi=0; pi<keepplanes.nr; pi++) {
      int plane=keepplanes.list[pi];
      /* Compute matnum */
      int matnum=mat_numcod(frame, plane, gate, matdata, bed);
      if(verbose>4) printf("frame %d plane %d -> matnum %d\n", frame, plane, matnum);
      /* Try to find this matnum in the original image */
      int omi=-1;
      for(int mi=0; mi<mlist63.matrixNr; mi++) 
        if(mlist63.matdir[mi].matstat==1 && matnum==mlist63.matdir[mi].matnum) {omi=mi; break;}
      if(omi>=0) {
        /* Try to read this matrix from the original image */
        float *fptr;
        ret=ecat63ReadImageMatrix(fp, mlist63.matdir[omi].strtblk, mlist63.matdir[omi].endblk, &image_header, &fptr);
        if(ret==0) {
          if(verbose>5) printf("copying matrix\n");
          ret=ecat63WriteImageMatrix(fp2, matnum, &image_header, fptr);
          free(fptr);
        }
        if(ret) omi=-1;
      }
      if(omi<0) {
        /* Not found in original image, thus we must save a zero matrix */
        if(verbose>5) printf("creating zero contents\n");
        /* Read sub headers from the same frame and another from the same plane to be used as 
           template; using the list that contains only valid matrices */
        int fmatnum=0, pmatnum=0;
        for(int mi=0; mi<validMatrixNr; mi++)
          if(matval[mi].plane!=plane) {
            pmatnum=mat_numcod(matval[mi].frame, plane, gate, matdata, bed); break;}
        for(int mi=0; mi<validMatrixNr; mi++)
          if(matval[mi].frame!=frame) {
            fmatnum=mat_numcod(frame, matval[mi].plane, gate, matdata, bed); break;}
        ECAT63_imageheader pih, fih;
        int fmi=0, pmi=0;
        for(int mi=0; mi<mlist63.matrixNr; mi++)
          if(mlist63.matdir[mi].matstat==1 && matnum==fmatnum) {fmi=mi; break;}
        for(int mi=0; mi<mlist63.matrixNr; mi++)
          if(mlist63.matdir[mi].matstat==1 && matnum==pmatnum) {pmi=mi; break;}
        ecat63ReadImageheader(fp, mlist63.matdir[fmi].strtblk, &fih, verbose-10, NULL);
        ecat63ReadImageheader(fp, mlist63.matdir[pmi].strtblk, &pih, verbose-10, NULL);
        /* Copy plane-related stuff into frame header */
        fih.slice_width=pih.slice_width;
        fih.slice_location=pih.slice_location;
        fih.plane_eff_corr_fctr=pih.plane_eff_corr_fctr;
        /* Set some to zero */
        fih.quant_scale=0.0;
        fih.image_min=fih.image_max=0;
        int pxlNr=fih.dimension_1*fih.dimension_2;
        float *fdata=malloc(pxlNr*sizeof(float));
        for(int i=0; i<pxlNr; i++) fdata[i]=0.0;
        if(verbose>5) printf("writing matrix\n");
        ret=ecat63WriteImageMatrix(fp2, matnum, &fih, fdata);
        free(fdata);
        if(ret!=0) {
          fprintf(stderr, "Error: cannot write image matrix.\n");
          ecat63EmptyMatlist(&mlist63); free(matval); fclose(fp); fclose(fp2); 
          integerListEmpty(&keepframes); integerListEmpty(&keepplanes);
          return(13);
        }
      }
    }
  }
  integerListEmpty(&keepframes);
  integerListEmpty(&keepplanes);


  /* Close files */
  fclose(fp); fclose(fp2);
  ecat63EmptyMatlist(&mlist63);
  free(matval);

  if(verbose>0) fprintf(stdout, "done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
