/** @file ehdrconv.c
 *  @brief Functions for converting ECAT 6.3 header contents to format 7 
 *  and vice versa.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "ehdrconv.h"
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT main header information from 6.3 format to 7.x.
    @return Returns 0 if ok.
 */
int ecatCopy63to7mainheader(ECAT63_mainheader *h1, ECAT7_mainheader *h2)
{
  int       i;

  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT7_mainheader));
  strlcpy(h2->original_file_name, h1->original_file_name, 20);
  h2->sw_version=72;
  if(h1->system_type==12) h2->system_type=12096; /* for GE Advance in Turku */
  else h2->system_type=h1->system_type;
  switch(h1->file_type) {
    case RAW_DATA:
      h2->file_type=ECAT7_3DSCAN;
      strcpy(h2->magic_number, ECAT7S_MAGICNR);
      break;
    case IMAGE_DATA:
      h2->file_type=ECAT7_VOLUME16;
      strcpy(h2->magic_number, ECAT7V_MAGICNR);
      break;
    case ATTN_DATA:
      h2->file_type=ECAT7_ATTEN;
      strcpy(h2->magic_number, ECAT7S_MAGICNR);
      break;
    case NORM_DATA:
      h2->file_type=ECAT7_2DNORM;
      strcpy(h2->magic_number, ECAT7S_MAGICNR);
      break;
    default: return(5);
  }
  strlcpy(h2->serial_number, h1->node_id, 10);
  h2->scan_start_time=ecat63Scanstarttime(h1);
  strlcpy(h2->isotope_name, h1->isotope_code, 8);
  h2->isotope_halflife=h1->isotope_halflife;
  strlcpy(h2->radiopharmaceutical, h1->radiopharmaceutical, 32);
  h2->gantry_tilt=h1->gantry_tilt;
  h2->gantry_rotation=h1->gantry_rotation;
  h2->bed_elevation=h1->bed_elevation;
  h2->intrinsic_tilt=15.0;
  h2->wobble_speed=h1->wobble_speed;
  h2->transm_source_type=h1->transm_source_type;
  h2->distance_scanned=h1->axial_fov;
  h2->transaxial_fov=h1->transaxial_fov;
  h2->angular_compression=h1->compression_code%256;
  h2->coin_samp_mode=h1->coin_samp_mode;
  h2->axial_samp_mode=h1->axial_samp_mode;
  h2->ecat_calibration_factor=h1->calibration_factor;
  switch(h1->calibration_units) {
    case 0: /* Unknown */
      h2->calibration_units=2;
      h2->calibration_units_label=0;
      strcpy(h2->data_units, "");
      break;
    case 1: /* MBq/cc */
      h2->calibration_units=1;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "MBq/cc");
      break;
    case 2: /* ECAT counts */
      h2->calibration_units=0;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "ECAT counts/sec");
      break;
    case 3: /* uCi/cc */
      h2->calibration_units=1;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "uCi/cc");
      break;
    case 4: /* LMRGlu */
    case 5: /* LMRGlu umol/min/100g */
    case 6: /* LMRGlu mg/min/100g */
      h2->calibration_units=2;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "LMRGlu");
      break;
    case 7: /* nCi/cc */
      h2->calibration_units=1;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "nCi/cc");
      break;
    case 8: /* Well counts */
    case 9: /* Becquerels */
      h2->calibration_units=1;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "Bq/cc");
      break;
    case 10: /* kBq/cc */
      h2->calibration_units=1;
      h2->calibration_units_label=1;
      strcpy(h2->data_units, "kBq/cc");
      break;
    default:
      h2->calibration_units=2;
      h2->calibration_units_label=0;
      strcpy(h2->data_units, "");
      break;
  }
  h2->compression_code=h1->compression_code;
  strlcpy(h2->study_type, h1->study_name, 12);
  strlcpy(h2->patient_id, h1->patient_id, 16);
  strlcpy(h2->patient_name, h1->patient_name, 32);
  h2->patient_sex=h1->patient_sex;
  h2->patient_dexterity=h1->patient_dexterity;
  h2->patient_age=atof(h1->patient_age);
  h2->patient_height=atof(h1->patient_height);
  h2->patient_weight=atof(h1->patient_weight);
  strlcpy(h2->physician_name, h1->physician_name, 32);
  strlcpy(h2->operator_name, h1->operator_name, 32);
  strlcpy(h2->study_description, h1->study_description, 32);
  h2->acquisition_type=h1->acquisition_type;
  if(h2->acquisition_type==0 && h1->num_frames>1) h2->acquisition_type=4;
  h2->patient_orientation=ECAT7_Head_First_Supine;
  strlcpy(h2->facility_name, h1->facility_name, 20);
  h2->num_planes=h1->num_planes; /* may need adjusting later */
  h2->num_frames=h1->num_frames;
  h2->num_gates=h1->num_gates;
  h2->num_bed_pos=h1->num_bed_pos; if(h2->num_bed_pos>0) h2->num_bed_pos-=1;
  h2->init_bed_position=h1->init_bed_position;
  for(i=0; i<15; i++) h2->bed_position[i]=h1->bed_offset[i];
  h2->plane_separation=h1->plane_separation;
  h2->lwr_sctr_thres=h1->lwr_sctr_thres;
  h2->lwr_true_thres=h1->lwr_true_thres;
  h2->upr_true_thres=h1->upr_true_thres;
  strlcpy(h2->user_process_code, h1->user_process_code, 10);
  h2->acquisition_mode=0;
  h2->bin_size=0; /* this value must be read from scan subheader */
  h2->branching_fraction=1.0;
  h2->dose_start_time=h2->scan_start_time;
  h2->dosage=0.0;
  h2->well_counter_corr_factor=0.0;
  h2->septa_state=h1->septa_type;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT image header information from 6.3 format to 7.x.
\return Returns 0 if ok.
 */
int ecatCopy63to7imageheader(ECAT63_imageheader *h1, ECAT7_imageheader *h2)
{
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT7_imageheader));
  h2->data_type=h1->data_type;
  h2->num_dimensions=3;
  h2->x_dimension=h1->dimension_1;
  h2->y_dimension=h1->dimension_2;
  h2->z_dimension=1;                  /* Fill this later with plane number! */
  h2->x_offset=h1->x_origin;
  h2->y_offset=h1->y_origin;
  h2->z_offset=0.0;
  h2->recon_zoom=h1->recon_scale;
  h2->scale_factor=h1->quant_scale;
  h2->image_min=h1->image_min;
  h2->image_max=h1->image_max;
  h2->x_pixel_size=h1->pixel_size;
  h2->y_pixel_size=h1->pixel_size;
  h2->z_pixel_size=h1->slice_width;
  h2->frame_duration=h1->frame_duration;
  h2->frame_start_time=h1->frame_start_time;
  h2->filter_code=h1->filter_code;
  h2->x_resolution=0.0;
  h2->y_resolution=0.0;
  h2->z_resolution=0.0;
  h2->num_r_elements=0.0;
  h2->num_angles=0.0;
  h2->z_rotation_angle=h1->image_rotation;
  h2->decay_corr_fctr=h1->decay_corr_fctr;
  h2->processing_code=h1->processing_code;
  h2->gate_duration=0;
  h2->r_wave_offset=0;
  h2->num_accepted_beats=0;
  h2->filter_cutoff_frequency=h1->filter_params[0];
  h2->filter_resolution=h1->filter_params[1];
  h2->filter_ramp_slope=h1->filter_params[2];
  h2->filter_order=(short int)temp_roundf(h1->filter_params[3]);
  h2->filter_scatter_fraction=h1->filter_params[4];
  h2->filter_scatter_slope=h1->filter_params[5];
  strncpy(h2->annotation, h1->annotation, 40);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT scan header information from 6.3 format to 7 (3D).
\return Returns 0 if ok.
 */
int ecatCopy63to7scanheader(ECAT63_scanheader *h1, ECAT7_scanheader *h2)
{
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT7_scanheader));
  h2->data_type=h1->data_type;
  h2->num_dimensions=4;
  h2->num_r_elements=h1->dimension_1;
  h2->num_angles=h1->dimension_2;
  h2->corrections_applied=h1->processing_code; /* ??? */
  h2->num_z_elements[0]=1; /* Fill these later with plane number(s) */
  h2->ring_difference=0;
  h2->storage_order=1; /* r, theta, z, d */
  h2->axial_compression=0;
  h2->x_resolution=h1->sample_distance; 
  h2->v_resolution=0.0; 
  h2->z_resolution=0.0;
  h2->w_resolution=0.0;
  h2->gate_duration=h1->gate_duration;
  h2->r_wave_offset=h1->r_wave_offset;
  h2->num_accepted_beats=0;
  h2->scale_factor=h1->scale_factor;
  h2->scan_min=h1->scan_min;
  h2->scan_max=h1->scan_max;
  h2->prompts=h1->prompts;
  h2->delayed=h1->delayed;
  h2->multiples=h1->multiples;
  h2->net_trues=h1->net_trues;
  h2->tot_avg_cor=h1->tot_avg_cor;
  h2->tot_avg_uncor=h1->tot_avg_uncor;
  h2->total_coin_rate=h1->total_coin_rate;
  h2->frame_start_time=h1->frame_start_time;
  h2->frame_duration=h1->frame_duration;
  h2->deadtime_correction_factor=h1->loss_correction_fctr;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT normalization header information from 6.3 format to 7 (2D).
\return Returns 0 if ok.
 */
int ecatCopy63to7normheader(ECAT63_normheader *h1, ECAT7_2Dnormheader *h2)
{
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT7_2Dnormheader));
  h2->data_type=h1->data_type;
  h2->num_dimensions=2;
  h2->num_r_elements=h1->dimension_1;
  h2->num_angles=h1->dimension_2;
  h2->num_z_elements=1;
  h2->ring_difference=0;
  h2->scale_factor=h1->scale_factor;
  h2->norm_min=0.0;
  h2->norm_max=0.0;
  h2->fov_source_width=h1->fov_source_width;
  h2->norm_quality_factor=1.0;
  h2->norm_quality_factor_code=0;
  h2->storage_order=1;
  h2->span=0;
  h2->z_elements[0]=1;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT main header information from 7.x format to 6.3.
\return Returns 0 if ok.
 */
int ecatCopy7to63mainheader(ECAT7_mainheader *h1, ECAT63_mainheader *h2)
{
  struct tm  scanStart;
  time_t t;

  //printf("ecatCopy7to63mainheader()\n");
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT63_mainheader));
  strlcpy(h2->original_file_name, h1->original_file_name, 20);
  h2->sw_version=2;
  h2->data_type=SUN_I2; /* fill later with better knowledge */
  if(h1->system_type==12096) h2->system_type=12; /* for GE Advance in Turku */
  else h2->system_type=h1->system_type;
  h2->system_type=h1->system_type;
  switch(h1->file_type) {
    case ECAT7_2DSCAN: h2->file_type=RAW_DATA; break;
    case ECAT7_IMAGE16: h2->file_type=IMAGE_DATA; break;
    case ECAT7_ATTEN: h2->file_type=ATTN_DATA; break;
    case ECAT7_2DNORM: h2->file_type=NORM_DATA; break;
    case ECAT7_VOLUME8: h2->file_type=IMAGE_DATA; break;
    case ECAT7_VOLUME16: h2->file_type=IMAGE_DATA; break;
    case ECAT7_IMAGE8: h2->file_type=IMAGE_DATA; break;
    case ECAT7_3DSCAN: h2->file_type=RAW_DATA; break;
    case ECAT7_3DSCANFIT: h2->file_type=RAW_DATA; break;
    case ECAT7_3DSCAN8: h2->file_type=RAW_DATA; break;
    default: return(5);
  }
  strlcpy(h2->node_id, h1->serial_number, 10);
  t=h1->scan_start_time;
  if(gmtime_r(&t, &scanStart)!=NULL) {
    h2->scan_start_year=scanStart.tm_year+1900;
    h2->scan_start_month=scanStart.tm_mon+1;
    h2->scan_start_day=scanStart.tm_mday;
    h2->scan_start_hour=scanStart.tm_hour;
    h2->scan_start_minute=scanStart.tm_min;
    h2->scan_start_second=scanStart.tm_sec;
  } else {
    h2->scan_start_year=1900;
    h2->scan_start_month=1;
    h2->scan_start_day=1;
    h2->scan_start_hour=0;
    h2->scan_start_minute=0;
    h2->scan_start_second=0;
  }
  strlcpy(h2->isotope_code, h1->isotope_name, 8);
  h2->isotope_halflife=h1->isotope_halflife;
  strlcpy(h2->radiopharmaceutical, h1->radiopharmaceutical, 32);
  h2->gantry_tilt=h1->gantry_tilt;
  h2->gantry_rotation=h1->gantry_rotation;
  h2->bed_elevation=h1->bed_elevation;
  h2->rot_source_speed=0;
  h2->wobble_speed=h1->wobble_speed;
  h2->transm_source_type=h1->transm_source_type;
  h2->axial_fov=h1->distance_scanned;
  h2->transaxial_fov=h1->transaxial_fov;
  h2->transaxial_samp_mode=0;
  h2->coin_samp_mode=h1->coin_samp_mode;
  h2->axial_samp_mode=h1->axial_samp_mode;
  h2->calibration_factor=h1->ecat_calibration_factor;
  if(h1->calibration_units==0) {
    h2->calibration_units=2;
  } else if(h1->calibration_units==1) {
    if(h1->calibration_units_label==0)
      h2->calibration_units=9;
    else if(strcasecmp(h1->data_units, "MBq/cc")==0)
      h2->calibration_units=1;
    else if(strcasecmp(h1->data_units, "MBq/mL")==0)
      h2->calibration_units=1;
    else if(strcasecmp(h1->data_units, "uCi/cc")==0)
      h2->calibration_units=3;
    else if(strcasecmp(h1->data_units, "uCi/mL")==0)
      h2->calibration_units=3;
    else if(strcasecmp(h1->data_units, "nCi/cc")==0)
      h2->calibration_units=7;
    else if(strcasecmp(h1->data_units, "Bq/cc")==0)
      h2->calibration_units=9;
    else if(strcasecmp(h1->data_units, "Bq/mL")==0)
      h2->calibration_units=9;
    else if(strcasecmp(h1->data_units, "kBq/cc")==0)
      h2->calibration_units=10;
    else if(strcasecmp(h1->data_units, "kBq/mL")==0)
      h2->calibration_units=10;
    else
      h2->calibration_units=0;
  } else {
    h2->calibration_units=0;
  }
  h2->compression_code=h1->compression_code;
  strlcpy(h2->study_name, h1->study_type, 12);
  strlcpy(h2->patient_id, h1->patient_id, 16);
  strlcpy(h2->patient_name, h1->patient_name, 32);
  h2->patient_sex=h1->patient_sex;
  sprintf(h2->patient_age, "%-9.2f", h1->patient_age);
  sprintf(h2->patient_height, "%-9.3f", h1->patient_height);
  sprintf(h2->patient_weight, "%-9.3f", h1->patient_weight);
  h2->patient_dexterity=h1->patient_dexterity;
  strlcpy(h2->physician_name, h1->physician_name, 32);
  strlcpy(h2->operator_name, h1->operator_name, 32);
  strlcpy(h2->study_description, h1->study_description, 32);
  h2->acquisition_type=h1->acquisition_type;
  if(h2->acquisition_type==0 && h1->num_frames>1) h2->acquisition_type=4;
  h2->bed_type=0;
  h2->septa_type=h1->septa_state;
  strlcpy(h2->facility_name, h1->facility_name, 20);
  h2->num_planes=h1->num_planes; /* may need adjusting later */
  h2->num_frames=h1->num_frames;
  h2->num_gates=h1->num_gates;
  h2->num_bed_pos=h1->num_bed_pos+1;
  h2->init_bed_position=h1->init_bed_position;
  for(int i=0; i<15; i++) h2->bed_offset[i]=h1->bed_position[i];
  h2->plane_separation=h1->plane_separation;
  h2->lwr_sctr_thres=h1->lwr_sctr_thres;
  h2->lwr_true_thres=h1->lwr_true_thres;
  h2->upr_true_thres=h1->upr_true_thres;
  h2->collimator=0;

  strlcpy(h2->user_process_code, h1->user_process_code, 10);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT 2D image header information from 7.x format to 6.3.
    @return Returns 0 if ok.
 */
int ecatCopy7to63imageheader(ECAT7_imageheader *h1, ECAT63_imageheader *h2)
{
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT63_imageheader));
  h2->data_type=h1->data_type;
  h2->num_dimensions=h1->num_dimensions;
  h2->dimension_1=h1->x_dimension;
  h2->dimension_2=h1->y_dimension;
  h2->x_origin=h1->x_offset;
  h2->y_origin=h1->y_offset;
  h2->recon_scale=h1->recon_zoom;
  h2->quant_scale=h1->scale_factor;
  h2->image_min=h1->image_min;
  h2->image_max=h1->image_max;
  h2->pixel_size=h1->x_pixel_size;
  h2->slice_width=h1->z_pixel_size;
  h2->frame_duration=h1->frame_duration;
  h2->frame_start_time=h1->frame_start_time;
  h2->slice_location=h1->z_offset;
  h2->recon_start_hour=0;
  h2->recon_start_min=0;
  h2->recon_start_sec=0;
  h2->recon_duration=0;
  h2->filter_code=h1->filter_code;
  h2->scan_matrix_num=0;
  h2->norm_matrix_num=0;
  h2->atten_cor_mat_num=0;
  h2->image_rotation=h1->z_rotation_angle;
  h2->plane_eff_corr_fctr=1.0;
  h2->decay_corr_fctr=h1->decay_corr_fctr;
  h2->loss_corr_fctr=1.0;
  h2->processing_code=(short int)h1->processing_code;
  h2->quant_units=0;
  h2->recon_start_day=0;
  h2->recon_start_month=0;
  h2->recon_start_year=0;
  h2->ecat_calibration_fctr=1.0;
  h2->well_counter_cal_fctr=1.0;
  h2->filter_params[0]=h1->filter_cutoff_frequency;
  h2->filter_params[1]=h1->filter_resolution;
  h2->filter_params[2]=h1->filter_ramp_slope;
  h2->filter_params[3]=(float)h1->filter_order;
  h2->filter_params[4]=h1->filter_scatter_fraction;
  h2->filter_params[5]=h1->filter_scatter_slope;
  strncpy(h2->annotation, h1->annotation, 40);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT 3D scan header information from 7.x format to 6.3.
    @return Returns 0 if ok.
 */
int ecatCopy7to63scanheader(ECAT7_scanheader *h1, ECAT63_scanheader *h2)
{
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT63_scanheader));
  h2->data_type=h1->data_type;
  h2->dimension_1=h1->num_r_elements;
  h2->dimension_2=h1->num_angles;
  h2->smoothing=0;
  h2->processing_code=h1->corrections_applied; /* ??? */
  h2->sample_distance=0.0; /* this must be set from main header bin_size */
  h2->isotope_halflife=0.0;  /* fill this from main header */
  h2->frame_duration_sec=(short int)(h1->frame_duration/1000);
  h2->gate_duration=h1->gate_duration;
  h2->r_wave_offset=h1->r_wave_offset;
  h2->scale_factor=h1->scale_factor;
  h2->scan_min=h1->scan_min;
  h2->scan_max=h1->scan_max;
  h2->prompts=h1->prompts;
  h2->delayed=h1->delayed;
  h2->multiples=h1->multiples;
  h2->net_trues=h1->net_trues;
  h2->tot_avg_cor=h1->tot_avg_cor;
  h2->tot_avg_uncor=h1->tot_avg_uncor;
  h2->total_coin_rate=h1->total_coin_rate;
  h2->frame_start_time=h1->frame_start_time;
  h2->frame_duration=h1->frame_duration;
  h2->loss_correction_fctr=h1->deadtime_correction_factor;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT 2D scan header information from 7.x format to 6.3.
    @return Returns 0 if ok.
 */
int ecatCopy7_2D_to63scanheader(ECAT7_2Dscanheader *h1, ECAT63_scanheader *h2)
{
  if(h1==NULL || h2==NULL) return(1);
  memset(h2, 0, sizeof(ECAT63_scanheader));
  h2->data_type=h1->data_type;
  h2->dimension_1=h1->num_r_elements;
  h2->dimension_2=h1->num_angles;
  h2->smoothing=0;
  h2->processing_code=h1->corrections_applied; /* ??? */
  /* this may need to be set from main header bin_size */
  h2->sample_distance=h1->x_resolution; 
  h2->isotope_halflife=0.0;  /* fill this from main header */
  h2->frame_duration_sec=(short int)(h1->frame_duration/1000);
  h2->gate_duration=h1->gate_duration;
  h2->r_wave_offset=h1->r_wave_offset;
  h2->scale_factor=h1->scale_factor;
  h2->scan_min=h1->scan_min;
  h2->scan_max=h1->scan_max;
  h2->prompts=h1->prompts;
  h2->delayed=h1->delayed;
  h2->multiples=h1->multiples;
  h2->net_trues=h1->net_trues;
  h2->tot_avg_cor=h1->tot_avg_cor;
  h2->tot_avg_uncor=h1->tot_avg_uncor;
  h2->total_coin_rate=h1->total_coin_rate;
  h2->frame_start_time=h1->frame_start_time;
  h2->frame_duration=h1->frame_duration;
  h2->loss_correction_fctr=h1->deadtime_correction_factor;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

