/** @file imgdim.c
 *  @brief Report PET image dimensions.
 *  @copyright (c) Turku PET Centre
 *  @remark Replaces the application eframenr.
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Report the image dimensions.",
  " ",
  "Usage: @P [Options] imgfile",
  " ",
  "Image file can be in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  " ",
  "Options:",
  " -x | -y | -z | -t",
  "     Report only the width (x), height (y), planes (z), or time frames (t);",
  "     by default all are printed in stdout.",
  " -size",
  "     Instead of image dimension, report the pixel size(s) in mm.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example: print dimensions of all ECAT 7 images in Windows command shell",
  "       for %f in (*.v) do @P %f",
  " ",
  "Example: print dimensions of all ECAT 7 images in Linux/OSX command shell",
  "       for file in ./*.v; do @P $file; done",
  " ",
  "See also: imgdelfr, imgdelpl, lmlist, imgbox, imgshrink, imgadd",
  " ",
  "Keywords: image, pixel, header, cropping, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char  imgfile[FILENAME_MAX];
  char  dim=(char)0;
  int   item=0; // 0=dim; 1=pxlsize

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(!strcasecmp(cptr, "X")) {dim='x'; continue;}
    if(!strcasecmp(cptr, "Y")) {dim='y'; continue;}
    if(!strcasecmp(cptr, "Z")) {dim='z'; continue;}
    if(!strcasecmp(cptr, "T")) {dim='t'; continue;}
    if(!strcasecmp(cptr, "P")) {dim='z'; continue;}
    if(!strcasecmp(cptr, "F")) {dim='t'; continue;}
    if(!strcasecmp(cptr, "SIZE")) {item=1; continue;}
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!imgfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    if(dim) printf("dim := %c\n", dim);
    if(item) printf("item := %d\n", item);
  }


  /*
   *  Read PET image
   */
  if(verbose>2) printf("reading %s\n", imgfile);
  IMG img; imgInit(&img);
  if(imgRead(imgfile, &img)) {fprintf(stderr, "Error: %s\n", img.statmsg); return(2);}

  /*
   *  Print the requested item(s)
   */
  if(item==0) {
    /* Print the dimension(s) */
    if(dim=='x')      fprintf(stdout, "%d\n", img.dimx);
    else if(dim=='y') fprintf(stdout, "%d\n", img.dimy);
    else if(dim=='z') fprintf(stdout, "%d\n", img.dimz);
    else if(dim=='t') fprintf(stdout, "%d\n", img.dimt);
    else fprintf(stdout, "%d %d %d %d\n", img.dimx, img.dimy, img.dimz, img.dimt);
  } else {
    /* Print the pixel size(s) */
    if(dim=='x')      fprintf(stdout, "%g\n", img.sizex);
    else if(dim=='y') fprintf(stdout, "%g\n", img.sizey);
    else if(dim=='z') fprintf(stdout, "%g\n", img.sizez);
    else fprintf(stdout, "%g %g %g\n", img.sizex, img.sizey, img.sizez);
  }

  imgEmpty(&img);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
