#!/bin/bash
#: Title      : test_imghead
#: Date       : 2016-06-14
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imghead$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


if [ ! -f plane1.dat ] || [ ! -f plane2.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f thr_10_5.dat ] || [ ! -f thr_60_2.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_310.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test data \n"

if [ ! -f dynamic.v ] || [ ! -f dynamic.img ] || [ ! -f regions.dft ]; then

  dft2img -scanner=HR+ -zoom=2 -dim=32 plane1.dat plane1.v plane1.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -scanner=HR+ -zoom=2 -dim=32 plane2.dat plane2.v plane2.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  rm -f dynamic.v
  imgadd dynamic.v plane1.v plane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  rm -f plane1.v plane2.v
  imgdecay -i=C-11 dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  img2dft -P=1 dynamic.v plane1.roi plane1.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2dft -P=2 dynamic.v plane2.roi plane2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr regions.dft plane1.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd regions.dft plane2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  e7to63 -c=o dynamic.v dynamic.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  rm -f ana\dynamic.*
  rmdir ana
  ecat2ana -sif -o=ana dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  rm -f nifti\dynamic.*
  rmdir nifti
  ecat2nii -sif -o=nifti dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: Headcurve from all pixels. \n"
printf " Test case 2: ECAT 7 format. \n"
printf " Expected result: Correct TAC saved. \n\n"


$PROGRAM -min -d2 dynamic.v allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat allhead.dft

cp -f regions.dft all.dft
dftavg -rm all.dft Pl
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg all.svg -s all.dft -l allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=5 all.dft allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Headcurve from all pixels. \n"
printf " Test case 2: NIfTI format. \n"
printf " Expected result: Correct TAC saved. \n\n"


$PROGRAM -min -d2 nifti/dynamic allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat allhead.dft

tac2svg all.svg -s all.dft -l allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.1 -rel=6 all.dft allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Headcurve from all pixels. \n"
printf " Test case 2: Analyze format. \n"
printf " Expected result: Correct TAC saved. \n\n"


$PROGRAM -min -d2 ana/dynamic allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat allhead.dft

tac2svg all.svg -s all.dft -l allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.1 -rel=6 all.dft allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Headcurve from all pixels. \n"
printf " Test case 2: ECAT 6 format. \n"
printf " Expected result: Correct TAC saved. \n\n"


$PROGRAM -min dynamic.img allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat allhead.dft

tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=5 all.dft allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: Headcurve as sum of all pixels. \n"
printf " Expected result: Correct TAC saved. \n\n"

$PROGRAM -sum -min dynamic.v allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat allhead.dft

cp -f regions.dft all.dft
dftavg -rm --silent all.dft Pl
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc all.dft x 2048 allsum.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg allsum.svg -s allsum.dft -l allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.5 -rel=5 allsum.dft allhead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case: Headcurve as count sum of all pixels. \n"
printf " Expected result: Correct TAC saved. \n\n"

$PROGRAM -cps -min -d3 dynamic.v allcpshead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat allcpshead.dft

cp -f regions.dft all.dft
dftavg -rm --silent all.dft Pl
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc all.dft x 2048 allsum.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# convert kBq to MBq
# multiply by voxel volume
taccalc allsum.dft x 0.0002486 allcps.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -yset=MBq allcps.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg allcpshead.svg -s allcps.dft -l allcpshead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=1 -rel=5 allcps.dft allcpshead.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Headcurve from default threshold pixels. \n"
printf " Expected result: Correct TAC saved. \n\n"

$PROGRAM -thr dynamic.v thrdef.head
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat thrdef.head
taccut thrdef.head 35 9999 temp.head
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -abs=0.1 -rel=5 thr_10_5.dat temp.head
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: Headcurve from 60 percent threshold and two last frames. \n"
printf " Expected result: Correct TAC saved. \n\n"

$PROGRAM -thr=60,2 -min dynamic.v thr_60_2.head
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cat thrdef.head
taccut thr_60_2.head 0.1 9999 temp.head
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -abs=0.1 -rel=5 thr_60_2.dat temp.head
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: Option -peakmax. \n"
printf " Expected result: Correct TAC saved. \n\n"

rm -f output.tac
$PROGRAM -peakmax -min dynamic.v output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr temp.tac plane2.dat 4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -tacnames=n -abs=0.1 -rel=1 temp.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Option -framemax. \n"
printf " Expected result: Correct TAC saved. \n\n"

rm -f output.tac
$PROGRAM -framemax -min dynamic.v output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.5 correct_310.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0


