/** @file img2flat.c
 *  @brief Save the individual pixel values in PET image as a flat binary file
 *  as 4-byte (32-bit) floating point values.
 *  @details Application name was previously ecat2flo.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Test other image formats than just ECAT. 
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <float.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Extract the pixel values in PET image to a binary flat file",
  "as 4-byte (32-bit) floating point values, in this order of matrices:",
  "All image planes of the first frame, then planes of the 2nd frame, and so on.",
  "The current platforms byte order (little or big endian) is used.",
  " ",
  "Plane and frame number and matrix dimensions x and y are written on screen",
  "or in specified matrix information file (MIF), if file name for it is given.",
  "These numbers are needed when binary data is imported in another application.",
  " ",
  "Usage: @P [Options] imagefile flatfile [MIF]",
  " ",
  "Options:",
/*
  " -inf=<Matrix information file>",
  "     Plane and frame number and matrix dimensions x and y are written in",
  "     specified text file; these numbers are needed when binary data",
  "     is imported in another application program",
*/
  " -bins=<Nr of bins>",
  "     This application supports also 2D ECAT sinogram data. With this option",
  "     only specified nr of bins in the mid part of sinogram are saved,",
  "     leaving out bins from both sides.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: flat2img, ecat2ana, ecat2nii, imgunit, eframe, convend",
  " ",
  "Keywords: ECAT, image, sinogram, format conversion",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       ret, binNr=0;
  int       pi, yi, xi;
  char      imgfile[FILENAME_MAX], datfile[FILENAME_MAX], miffile[FILENAME_MAX];



  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=datfile[0]=miffile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "BINS=", 5)==0) {
      binNr=atoi(cptr+5); if(binNr>0.0) continue;
    } else if(strncasecmp(cptr, "INF=", 4)==0) {
      strcpy(miffile, cptr+4); if(strlen(miffile)>0) continue;
    } else if(strncasecmp(cptr, "MIF=", 4)==0) {
      strcpy(miffile, cptr+4); if(strlen(miffile)>0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(datfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(miffile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}


  /* Is something missing? */
  if(!datfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(strcasecmp(imgfile, datfile)==0 || strcasecmp(imgfile, miffile)==0) {
    fprintf(stderr, "Error: same name for input and output file.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("datfile := %s\n", datfile);
    printf("miffile := %s\n", miffile);
    printf("binNr := %d\n", binNr);
  }


  if(verbose>1) printf("reading %s frame-by-frame\n", imgfile);
  IMG img; imgInit(&img);
  int frameNr=0; 
  FILE *fp=NULL;
  float *fdata=NULL, *fptr;
  size_t pxlNr=0;

  if(verbose>0) fprintf(stdout, "  processing frames");
  while((ret=imgReadFrame(imgfile, frameNr+1, &img, 0)) == 0) {
    /* report to user that we are still doing something */
    if(verbose>0) {fprintf(stdout, "."); fflush(stdout);}
    /* If first frame, and sinogram, then check and set binNr */
    if(frameNr==0) {
      if(binNr>0 && img.type!=IMG_TYPE_RAW) {
        fprintf(stderr, "\nError: bin nr can be used only with sinograms.\n"); 
        imgEmpty(&img); return(1);
      }
      if(binNr>0 && img.dimx<=binNr) {
        fprintf(stderr, "\nError: bin nr should be smaller than sinogram dimx.\n"); 
        imgEmpty(&img); return(1);
      }
      if(binNr<1) binNr=img.dimx; /* If bin nr was not set; also for images ! */
    }
    /* When the 1st frame, then open output file and allocate memory for
       output data */
    if(frameNr==0) {
      /* Open output data file */
      if(verbose>1) printf("\nOpening output datafile %s\n", datfile);
      if((fp=fopen(datfile, "wb")) == NULL) {
        fprintf(stderr, "\nError: cannot open file %s\n", datfile); 
        imgEmpty(&img); return(11);
      }
      /* Allocate memory for float data only once */
      pxlNr=binNr*img.dimy;
      fdata=(float*)malloc(pxlNr*sizeof(float));
      if(fdata==NULL) {
        fprintf(stderr, "\nError: out of memory\n");
        imgEmpty(&img); fclose(fp); return(5);
      }
    }
    /* Write the frame */
    for(pi=0; pi<img.dimz; pi++) {
      fptr=fdata;
      for(yi=0; yi<img.dimy; yi++)
        for(xi=(img.dimx-binNr)/2; xi<(img.dimx+binNr)/2; xi++)
          *fptr++=img.m[pi][yi][xi][0];
      fptr=fdata;
      if(fwrite((float*)fdata, 4, pxlNr, fp) != pxlNr) {
        fprintf(stderr, "\nError: cannot write matrix data in %s\n", datfile);
        fclose(fp); remove(datfile); free(fdata); imgEmpty(&img);
        return(12);
      }
    } /* next plane */
    /* Prepare to the next frame */
    frameNr++;
  }
  if(verbose>0) {
    if(ret==STATUS_NOMATRIX) fprintf(stdout, " ok");
    fprintf(stdout, "\n");
  }
  if(fp!=NULL) fclose(fp); // close the output binary file
  free(fdata);
  if(verbose>3) printf("imgReadFrame() return value := %d\n", ret);
  if(ret!=0 && ret!=STATUS_NOMATRIX) {
    fprintf(stderr, "Error: %s\n", imgStatus(ret)); //if(TEST) imgInfo(&img);
    imgEmpty(&img);
    return(2);
  }
  if(frameNr<1) {
    fprintf(stderr, "Error: no frames could be read in %s\n", imgfile);
    imgEmpty(&img);
    return(3);
  }

  /*
   *  Write matrix information on stdout
   */
  if(verbose>0 || !miffile[0]) { 
    fp=stdout;
    fprintf(fp, "%d %d %d %d\n", img.dimz, frameNr, binNr, img.dimy);
  }

  /*
   *  Write matrix information in specified file
   */
  if(miffile[0]) {
    if(verbose>1) printf("Opening matrix information file %s\n", miffile);
    if((fp=fopen(miffile, "w")) == NULL) {
      fprintf(stderr, "Error: cannot open file %s\n", miffile); 
      imgEmpty(&img); return(21);
    }
    ret=fprintf(fp, "%d %d %d %d\n", img.dimz, frameNr, binNr, img.dimy);
    fclose(fp);
    if(ret<7) {
      fprintf(stderr, "Error: cannot write in file %s\n", miffile); 
      imgEmpty(&img); return(22);
    }
    if(verbose>0) fprintf(stdout, "Matrix information saved in %s\n", miffile);
  }

  /* Quit */
  if(verbose>0) 
    fprintf(stdout, "written data as 4-byte floats in %s\n", datfile);
  imgEmpty(&img);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
