/** @file imgdelpl.c
 *  @brief Delete specified planes from dynamic or static PET image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Delete specified planes (z dimension) from PET image in ECAT 6.3, 7.x,",
  "NIfTI-1, or Analyze 7.5 format. If name for output file is not given,",
  "then the original image is overwritten.",
  " ",
  "Usage: @P [Options] image planenr1 planenr2 [newimage]",
  " ",
  "Planes to be deleted can be given either as a range with planenr1",
  "representing the first and planenr2 the last plane to delete (both >0),",
  "or the number of planes to delete from the beginning and end (specified",
  "with negative number or zero).",
  " ",
  "Options:",
  " -zero",
  "     Planes are not actually deleted, but pixel values are set to zero.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: delete planes 2-3",
  "     @P a234dy1.v 2 3 a234dy1_wo2-3.v",
  "Example 2: delete three first and two last planes",
  "     @P a234dy1.v -3 -2 a234dy1_4-13.v",
  " ",
  "See also: imgdelfr, imgdim, imgadd, esplit, imgslim, imgbox, imgzavg",
  " ",
  "Keywords: image, cropping, compression, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      doCrop=1; // 0=zeroes; 1=crop
  char     petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int      pstart=0, pstop=0;
  int      fstart=0, fend=0;
  
  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=outfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strcasecmp(cptr, "CROP")==0) {
      doCrop=1; continue;
    } else if(strcasecmp(cptr, "ZERO")==0) {
      doCrop=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    if(atoi_with_check(argv[ai], &pstart)) {
      fprintf(stderr, "Error: invalid cut start '%s'.\n", argv[ai]); return(1);}
    ai++;
  }
  if(ai<argc) {
    if(atoi_with_check(argv[ai], &pstop)) {
      fprintf(stderr, "Error: invalid cut stop '%s'.\n", argv[ai]); return(1);}
    ai++;
  }
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]); return(1);}

  /* Did we get all the information that we need? */
  if(!petfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) strcpy(outfile, petfile);
  if(pstart>0 && pstop>=pstart) {
    /* convert plane range to indices */
    pstart--; pstop--;
  } else if(pstart<=0 && pstop<=0 && (pstart<0 || pstop<0)) {
    /* delete number of planes from beginning/end */
    if(pstart<0) fstart=-pstart;
    if(pstop<0)  fend=-pstop;
    pstart=pstop=-1;
  } else {
    fprintf(stderr, "Error: invalid plane range.\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    printf("doCrop := %d\n", doCrop);
    if(pstart>=0) {
      printf("cut_start_plane := %d\n", pstart);
      printf("cut_stop_plane := %d\n", pstop);
    } else {
      printf("cut_planes_from_beginning := %d\n", fstart);
      printf("cut_planes_from_end := %d\n", fend);
    }
  }
  if(verbose>9) IMG_TEST=verbose-9; else IMG_TEST=0;
  

  /*
   *  Read image
   */
  if(verbose>0) fprintf(stdout, "reading image %s\n", petfile);
  IMG img; imgInit(&img);
  if(imgRead(petfile, &img)) {fprintf(stderr, "Error: %s\n", img.statmsg); return(2);}
  if(verbose>1) {
    printf("dimx := %d\n", img.dimx);
    printf("dimy := %d\n", img.dimy);
    printf("dimz := %d\n", img.dimz);
    printf("dimt := %d\n", img.dimt);
  }

  /* Check the plane range */
  if(pstart>=0 && pstop>=img.dimz) {
    fprintf(stderr, "Error: invalid plane range.\n");
    imgEmpty(&img); return(1);
  }
  if((fstart+fend)>=img.dimz) {
    fprintf(stderr, "Error: all planes would be cropped.\n"); 
    imgEmpty(&img); return(1);
  }


  /*
   *  If contents of selected planes are just to be set to zero,
   *  then do it in place, save the image in new file, and exit.
   */
  if(doCrop==0) {
    if(verbose>1) printf("setting selected pixel values to zero...\n");
    if(pstart>=0) { // plane range was given
      for(int zi=pstart; zi<=pstop; zi++)
        for(int yi=0; yi<img.dimy; yi++)
          for(int xi=0; xi<img.dimx; xi++)
            for(int fi=0; fi<img.dimt; fi++)
              img.m[zi][yi][xi][fi]=0.0;
    }
    if(fstart>0) { // delete from beginning
      for(int zi=0; zi<fstart; zi++)
        for(int yi=0; yi<img.dimy; yi++)
          for(int xi=0; xi<img.dimx; xi++)
            for(int fi=0; fi<img.dimt; fi++)
              img.m[zi][yi][xi][fi]=0.0;
    }
    if(fend>0) { // delete from end
      for(int zi=img.dimz-fend; zi<img.dimz; zi++)
        for(int yi=0; yi<img.dimy; yi++)
          for(int xi=0; xi<img.dimx; xi++)
            for(int fi=0; fi<img.dimt; fi++)
              img.m[zi][yi][xi][fi]=0.0;
    }
    if(verbose>1) fprintf(stdout, "writing image...\n");
    if(imgWrite(outfile, &img)) {
      fprintf(stderr, "Error: %s\n", img.statmsg);
      imgEmpty(&img); return(11);
    }
    if(verbose>0) fprintf(stdout, "%s written.\n", outfile);
    imgEmpty(&img);
    return(0);
  }


  /*
   *  Create new IMG without the planes marked to be deleted,
   *  and save it.
   */
  if(verbose>1) printf("deleting selected planes...\n");
  /* Calculate the nr of remaining planes */
  int planeNr;
  if(pstart>=0) planeNr=img.dimz-(1+pstop-pstart); else planeNr=img.dimz-fstart-fend;
  if(planeNr<1) {
    fprintf(stderr, "Error: all planes would be cropped.\n"); 
    imgEmpty(&img); return(4);
  }
  if(verbose>2) fprintf(stdout, "allocating memory for %d plane(s)\n", planeNr);
  IMG out; imgInit(&out);
  if(imgAllocateWithHeader(&out, planeNr, img.dimy, img.dimx, img.dimt, &img)) {
    fprintf(stderr, "Error: cannot allocate memory for new image.\n"); 
    imgEmpty(&img); return(5);
  }
  /* Copy data from the original image */
  if(verbose>2) fprintf(stdout, "copying data\n");
  if(pstart>=0) { // copy planes except inside given range
    int zj=0;
    for(int zi=0; zi<pstart; zi++, zj++) {
      for(int yi=0; yi<img.dimy; yi++)
        for(int xi=0; xi<img.dimx; xi++)
          for(int fi=0; fi<img.dimt; fi++)
            out.m[zj][yi][xi][fi]=img.m[zi][yi][xi][fi];
    }
    for(int zi=pstop+1; zi<img.dimz; zi++, zj++) {
      for(int yi=0; yi<img.dimy; yi++)
        for(int xi=0; xi<img.dimx; xi++)
          for(int fi=0; fi<img.dimt; fi++)
            out.m[zj][yi][xi][fi]=img.m[zi][yi][xi][fi];
    }
  } else { // copy planes from middle
    int zj=0;
    for(int zi=fstart; zi<img.dimz-fend; zi++, zj++) {
      for(int yi=0; yi<img.dimy; yi++)
        for(int xi=0; xi<img.dimx; xi++)
          for(int fi=0; fi<img.dimt; fi++)
            out.m[zj][yi][xi][fi]=img.m[zi][yi][xi][fi];
    }
  }
  imgEmpty(&img);
  /* Set plane numbers */
  for(int zi=0; zi<out.dimz; zi++) out.planeNumber[zi]=1+zi;
  if(verbose>1) fprintf(stdout, "writing cropped image...\n");
  if(imgWrite(outfile, &out)) {
    fprintf(stderr, "Error: %s\n", out.statmsg);
    imgEmpty(&out); return(11);
  }
  if(verbose>0) fprintf(stdout, "%s written.\n", outfile);
  imgEmpty(&out);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
