/** @file imgfiltl.c
 *  @brief Applies mean filter for dynamic or static PET image data.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Applying mean filter for a static or dynamic PET image in ECAT, NIfTI,",
  "or Analyze format. Filtering can be applied to x,y,z-volume (3D) by",
  "specifying the number of adjacent pixels for x,y- and z-dimensions, or only",
  "to the x,y-planes by giving the number only for x,y-dimensions.",
  "Only zero and odd numbers 3, 5, ... are accepted", 
  " ",
  "Usage: @P [Options] imgfile outputfile Nxy [Nz]",
  " ",
  "Options:",
  " -nonegatives",
  "     Negative voxel values are set to 0.",
  " -difference",
  "     Filtered image is subtracted from original image.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgfiltg, imgdysmo, imgfsegm, imgthrs, imgbkgrm, fvar4img",
  " ",
  "Keywords: image, smoothing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int      Nx=0, Ny=0, Nz=0;
  int      leaveNegat=1;
  int      subtract=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=outfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "NONEGATIVES", 1)==0) {
      leaveNegat=0; continue;
    } else if(strncasecmp(cptr, "DIFFERENCE", 3)==0) {
      subtract=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    strlcpy(outfile, argv[ai++], FILENAME_MAX);
    double v; // check that user did not forget file name
    if(atof_with_check(outfile, &v)==0) {
      fprintf(stderr, "Error: invalid output file name '%s'\n", outfile);
      return(1);
    }
  }
  if(ai<argc) {
    Nx=Ny=atoi(argv[ai]);
    if(Nx>0 || strcmp(argv[ai], "0")==0) {
      ai++;
    } else {
      fprintf(stderr, "Error: invalid Nxy\n");
      return(1);
    }
  }
  if(ai<argc) {
    Nz=atoi(argv[ai]);
    if(Nz>0 || strcmp(argv[ai], "0")==0) {
      ai++;
    } else {
      fprintf(stderr, "Error: invalid Nz\n");
      return(1);
    }
  }
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }
  if(strcasecmp(outfile, petfile)==0) {
    fprintf(stderr, "Error: check the output file name.\n");
    return(1);
  }
  if(!(Nx>0) && !(Ny>0) && !(Nz>0)) {
    fprintf(stderr, "Error: with given parameters no filtering is applied.\n");
    return(1);
  }
  if((Nx>0 && Nx<3) || (Ny>0 && Ny<3) || (Nz>0 && Nz<3)) {
    fprintf(stderr, "Error: invalid pixel number.\n");
    return(1);
  }
  if((Nx>0 && Nx%2==0) || (Ny>0 && Ny%2==0) || (Nz>0 && Nz%2==0)) {
    fprintf(stderr, "Error: pixel numbers must be odd.\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    printf("Nx := %d\n", Nx);
    printf("Ny := %d\n", Ny);
    printf("Nz := %d\n", Nz);
    printf("leaveNegat := %d\n", leaveNegat);
    fflush(stdout);
  }


  /*
   *  Read the contents of the PET file to img data structure
   */
  IMG img; imgInit(&img);
  if(verbose>0) printf("reading %s\n", petfile);
  {
    int ret=imgRead(petfile, &img);
    if(ret) {
      fprintf(stderr, "Error: %s\n", img.statmsg);
      if(verbose>1) printf("ret=%d\n", ret);
      imgEmpty(&img); return(2);
    }
    /* Check if PET data is raw or image */
    if(img.type!=IMG_TYPE_IMAGE) {
      fprintf(stderr, "Error: %s is not an image.\n", petfile);
      imgEmpty(&img); return(2);
    }
  }
  if(verbose>1)
    printf("image_xyzf_dimensions := %d %d %d %d\n", img.dimx, img.dimy, img.dimz, img.dimt);

  /* Check that image size in smoothing dimension is at least 4 */
  if(Nz>0 && img.dimz<=Nz) {
    fprintf(stderr, "Error: invalid Z dimension for 3D filtering.\n");
    imgEmpty(&img); return(3);
  }
  if((Nx>0 && img.dimx<=Nx) || (Ny>0 && img.dimy<=Ny)) {
    fprintf(stderr, "Error: invalid x,y dimensions for filtering.\n");
    imgEmpty(&img); return(3);
  }

  /*
   *  Make backup image, in case subtraction was requested
   */
  IMG orig; imgInit(&orig);
  if(subtract!=0) {
    int ret=imgDup(&img, &orig);
    if(ret) {
      fprintf(stderr, "Error: cannot setup IMG data.\n");
      if(verbose>1) fprintf(stderr, "ret := %d\n", ret);
      imgEmpty(&img); return(6);
    }
  }


  /*
   *  Smoothing
   */
  int ret=imgMeanFilter(&img, Nx, Ny, Nz, 0, verbose-2);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot filter the image.\n");
    if(verbose>1) fprintf(stderr, "ret := %d\n", ret);
    imgEmpty(&img); imgEmpty(&orig); return(7);
  }


  /*
   *  Subtract filtered image from original image, if requested
   */
  if(subtract!=0) {
    for(int z=0; z<img.dimz; z++)
      for(int y=0; y<img.dimy; y++)
        for(int x=0; x<img.dimx; x++)
          for(int t=0; t<img.dimt; t++)
            img.m[z][y][x][t]=orig.m[z][y][x][t]-img.m[z][y][x][t];
    imgEmpty(&orig);
  }

  /* Set negative pixel values to zero, if required */
  if(leaveNegat==0) {
    if(verbose>1) printf("setting negative pixel values to zero\n");
    imgCutoff(&img, 0.0, 1);
  }


  /*
   *  Save filtered image
   */
  if(verbose>2) fprintf(stdout, "writing %s\n", outfile);
  if(imgWrite(outfile, &img)) {
    fprintf(stderr, "Error: %s\n", img.statmsg); 
    imgEmpty(&img);
    return(11);
  }
  if(verbose>0) printf("Written %s\n", outfile);
  imgEmpty(&img);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
