/** @file imgfrsmo.c
 *  @brief Simple smoothing over time frames for dynamic PET image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Simple smoothing over time frames for dynamic PET image in ECAT 6.3 and 7,",
  "Analyze 7.5, and NIfTI format.",
  "Replaces pixel TACs with a frame duration weighted average TAC.",
  "Note that this program can only be used for visualization purposes.",
  " ",
  "Usage: @P [Options] dynamic_image smoothed_image",
  " ",
  "Options:",
  " -n=<nr of frames to average>",
  "     Enter an odd number; by default 3.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1.",
  "  @P -n=5 b123dy1.v b123dy1smoothed.v",
  " ",
  "See also: imgdysmo, imgthrs, imgbkgrm, imgfsegm, imgpeak, fvar4img",
  " ",
  "Keywords: image, smoothing, modelling",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      ret=0, avgNr=3;
  char     imgfile[FILENAME_MAX], smofile[FILENAME_MAX], tmp[FILENAME_MAX];
  char    *cptr;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=smofile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "N=", 2)==0) {
      cptr+=2; avgNr=atoi(cptr); if(avgNr>2) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!imgfile[0]) {
      strcpy(imgfile, argv[ai]); continue;
    } else if(!smofile[0]) {
      strcpy(smofile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!smofile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  /* Check that frame number is odd */
  if((avgNr%2)==0) {
    fprintf(stderr, "Error: even number given with option -n.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("smofile := %s\n", smofile);
    printf("avgNr := %d\n", avgNr);
  }


  /*
   *  Read dynamic image
   */
  if(verbose>=0) fprintf(stdout, "reading dynamic image %s\n", imgfile);
  IMG img; 
  imgInit(&img);
  if(imgRead(imgfile, &img)) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) imgInfo(&img);
    imgEmpty(&img);
    return(2);
  }
  if(verbose>10) imgInfo(&img);
  /* Check that image is image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, 
      "Error: raw PET data cannot be smoothed with this program.\n");
    imgEmpty(&img); return(1);
  }
  /* Check that frame nr is ok */
  if(img.dimt<avgNr) {
    fprintf(stderr, "Error: too few image frames for smoothing.\n");
    imgEmpty(&img); return(1);
  }


  /*
   *  Calculate the smoothed image
   */
  if(verbose>=0) printf("Calculating smoothed image\n");
  ret=imgSmoothOverFrames(&img, avgNr);
  if(ret) {
    fprintf(stderr, "Error: cannot smooth.\n");
    imgEmpty(&img); return(4);
  }

  /* Save the smoothed image */
  if(verbose>=0) printf("writing smoothed image %s\n", smofile);
  backupExistingFile(smofile, NULL, tmp);
  if(imgWrite(smofile, &img)) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    imgEmpty(&img);
    return(11);
  }

  /*
   *  Free up memory
   */
  imgEmpty(&img);

  if(verbose>=0) fprintf(stdout, "done.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
