/** @file imgmove.c
 *  @brief Move specified image frames in x,y-directions.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Move specified image frames in x,y-directions. Only for simulation use:",
  "this is not convenient nor valid method for movement correction, because",
  "attenuation correction is not moved accordingly.",
  " ",
  "Usage: @P [Options] imgfile x y [frames [outputfile]]",
  " ",
  "Image file can be in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  "Movements in x and y directions are given in pixels, with positive number",
  "meaning right/up and negative number meaning left/down.",
  "If name for output file is not given, then the original image is overwritten.",
  "If also frame(s) are not given, then all frames are moved.",
  " ",
  "Options:",
  " -fill",
  "     New image area is filled with neighbouring pixel values",
  "     (by default with zeroes).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example: Move pixels in frames 14-16 upwards by two pixels",
  "     @P ua2918dy1.v 0 2 14-16 ua2918dy1mov.v",
  " ",
  "See also: imgadd, simcirc, simboxes, flat2img",
  " ",
  "Keywords: image, simulation, software testing, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      ret;
  int      fill_area=0; // 0=fill with zeroes, 1=fill with neighbour values
  char     imgfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int      xmov=0, ymov=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=(char)0; 
  INTEGER_LIST framelist; integerListInit(&framelist);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "FILL", 3)==0) {
      fill_area=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {
    if(atoi_with_check(argv[ai++], &xmov)) {fprintf(stderr, "Error: invalid x."); return(1);}
  }
  if(ai<argc) {
    if(atoi_with_check(argv[ai++], &ymov)) {fprintf(stderr, "Error: invalid y."); return(1);}
  } else {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(ai<argc) {
    if(verbose>3) printf("requested_frames: '%s'\n", argv[ai]);  
    if(integerListExpandFromString(argv[ai++], ",;", &framelist, 1)<1) {
      fprintf(stderr, "Error: invalid frame definition.\n");
      return(1);
    }
    integerListSort(&framelist);
  }
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!imgfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(xmov==0 && ymov==0) {fprintf(stderr, "Error: no movement was required.\n"); return(1);}
  if(!outfile[0]) strcpy(outfile, imgfile);


  /* In verbose mode print options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("outfile := %s\n", outfile);
    printf("fill := %d\n", fill_area);
    printf("x := %d\n", xmov);
    printf("y := %d\n", ymov);
    if(framelist.nr>0) {
      printf("frames := %d", framelist.list[0]);
      for(int li=1; li<framelist.nr; li++) printf(", %d", framelist.list[li]);
      printf("\n");
    }
  }


  /*
   *  Read PET image
   */
  if(verbose>0) printf("reading %s\n", imgfile);
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    integerListEmpty(&framelist); return(2);
  }
  /* Check if PET data is raw or image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", imgfile);
    imgEmpty(&img); integerListEmpty(&framelist); return(2);
  }
  if(verbose>1) {
    printf("dimt := %d\n", img.dimt);
    printf("dimx := %d\n", img.dimx);
    printf("dimy := %d\n", img.dimy);
    printf("dimz := %d\n", img.dimz);
  }
  /* Check that xmov and ymov are inside matrix size */
  if(abs(xmov)>=img.dimx || abs(ymov)>=img.dimy) {
    fprintf(stderr, "Error: cannot move outsize image matrix.\n");
    imgEmpty(&img); integerListEmpty(&framelist); return(2);
  }
  /* Check that at least one of specified frames is found in image */
  int movNr=0;
  if(framelist.nr>0) {
    for(int li=0; li<framelist.nr; li++)
      if(framelist.list[li]>0 && framelist.list[li]<=img.dimt) movNr++;
    if(movNr==0) {
      fprintf(stderr, "Error: image does not contain specified frames.\n");
      imgEmpty(&img); return(2);
    }
    if(verbose>1 && movNr!=framelist.nr) printf("%d frame(s) will be moved.\n", movNr);
  } else {
    movNr=img.dimt;
  }
  /* Fill framelist with all frame numbers, if not given by user */
  if(framelist.nr==0) {
    for(int fi=1; fi<=img.dimt; fi++) integerListAdd(&framelist, fi, 0);
    if(verbose>1) {
      printf("frames := %d", framelist.list[0]);
      for(int li=1; li<framelist.nr; li++) printf(", %d", framelist.list[li]);
      printf("\n");
    }
  }


  /*
   *  Movement
   */
  if(verbose>0) printf("processing...\n");
  ymov=-ymov;
  int fi, zi, yi, xi;
  for(int li=0, movNr=0; li<framelist.nr; li++) {
    fi=framelist.list[li]-1; if(fi<0 || fi>=img.dimt) continue;
    if(verbose>2) printf("  moving pixels on frame %d\n", fi+1);
    /* Movement in y direction */
    if(ymov<0) {
      for(yi=-ymov; yi<img.dimy; yi++) {
        if(verbose>8) printf("y: [%d] -> [%d]\n", yi, yi+ymov);
        for(zi=0; zi<img.dimz; zi++) for(xi=0; xi<img.dimx; xi++)
          img.m[zi][yi+ymov][xi][fi]=img.m[zi][yi][xi][fi];
      }
      for(yi=img.dimy+ymov; yi<img.dimy; yi++) {
        if(verbose>8) printf("y: [%d]=0\n", yi);
        for(zi=0; zi<img.dimz; zi++) for(xi=0; xi<img.dimx; xi++)
          if(fill_area && yi>0)
            img.m[zi][yi][xi][fi]=img.m[zi][yi-1][xi][fi];
          else
            img.m[zi][yi][xi][fi]=0.0;
      }
    } else if(ymov>0) {
      for(yi=img.dimy-ymov-1; yi>=0; yi--) { 
        if(verbose>8) printf("y: [%d] -> [%d]\n", yi, yi+ymov);
        for(zi=0; zi<img.dimz; zi++) for(xi=0; xi<img.dimx; xi++)
          img.m[zi][yi+ymov][xi][fi]=img.m[zi][yi][xi][fi];
      }
      for(yi=ymov-1; yi>=0; yi--) {
        if(verbose>8) printf("y: [%d]=0\n", yi);
        for(zi=0; zi<img.dimz; zi++) for(xi=0; xi<img.dimx; xi++)
          if(fill_area && yi<img.dimy-1)
            img.m[zi][yi][xi][fi]=img.m[zi][yi+1][xi][fi];
          else
            img.m[zi][yi][xi][fi]=0.0;
      }
    }
    /* Movement in x direction */
    if(xmov<0) {
      for(xi=-xmov; xi<img.dimx; xi++) {
        if(verbose>8) printf("x: [%d] -> [%d]\n", xi, xi+xmov);
        for(zi=0; zi<img.dimz; zi++) for(yi=0; yi<img.dimy; yi++)
          img.m[zi][yi][xi+xmov][fi]=img.m[zi][yi][xi][fi];
      }
      for(xi=img.dimx+xmov; xi<img.dimx; xi++) {
        if(verbose>8) printf("x: [%d]=0\n", xi);
        for(zi=0; zi<img.dimz; zi++) for(yi=0; yi<img.dimy; yi++)
          if(fill_area && xi>0)
            img.m[zi][yi][xi][fi]=img.m[zi][yi][xi-1][fi];
          else
            img.m[zi][yi][xi][fi]=0.0;
      }
    } else if(xmov>0) {
      for(xi=img.dimx-xmov-1; xi>=0; xi--) { 
        if(verbose>8) printf("x: [%d] -> [%d]\n", xi, xi+xmov);
        for(zi=0; zi<img.dimz; zi++) for(yi=0; yi<img.dimy; yi++)
          img.m[zi][yi][xi+xmov][fi]=img.m[zi][yi][xi][fi];
      }
      for(xi=xmov-1; xi>=0; xi--) {
        if(verbose>8) printf("x: [%d]=0\n", xi);
        for(zi=0; zi<img.dimz; zi++) for(yi=0; yi<img.dimy; yi++)
          if(fill_area && xi<img.dimx-1)
            img.m[zi][yi][xi][fi]=img.m[zi][yi][xi+1][fi];
          else
            img.m[zi][yi][xi][fi]=0.0;
      }
    }
    movNr++;
  }
  integerListEmpty(&framelist); 
  if(verbose>1) printf("%d moved frames.\n", movNr);


  /*
   *  Save the modified image
   */
  if(verbose>0) printf("writing image %s\n", outfile);
  ret=imgWrite(outfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    imgEmpty(&img); return(11);
  }
  imgEmpty(&img);
  if(verbose>0) printf("done.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
