/** @file imgshrink.c
 *  @brief Dynamic image is shrunk to take less memory and disk space.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Shrink the size of a dynamic or static PET image to take less memory and",
  "disk space, and to reduce noise and computation time when testing",
  "pixel-by-pixel modeling methods.",
  "Eight neighbouring pixels are averaged into one pixel; thus image size",
  "will be shrunk into 1/8 of the size of the original image.",
  "ECAT 6.3, 7.x, and Analyze 7.5 and NIfTI-1 formats are accepted.",
  " ",
  "Usage: @P [Options] image shrunkenimage",
  " ",
  "Options:",
  " -z=<Yes|no>",
  "     Nr of image planes (z dimension) is halved (default) or not changed.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P i5998dy1.v i5998dy1_shrunken.v",
  " ",
  "See also: imgslim, e7vplavg, imgdim, esplit, imgbox, imgswell, imginteg",
  " ",
  "Keywords: image, compression, smoothing, cropping, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int      shrink_planes=1; 
  char    *cptr;
  int      ret;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=outfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "Z=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {shrink_planes=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0) {shrink_planes=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    printf("shrink_planes := %d\n", shrink_planes);
  }
  if(verbose>9) IMG_TEST=verbose-10; else IMG_TEST=0;


  /*
   *  Read image
   */
  if(verbose>0) fprintf(stdout, "reading image %s\n", petfile);
  IMG img; imgInit(&img);
  ret=imgRead(petfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg); if(verbose>1) imgInfo(&img);
    return(2);
  }
  if(verbose>1) {
    printf("image dimensions: %d %d %d\n", img.dimz, img.dimy, img.dimx);
    printf("image frame nr: %d\n", img.dimt);
  }
  /* Check if PET data is raw or image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", petfile);
    imgEmpty(&img); return(2);
  }


  /*
   *  Shrink the image
   */
  if(verbose>0) fprintf(stdout, "shrinking...\n");
  IMG out; imgInit(&out); 
  ret=imgShrink(&img, &out, shrink_planes);
  if(ret) {
    fprintf(stderr, "Error: cannot shrink image.\n");
    if(verbose>1) printf("ret=%d\n", ret);
    imgEmpty(&img); imgEmpty(&out); return(3);
  }

  /* Free memory allocated for original image */
  imgEmpty(&img);


  /*
   *  Save the shrunk image
   */
  if(verbose>1) fprintf(stdout, "writing shrunken image in %s\n", outfile);
  ret=imgWrite(outfile, &out);
  if(ret) {
    fprintf(stderr, "Error: %s\n", out.statmsg);
    imgEmpty(&out); return(11);
  }
  if(verbose>0) fprintf(stdout, "Shrunk image was written in %s\n", outfile);

  /* Free memory allocated for output image */
  imgEmpty(&out);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
