#!/bin/bash
#: Title      : test_imgcbv
#: Date       : 2015-12-01
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imgcbv$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

# Check that required files exist
if [ ! -f $PROGRAM ]; then
  printf "\nError: application file missing.\n\n"
  exit 1
fi
if [ ! -f blood.dat ] || [ ! -f tissue.dat ] || [ ! -f vb.dat ]; then
  printf "\nError: required file missing.\n\n"
  exit 1
fi
if [ ! -f correct_1_0_0.dft ] || [ ! -f correct_1_1_0.dft ] || [ ! -f correct_1_2_0.dft ]; then
  printf "\nError: required file missing.\n\n"
  exit 1
fi


if [ ! -f orig.v ] || [ ! -f orig.roi ]; then
  rm -f orig.img
  printf "\nCreating test image\n"
  dft2img -scanner=HR+ -zoom=2 -dim=32 tissue.dat plane1.v orig.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc plane1.v x 2 plane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd orig.v plane1.v plane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=C-11 -dc orig.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2dft -p orig.v orig.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n\nValues in original image:\n"
  cat orig.dft
  printf "\n"
fi
if [ ! -f orig.img ] ; then
  e7to63 -c=o orig.v orig.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f vb.v ] ; then
  rm -f vb.img
  printf "\nCreating Vb image for testing\n"
  dft2img -scanner=HR+ -zoom=2 -dim=32 vb.dat vbplane1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc vbplane1.v x 2 vbplane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd vb.v vbplane1.v vbplane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=O-15 -dc vb.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2dft -p vb.v orig.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n\nValues in original Vb image:\n"
  cat vb.dft
  printf "\n"
fi
if [ ! -f vb.img ] ; then
  e7to63 -c=o vb.v vb.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

printf "\nCreating Analyze images for testing\n"
ecat2ana -o=analyze -sif orig.v vb.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi


printf "\nCreating NIfTI images for testing\n"
ecat2nii -o=nifti -sif orig.v vb.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Usual correction with given Vb fraction. \n"
printf " Test case 2: Image format is ECAT7. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f output_1_0_0.v
$PROGRAM orig.v blood.dat 0.05 output_1_0_0.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_1_0_0.v orig.roi
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_0_0.dft output_1_0_0.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case 1: Usual correction with given Vb fraction. \n"
printf " Test case 2: Image format is ECAT6. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f output_1_0_1.img
$PROGRAM orig.img blood.dat 0.05 output_1_0_1.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_1_0_1.img orig.roi
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_0_0.dft output_1_0_1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Same as before but image format is Analyze. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f analyze/output.img analyze/output.hdr analyze/output.sif
$PROGRAM analyze/orig blood.dat 0.05 analyze/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P analyze/output orig.roi output_1_0_2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_0_0.dft output_1_0_2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Same as before but image format is NIfTI. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f nifti/output.nii nifti/output.sif
$PROGRAM nifti/orig blood.dat 0.05 nifti/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P nifti/output orig.roi output_1_0_3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_0_0.dft output_1_0_3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"


printf "\n 1.1.0 \n"
printf " Test case: Option -TV with given Vb fraction. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f output_1_1_0.v
$PROGRAM -tv orig.v blood.dat 0.05 output_1_1_0.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_1_1_0.v orig.roi
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001  correct_1_1_0.dft output_1_1_0.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 1.2.0 \n"
printf " Test case 1: Vb fractions from Vb image. \n"
printf " Test case 2: Negative voxels are set to zero with option -noneg. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

$PROGRAM -noneg orig.v blood.dat vb.v output_1_2_0.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_1_2_0.v orig.roi
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_2_0.dft output_1_2_0.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.1 \n"
printf " Test case 1: Vb fractions from Vb image. \n"
printf " Test case 2: Negative voxels are not set to zero. \n"
printf " Expected result: Result image is created, and results differ from previous one \n\n"

$PROGRAM orig.v blood.dat vb.v output_1_2_1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_1_2_1.v orig.roi
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 output_1_2_0.dft output_1_2_1.dft
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.2 \n"
printf " Test case 1: Vb fractions from Vb image. \n"
printf " Test case 2: Negative voxels are set to zero with option -noneg. \n"
printf " Test case 3: Both images in ECAT6 format. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

$PROGRAM -noneg orig.img blood.dat vb.img output_1_2_2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_1_2_2.v orig.roi
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_2_0.dft output_1_2_2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.3 \n"
printf " Test case: Same as before but image format is Analyze. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f analyze/output.img analyze/output.hdr analyze/output.sif
$PROGRAM -noneg analyze/orig blood.dat analyze/vb analyze/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P analyze/output orig.roi output_1_2_3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 correct_1_2_0.dft output_1_2_3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Testing simulation option. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

rm -f temp.v output_2_0_0.v
$PROGRAM orig.v blood.dat vb.v temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -sim temp.v blood.dat vb.v output_2_0_0.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=0.1 -abs=0.001 orig.v output_2_0_0.v
printf "\n passed. \n\n"



printf "\n===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: Subtract a TAC from image by setting Vb to 1. \n"
printf " Expected result: Result image is created, and results are close to correct \n\n"

tacadd -ovr subtr.dft orig.dft 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM orig.v subtr.dft 1 output_3_0_0.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -P output_3_0_0.v orig.roi output_3_0_0.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc orig.dft - subtr.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.001 temp.dft output_3_0_0.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0
