/** @file avgfract.c
 *  @brief Calculation of mean of fraction curves.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculation of the mean of fraction curves. Data is not interpolated or",
  "extrapolated. Resulting average datafile will contain a weight column;",
  "weights represent the number of fractions from which each mean",
  "value was calculated.",
  " ",
  "Usage: @P [Options] meanfile fractionfiles",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P mean.rat ue*ap.rat",
  " ",
  "See also: metabcor, fit_ppf, avgbolus, simframe, taccalc, tacadd, tacblend",
  " ",
  "Keywords: TAC, input, simulation",
  0};
/******************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/******************************************************************************/

/******************************************************************************/
/* Local functions */
int samplet_eval(const void *, const void *);
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int fi, ti, ri, ret, n;
  int fileNr=0, file1=0, timeNr=0, timeSpace=0, min_tacs;
  char *cptr, ofile[FILENAME_MAX], dfile[FILENAME_MAX];
  DFT dft, mean;
  double *samplet, *dptr;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ofile[0]=dfile[0]=(char)0;
  dftInit(&dft); dftInit(&mean);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!ofile[0]) {
      strcpy(ofile, argv[ai]); continue;
    } else {
      if(fileNr==0) file1=ai;
      fileNr++;
    }
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", fileNr);
    printf("ofile := %s\n", ofile);
  }

  /* Is something missing? */
  if(fileNr==0) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n",
      argv[0]);
    return(1);
  }
  if(fileNr==1) {
    fprintf(stderr, "Error: only one input file specified.\n");
    return(1);
  }

  /*
   *  Collect all sample times from all input files
   *  and get the min nr of TACs
   */
  if(verbose>0) printf("collecting sample times\n");
  timeNr=timeSpace=0; samplet=(double*)NULL; min_tacs=10000;
  for(ai=file1; ai<argc; ai++) if(*argv[ai]!='-') {
    /* Read next input file */
    strcpy(dfile, argv[ai]);
    if(verbose>1) printf("dfile := %s\n", dfile);
    if(dftRead(dfile, &dft)) {
      fprintf(stderr, "Error in reading '%s': %s\n", dfile, dfterrmsg);
      free(samplet); dftEmpty(&dft); return(2);
    }
    /* Get the nr of TACs */
    if(dft.voiNr<min_tacs) min_tacs=dft.voiNr;
    /* Allocate more memory for sample time list, if necessary */
    if(timeSpace-timeNr<dft.frameNr) {
      timeSpace+=dft.frameNr;
      samplet=(double*)realloc(samplet, timeSpace*sizeof(double));
      if(samplet==NULL) {
        fprintf(stderr, "Error in allocation of memory.\n");
        dftEmpty(&dft); return(3);
      }
    }
    /* Copy new sample times to the list */
    for(fi=0; fi<dft.frameNr; fi++) {
      /* if all fractions are NA, then the time is not needed */
      for(ri=n=0; ri<dft.voiNr; ri++) if(!isnan(dft.voi[ri].y[fi])) n++;
      if(n==0) continue;
      /* if sample time is already listed, do nothing */
      for(ti=ret=0; ti<timeNr; ti++) if(samplet[ti]==dft.x[fi]) {ret=1; break;}
      /* otherwise add it */
      if(ret==0) {
        if(verbose>8) printf("  adding time %g\n", dft.x[fi]);
        samplet[timeNr++]=dft.x[fi];
      }
    }
    dftEmpty(&dft);
  }
  if(timeNr==0) {
    fprintf(stderr, "Error: could not read sample times.\n");
    free(samplet); return(2);
  }
  if(verbose>4) printf("sorting %d sample times\n", timeNr);
  dptr=samplet;
  qsort(dptr, timeNr, sizeof(double), samplet_eval);
  if(verbose>3) {
    printf("Collected sample times:\n  %g", samplet[0]);
    for(ti=1; ti<timeNr; ti++) printf(", %g", samplet[ti]);
    printf("\n");
  }
  if(verbose>1) printf("min TAC number: %d\n", min_tacs);

  /*
   *  Set space for mean fraction data
   */
  ret=dftSetmem(&mean, timeNr, min_tacs);
  if(ret) {
    fprintf(stderr, "Error in allocation of memory.\n");
    free(samplet); dftEmpty(&mean); return(5);
  }
  mean.frameNr=timeNr; mean.voiNr=min_tacs;
  for(ti=0; ti<timeNr; ti++) mean.x[ti]=samplet[ti];
  free(samplet);
  mean.isweight=1;
  mean._type=DFT_FORMAT_STANDARD;


  /*
   *  Add the fractions from fraction datafiles
   */
  if(verbose>0) printf("collecting the fractions\n");
  for(ai=file1; ai<argc; ai++) if(*argv[ai]!='-') {
    /* Read next input file */
    strcpy(dfile, argv[ai]);
    if(dftRead(dfile, &dft)) {
      fprintf(stderr, "Error in reading '%s': %s\n", dfile, dfterrmsg);
      free(samplet); return(2);
    }
    for(fi=0; fi<dft.frameNr; fi++) {
      /* find the correct sample time */
      for(ti=0; ti<mean.frameNr; ti++) if(mean.x[ti]==dft.x[fi]) break;
      if(ti==mean.frameNr) continue; /* not found; should not happen */
      /* add the fractions */
      for(ri=n=0; ri<mean.voiNr; ri++) if(!isnan(dft.voi[ri].y[fi])) {
        mean.voi[ri].y[ti]+=dft.voi[ri].y[fi]; n++;}
      /* increase weight (if at least one was available) */
      if(n>0) mean.w[ti]++;
    }
    /* Get also some header information */
    if(ai==file1) {
      strcpy(mean.unit, dft.unit);
      mean.timeunit=dft.timeunit;
      for(ri=0; ri<mean.voiNr; ri++) {
        strcpy(mean.voi[ri].name, dft.voi[ri].name);
        strcpy(mean.voi[ri].voiname, dft.voi[ri].voiname);
        strcpy(mean.voi[ri].hemisphere, dft.voi[ri].hemisphere);
        strcpy(mean.voi[ri].place, dft.voi[ri].place);
      }
    }
    dftEmpty(&dft);
  }
  /* Calculate the means by dividing by weights */
  for(ti=0; ti<mean.frameNr; ti++) {
    if(mean.w[ti]<=0.0)
      for(ri=0; ri<mean.voiNr; ri++) mean.voi[ri].y[ti]=nan("");
    else
      for(ri=0; ri<mean.voiNr; ri++) mean.voi[ri].y[ti]/=mean.w[ti];
  }


  /*
   *  Save the mean fraction curves
   */
  if(verbose>1) printf("writing %s\n", ofile);
  if(dftWrite(&mean, ofile)) {
    fprintf(stderr, "Error in writing '%s': %s\n", ofile, dfterrmsg);
    dftEmpty(&mean);
    return(11);
  }
  if(verbose>=0) fprintf(stdout, "  %s written.\n", ofile),

  dftEmpty(&mean);

  return(0);
}
/******************************************************************************/

/******************************************************************************/
int samplet_eval(const void *n1, const void *n2)
{
  if(*(double*)n1 < *(double*)n2) return -1;
  else if(*(double*)n1 > *(double*)n2) return +1;
  else return 0;
}
/******************************************************************************/

/******************************************************************************/
/// @endcond
