/// @file idimask.c
/// @brief Functions using mask image for extracting input.
/// @author Vesa Oikonen
///
/*****************************************************************************/
#include "libtpcidi.h"
/*****************************************************************************/

/*****************************************************************************/
/** Extract TACs of every image voxel which has a value > thrs in the mask image.

    Mask value is saved as region 'size' in DFT. Mainly for testing IDI methods.

    @sa imgGetConcWeightedPeakPos, imgCircleMask, imgSimulateSphere
    @return Returns 0 if successful.
 */
int imgMaskPixelTACs(
  /** Pointer to allocated image from which the weighted TAC is calculated. */
  IMG *img,
  /** Pointer to mask image; x, y, and z dimensions must be the same as in
      the image to which the mask is applied. */
  IMG *mask,
  /** Mask threshold: pixels with mask values above this are included in TACs. */
  double thrs,
  /** Pointer to initiated DFT struct where pixel TACs will be written;
      any previous contents are deleted. */
  DFT *dft,
  /** Verbose level; set to <=0 to prevent all prints to stdout .*/
  int verbose
) {
  if(verbose>0) printf("%s()\n", __func__);

  int xi, yi, zi, fi, ri, ret;

  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(mask->status<IMG_STATUS_OCCUPIED) return(2);
  if(mask->dimz!=img->dimz) return(3);
  if(mask->dimy!=img->dimy) return(4);
  if(mask->dimx!=img->dimx) return(5);
  if(img->dimt<1 || mask->dimt<1) return(6);
  if(dft==NULL) return(7);

  /* Delete any previous TAC data; do it first so that caller can check
     the nr of extracted TACs inside DFT */
  dftEmpty(dft);

  /* Calculate the nr of voxels to extract */
  long long nr=0;
  for(zi=0; zi<mask->dimz; zi++)
    for(yi=0; yi<mask->dimy; yi++)
      for(xi=0; xi<mask->dimx; xi++)
        if(mask->m[zi][yi][xi][0]>thrs) nr++;
  if(verbose>1) printf("mask_pixel_nr := %lld\n", nr);
  /* Return, if no voxels were marked in mask image */
  if(nr==0) return(0);

  /* Allocate memory for pixel TACs */
  ret=dftAllocateWithIMG(dft, nr, img);
  if(ret!=0) {
    if(verbose>0) fprintf(stderr, "Error: cannot allocate memory for %lld pixel TACs.\n", nr);
    return(10);
  }
  strcpy(dft->studynr, img->studyNr);

  /* Save the pixel TACs in DFT */
  ri=0;
  for(zi=0; zi<mask->dimz; zi++)
    for(yi=0; yi<mask->dimy; yi++)
      for(xi=0; xi<mask->dimx; xi++) if(mask->m[zi][yi][xi][0]>thrs)
  {
    sprintf(dft->voi[ri].voiname, "%d", 1+xi);
    sprintf(dft->voi[ri].hemisphere, "%d", 1+yi);
    sprintf(dft->voi[ri].place, "%d", 1+zi);
    sprintf(dft->voi[ri].name, "%d %d %d", 1+xi, 1+yi, 1+zi);
    for(fi=0; fi<img->dimt; fi++) dft->voi[ri].y[fi]=img->m[zi][yi][xi][fi];
    dft->voi[ri].size=mask->m[zi][yi][xi][0];
    ri++;
  }
  dft->voiNr=ri;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
