/// @file recovery.c
/// @brief Functions for computing recovery errors for vessel.
/// @author Vesa Oikonen
///
/*****************************************************************************/
#include "libtpcidi.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the traditional recovery coefficient (RC) of peak value of a circular 
    radioactive object in 2D PET image, assuming that the object length in 3D is 
    relatively long, and assuming that activity of background is zero.

    True object activity A0 can be calculated using RC, background activity Abkg,
    and measured peak activity Aexp using equation
         A0 = Abkg + (Aexp - Abkg)/RC
    , or if Abkg=0,
         A0 = Aexp/RC.
    
    References: Germano et al. JNM 1992; 33: 613-620 and Brix et al. Nuklearmedizin 2002;41:184-190.
    @sa idiSimulateTubeVol, heartRecoverySpilloverCorrectionFactors
    @return Returns the recovery coefficient [0-1].
 */
double rcPeakPET(
  /** Full-width half-maximum value. */
  double FWHM,
  /** Radius of the object. */
  double R
) {
  double s, rc;
  s=FWHM/2.354820; // 2*sqrt(2*ln(2))
  rc=1.0-exp(-R*R/(2.0*s*s));
  return(rc);
}
/*****************************************************************************/

/*****************************************************************************/
