/// @file imgfilter.c
/// @brief Gaussian IMG filters.
///
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
/** Gaussian finite impulse response (FIR) isotropic or anisotropic image 
    filtering in 1-3 dimensions.

    Based on the code in:
    Getreuer P. A survey of Gaussian convolution algorithms.
    Image Processing On Line 2013;3:286–310. https://doi.org/10.5201/ipol.2013.87

    @sa imgGaussianAMFilter, imgSS
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
 */
int imgGaussianFIRFilter(
  /** Image data to be processed; data is overwritten with filtered image.
      All image frames are processed. */
  IMG *img,
  /** Gaussian S.D. in pixels in x dimension; SD=FWHM/2.355. Enter 0 to not filter in x dimension. */
  float xsd,
  /** Gaussian S.D. in pixels in y dimension; SD=FWHM/2.355. Enter 0 to not filter in y dimension. */
  float ysd,
  /** Gaussian S.D. in pixels in z dimension; SD=FWHM/2.355. Enter 0 to not filter in z dimension. */
  float zsd,
  /** Tolerance; enter 0.0 to use the default (1.0E-03). */
  double tolerance,
  /** Verbose level; if zero, then only warnings are printed into stderr. */
  int verbose
) {
  if(verbose>0) printf("%s(*img, %g, %g, %g, %g, %d)\n", __func__, xsd, ysd, zsd, tolerance, verbose); 

  if(img==NULL || img->dimz<1 || img->dimy<1 || img->dimx<1 || img->dimt<1) return 1;
  int dimx=img->dimx, dimy=img->dimy, dimz=img->dimz, dimt=img->dimt;
  if(dimx==1) xsd=0.0;
  if(dimy==1) ysd=0.0;
  if(dimz==1) zsd=0.0;
  if(xsd>0.0 && (dimx<4 || !(img->sizex>0.0))) return(2);
  if(ysd>0.0 && (dimy<4 || !(img->sizey>0.0))) return(3);
  if(zsd>0.0 && (dimz<4 || !(img->sizez>0.0))) return(4);
  if(!(xsd>0.0) && !(ysd>0.0) && !(zsd>0.0)) {
    if(verbose>1) printf("  no filtering done.\n");
    return(0);
  }
  if(!(tolerance>0.0)) tolerance=1.0E-03;
  if(tolerance>=1.0) return(5);

  double sqrt_pi2=sqrt(M_PI)/M_SQRT2; // sqrt(pi)/sqrt(2)

  /* 
   *  Filtering in x dimension
   */
  if(xsd>0.0) {
    if(verbose>1) {printf("  filtering in x dimension\n"); fflush(stdout);}
    int r=ceil(M_SQRT2*xsd*inverfc(0.5*tolerance)); // Radius of the filter
    double gtrunc[r+1]; // Truncated Gaussian filter for FIR convolution
    double sdsqrt2=M_SQRT2*xsd;
    gtrunc[0]=1.0;
//    for(int i=1; i<=r; i++) gtrunc[i]=exp(-0.5*((double)i/xsd)*((double)i/xsd));
    for(int i=1; i<=r; i++)
      gtrunc[i]=sqrt_pi2*xsd*(erf(((double)i+0.5)/sdsqrt2) - erf(((double)i-0.5)/sdsqrt2));
    double sum=gtrunc[0]; for(int i=1; i<=r; i++) sum+=2.0*gtrunc[i];
    for(int i=0; i<=r; i++) gtrunc[i]/=sum;
    if(verbose>2) printf("  r=%d \n", r);
    if(verbose>3) {
      printf("  gtrunc[]= %g", gtrunc[0]); for(int i=1; i<=r; i++) printf(",%g", gtrunc[i]);
      printf("\n");
    }

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      for(int z=0; z<dimz; z++) for(int y=0; y<dimy; y++) {
        double a[dimx];
        for(int x=0; x<dimx; x++) a[x]=img->m[z][y][x][t];
        for(int x=0; x<dimx; x++) {
          double v=gtrunc[0]*a[x];
          for(int m=1; m<=r; m++)
            v+=gtrunc[m]*(a[filterHSSBE(dimx, x-m)] + a[filterHSSBE(dimx, x+m)]);
          img->m[z][y][x][t]=v;
        }
      }
    }
  }


  /* 
   *  Filtering in y dimension
   */
  if(ysd>0.0) {
    if(verbose>1) {printf("  filtering in y dimension\n"); fflush(stdout);}
    int r=ceil(M_SQRT2*ysd*inverfc(0.5*tolerance)); // Radius of the filter
    double gtrunc[r+1]; // Truncated Gaussian filter for FIR convolution
    double sdsqrt2=M_SQRT2*ysd;
    gtrunc[0]=1.0;
//    for(int i=1; i<=r; i++) gtrunc[i]=exp(-0.5*((double)i/ysd)*((double)i/ysd));
    for(int i=1; i<=r; i++)
      gtrunc[i]=sqrt_pi2*ysd*(erf(((double)i+0.5)/sdsqrt2) - erf(((double)i-0.5)/sdsqrt2));
    double sum=gtrunc[0]; for(int i=1; i<=r; i++) sum+=2.0*gtrunc[i];
    for(int i=0; i<=r; i++) gtrunc[i]/=sum;
    if(verbose>2) printf("  r=%d \n", r);
    if(verbose>3) {
      printf("  gtrunc[]= %g", gtrunc[0]); for(int i=1; i<=r; i++) printf(",%g", gtrunc[i]);
      printf("\n");
    }

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      for(int z=0; z<dimz; z++) for(int x=0; x<dimx; x++) {
        double a[dimy];
        for(int y=0; y<dimy; y++) a[y]=img->m[z][y][x][t];
        for(int y=0; y<dimy; y++) {
          double v=gtrunc[0]*a[y];
          for(int m=1; m<=r; m++)
            v+=gtrunc[m]*(a[filterHSSBE(dimy, y-m)] + a[filterHSSBE(dimy, y+m)]);
          img->m[z][y][x][t]=v;
        }
      }
    }
  }


  /* 
   *  Filtering in z dimension
   */
  if(zsd>0.0) {
    if(verbose>1) {printf("  filtering in z dimension\n"); fflush(stdout);}
    int r=ceil(M_SQRT2*zsd*inverfc(0.5*tolerance)); // Radius of the filter
    double gtrunc[r+1]; // Truncated Gaussian filter for FIR convolution
    double sdsqrt2=M_SQRT2*zsd;
    gtrunc[0]=1.0;
//    for(int i=1; i<=r; i++) gtrunc[i]=exp(-0.5*((double)i/zsd)*((double)i/zsd));
    for(int i=1; i<=r; i++)
      gtrunc[i]=sqrt_pi2*zsd*(erf(((double)i+0.5)/sdsqrt2) - erf(((double)i-0.5)/sdsqrt2));
    double sum=gtrunc[0]; for(int i=1; i<=r; i++) sum+=2.0*gtrunc[i];
    for(int i=0; i<=r; i++) gtrunc[i]/=sum;
    if(verbose>2) printf("  r=%d \n", r);
    if(verbose>3) {
      printf("  gtrunc[]= %g", gtrunc[0]); for(int i=1; i<=r; i++) printf(",%g", gtrunc[i]);
      printf("\n");
    }

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      for(int y=0; y<dimy; y++) for(int x=0; x<dimx; x++) {
        double a[dimz];
        for(int z=0; z<dimz; z++) a[z]=img->m[z][y][x][t];
        for(int z=0; z<dimz; z++) {
          double v=gtrunc[0]*a[z];
          for(int m=1; m<=r; m++)
            v+=gtrunc[m]*(a[filterHSSBE(dimz, z-m)] + a[filterHSSBE(dimz, z+m)]);
          img->m[z][y][x][t]=v;
        }
      }
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Alvarez-Mazorra approximate Gaussian image filtering in 1-3 dimensions.

    References:
    - Getreuer P. A survey of Gaussian convolution algorithms. https://doi.org/10.5201/ipol.2013.87
    - Alvarez L, Mazorra L, Signal and image restoration using shock Filters and anisotropic
      Diffusion. https://www.jstor.org/stable/2158018

    @sa imgGaussianFIRFilter, imgSS
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
 */
int imgGaussianAMFilter(
  /** Image data to be processed; data is overwritten with filtered image.
      All image frames are processed. */
  IMG *img,
  /** Gaussian S.D. in pixels in x dimension; SD=FWHM/2.355. Enter 0 to not filter in x dimension. */
  float xsd,
  /** Gaussian S.D. in pixels in y dimension; SD=FWHM/2.355. Enter 0 to not filter in y dimension. */
  float ysd,
  /** Gaussian S.D. in pixels in z dimension; SD=FWHM/2.355. Enter 0 to not filter in z dimension. */
  float zsd,
  /** Number of filtering passes, usually 4 or more; enter 0 to use the default steps (4). */
  int passNr,
  /** Tolerance; enter 0.0 to use the default (1.0E-06). */
  double tolerance,
  /** Verbose level; if zero, then only warnings are printed into stderr. */
  int verbose
) {
  if(verbose>0) printf("%s(*img, %g, %g, %g, %d, %d)\n", __func__, xsd, ysd, zsd, passNr, verbose); 

  if(img==NULL || img->dimz<1 || img->dimy<1 || img->dimx<1 || img->dimt<1) return 1;
  int dimx=img->dimx, dimy=img->dimy, dimz=img->dimz, dimt=img->dimt;
  if(dimx==1) xsd=0.0;
  if(dimy==1) ysd=0.0;
  if(dimz==1) zsd=0.0;
  if(xsd>0.0 && (dimx<4 || !(img->sizex>0.0))) return(2);
  if(ysd>0.0 && (dimy<4 || !(img->sizey>0.0))) return(3);
  if(zsd>0.0 && (dimz<4 || !(img->sizez>0.0))) return(4);
  if(!(xsd>0.0) && !(ysd>0.0) && !(zsd>0.0)) {
    if(verbose>1) printf("  no filtering done.\n");
    fflush(stdout); return(0);
  }
  if(passNr<=0) passNr=4;
  if(!(tolerance>0.0)) tolerance=1.0E-06;


  /* 
   *  Filtering in x dimension
   */
  if(xsd>0.0) {
    if(verbose>1) {printf("  filtering in x dimension\n"); fflush(stdout);}
    double q=xsd;
    if(0) { // Adjust SD for improved accuracy at low step numbers
      double f=0.7818+(double)passNr;
      q*= 1.0 + (0.3165*(double)passNr + 0.5695)/(f*f);
    }
    double lambda=(q*q)/(2.0*(double)passNr);
    double nu=(1.0 + 2.0*lambda - sqrt(1.0 + 4.0*lambda))/(2.0*lambda);
    double scale=pow(nu/lambda, passNr);
    int termNr=ceil(log((1.0-nu)*tolerance)/log(nu));
    if(verbose>2) printf("  nu=%g scale=%g termNr=%d\n", nu, scale, termNr);

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      double sum1=0.0, sum2=0.0;
      for(int z=0; z<dimz; z++) for(int y=0; y<dimy; y++) {
        double d[dimx];
        for(int x=0; x<dimx; x++) sum1+=d[x]=img->m[z][y][x][t];
        filterAMGaussian(d, dimx, passNr, nu, scale, termNr);
        for(int x=0; x<dimx; x++) sum2+=img->m[z][y][x][t]=d[x];
      }
      double f=sum1/sum2;
      if(verbose>1 && fabs(1.0-f)>1.0E-10) {
        printf("  frame %d: sum of voxels differs pre/post filtering %g/%g=%g\n", 1+t, sum1, sum2, f);
      }
    }
  }

  /* 
   *  Filtering in y dimension
   */
  if(ysd>0.0) {
    if(verbose>1) {printf("  filtering in y dimension\n"); fflush(stdout);}
    double q=ysd;
    if(0) { // Adjust SD for improved accuracy at low step numbers
      double f=0.7818+(double)passNr;
      q*= 1.0 + (0.3165*(double)passNr + 0.5695)/(f*f);
    }
    double lambda=(q*q)/(2.0*(double)passNr);
    double nu=(1.0 + 2.0*lambda - sqrt(1.0 + 4.0*lambda))/(2.0*lambda);
    double scale=pow(nu/lambda, passNr);
    int termNr=ceil(log((1.0-nu)*tolerance)/log(nu));
    if(verbose>2) printf("  nu=%g scale=%g termNr=%d\n", nu, scale, termNr);

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      double sum1=0.0, sum2=0.0;
      for(int z=0; z<dimz; z++) for(int x=0; x<dimx; x++) {
        double d[dimy];
        for(int y=0; y<dimy; y++) sum1+=d[y]=img->m[z][y][x][t];
        filterAMGaussian(d, dimy, passNr, nu, scale, termNr);
        for(int y=0; y<dimy; y++) sum2+=img->m[z][y][x][t]=d[y];
      }
      double f=sum1/sum2;
      if(verbose>1 && fabs(1.0-f)>1.0E-10) {
        printf("  frame %d: sum of voxels differs pre/post filtering %g/%g=%g\n", 1+t, sum1, sum2, f);
      }
    }
  }

  /* 
   *  Filtering in z dimension
   */
  if(zsd>0.0) {
    if(verbose>1) {printf("  filtering in z dimension\n"); fflush(stdout);}
    double q=xsd;
    if(0) { // Adjust SD for improved accuracy at low step numbers
      double f=0.7818+(double)passNr;
      q*= 1.0 + (0.3165*(double)passNr + 0.5695)/(f*f);
    }
    double lambda=(q*q)/(2.0*(double)passNr);
    double nu=(1.0 + 2.0*lambda - sqrt(1.0 + 4.0*lambda))/(2.0*lambda);
    double scale=pow(nu/lambda, passNr);
    int termNr=ceil(log((1.0-nu)*tolerance)/log(nu));
    if(verbose>2) printf("  nu=%g scale=%g termNr=%d\n", nu, scale, termNr);

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      double sum1=0.0, sum2=0.0;
      for(int y=0; y<dimy; y++) for(int x=0; x<dimy; x++) {
        double d[dimz];
        for(int z=0; z<dimz; z++) sum1+=d[z]=img->m[z][y][x][t];
        filterAMGaussian(d, dimz, passNr, nu, scale, termNr);
        for(int z=0; z<dimz; z++) sum2+=img->m[z][y][x][t]=d[z];
      }
      double f=sum1/sum2;
      if(verbose>1 && fabs(1.0-f)>1.0E-10) {
        printf("  frame %d: sum of voxels differs pre/post filtering %g/%g=%g\n", 1+t, sum1, sum2, f);
      }
    }
  }


  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** 1D Alvarez-Mazorra Gaussian filtering.
    @sa imgGaussianAMFilter
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
 */
int filterAMGaussian(
  /** Pointer to source data array, overwritten by the results. */
  double *a,
  /** Sample number. */
  int n,
  /** Number of filtering passes; usually 4. */
  int passNr,
  /** Filter coefficient. */
  double nu,
  /** Filter scale. */
  double scale,
  /** Number of terms needed to approximate the sum with required accuracy for the left boundary. */
  int termNr
) {
  if(a==NULL || n<3) return(1);
  if(!(nu>0.0)) return(2);
  if(passNr<1) passNr=4;
  if(termNr<6) termNr=6;

  /* Scale the data. */
  for(int i=0; i<n; i++) a[i]*=scale;

  /* Loop through the passes of filtering */
  for(int pass=0; pass<passNr; pass++) {
    a[0]=filterAMLB(a, n, nu, termNr);
    for(int i=1; i<n; i++) a[i]+=nu*a[i-1]; // causal filter
    a[n-1]/=(1.0-nu);
    for(int i=n-1; i>0; i--) a[i-1]+=nu*a[i]; // anti-causal filter
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Half-sample symmetric boundary extension.
    @sa filterEbox, filterAMLB
    @return Reflected sample in array[0..n-1].
 */
inline int filterHSSBE(
  /** Sample number. */
  int n,
  /** Requested sample, possibly outside array[0..N-1]. */
  int r
) {
  while(1) {
    if(r<0) r=-1-r;
    else if(r>=n) r=2*n-1-r;
    else break;
  }
  return(r);
}
/*****************************************************************************/

/*****************************************************************************/
/** Left boundary for 1D Alvarez-Mazorra Gaussian filtering.
    @sa filterAMGaussian, filterHSSBE
    @return The sum approximating the first filtered sample value, or NaN in case of an error.
 */
double filterAMLB(
  /** Pointer to source data array; not modified. */
  double *a,
  /** Sample number. */
  int n,
  /** Filtering parameter. */
  double nu,
  /** Number of terms. */
  int termNr
) {
  if(a==NULL || n<1) return(nan(""));
  double h=1.0, sum=a[0];
  for(int m=1; m<termNr; m++) {
    h*=nu;
    sum+=h*a[filterHSSBE(n, -m)];
  }
  return(sum);
}
/*****************************************************************************/

/*****************************************************************************/
/** Gaussian extended box image filtering in 1-3 dimensions.

    References:
    - Getreuer P. A survey of Gaussian convolution algorithms.
      https://doi.org/10.5201/ipol.2013.87
    - Gwosdek et al. Theoretical foundations of Gaussian convolution by extended box filtering.
      https://doi.org/10.1007/978-3-642-24785-9_38
    - Elboher E, Werman M. Efficient and accurate Gaussian image filtering using running sums.
      https://arxiv.org/pdf/1107.4958.pdf
    - Wells WM. Efficient synthesis of Gaussian filters by cascaded uniform filters.
      https://doi.org/10.1109/TPAMI.1986.4767776

    @sa imgFast2DGaussianFilter, imgGaussianFilter, imgSS
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
 */
int imgGaussianEBoxFilter(
  /** Image data to be processed; data is overwritten with filtered image.
      All image frames are processed. */
  IMG *img,
  /** Gaussian S.D. in pixels in x dimension; SD=FWHM/2.355. Enter 0 to not filter in x dimension. */
  float xsd,
  /** Gaussian S.D. in pixels in y dimension; SD=FWHM/2.355. Enter 0 to not filter in y dimension. */
  float ysd,
  /** Gaussian S.D. in pixels in z dimension; SD=FWHM/2.355. Enter 0 to not filter in z dimension. */
  float zsd,
  /** Number of filtering passes, usually 3-5; enter 0 to use the default steps (4). */
  int passNr,
  /** Verbose level; if zero, then only warnings are printed into stderr. */
  int verbose
) {
  if(verbose>0) printf("%s(*img, %g, %g, %g, %d, %d)\n", __func__, xsd, ysd, zsd, passNr, verbose); 

  if(img==NULL || img->dimz<1 || img->dimy<1 || img->dimx<1 || img->dimt<1) return 1;
  int dimx=img->dimx, dimy=img->dimy, dimz=img->dimz, dimt=img->dimt;
  if(dimx==1) xsd=0.0;
  if(dimy==1) ysd=0.0;
  if(dimz==1) zsd=0.0;
  if(xsd>0.0 && (dimx<4 || !(img->sizex>0.0))) return(2);
  if(ysd>0.0 && (dimy<4 || !(img->sizey>0.0))) return(3);
  if(zsd>0.0 && (dimz<4 || !(img->sizez>0.0))) return(4);
  if(!(xsd>0.0) && !(ysd>0.0) && !(zsd>0.0)) {
    if(verbose>1) printf("  no filtering done.\n");
    return(0);
  }
  if(passNr<=0) passNr=4;


  /* 
   *  Filtering in x dimension
   */
  if(xsd>0.0) {
    if(verbose>1) {printf("  filtering in x dimension\n"); fflush(stdout);}
    int rib;
    double wob, wib, s2p=xsd*xsd/(double)passNr;
    /* Calculate inner box radius (by Wells) */
    rib=0.5*sqrt(1.0 + 12.0*s2p) - 0.5;
    if(verbose>3) printf("  precise_r := %f\n", 0.5*sqrt(1.0+12.0*s2p));
    /* Alpha */
    double a=(2*rib+1)*(rib*(rib+1) - 3.0*s2p) / (6.0*(s2p - (rib+1)*(rib+1)));
    /* Outer box weight */
    wob = a/(2.0*(a+rib)+1.0);
    /* Inner box weight */
    wib = 1.0-wob;
    if(verbose>2) {printf("  rib=%d wob=%g wib=%g a=%g\n", rib, wob, wib, a); fflush(stdout);}

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      double sum1=0.0, sum2=0.0;
      for(int z=0; z<dimz; z++) for(int y=0; y<dimy; y++) {
        double d[dimx];
        for(int x=0; x<dimx; x++) sum1+=d[x]=img->m[z][y][x][t];
        filterEbox(d, dimx, wob, wib, rib, passNr);
        for(int x=0; x<dimx; x++) sum2+=img->m[z][y][x][t]=d[x];
      }
      double f=sum1/sum2;
      if(fabs(1.0-f)>1.0E-10) {
        if(verbose>1)
          printf("  frame %d: sum of voxels differs pre/post filtering %g/%g=%g\n", 1+t, sum1, sum2, f);
        for(int z=0; z<dimz; z++) for(int y=0; y<dimy; y++) for(int x=0; x<dimx; x++)
          img->m[z][y][x][t]*=f;
      }
    }
  }

  /* 
   *  Filtering in y dimension
   */
  if(ysd>0.0) {
    if(verbose>1) {printf("  filtering in y dimension\n"); fflush(stdout);}
    int rib;
    double wob, wib, s2p=ysd*ysd/(double)passNr;
    /* Calculate inner box radius */
    rib=0.5*sqrt(1.0 + 12.0*s2p) - 0.5;
    /* Alpha */
    double a=(2*rib+1)*(rib*(rib+1) - 3.0*s2p) / (6.0*(s2p - (rib+1)*(rib+1)));
    /* Outer box weight */
    wob = a/(2.0*(a+rib)+1);
    /* Inner box weight */
    wib = 1.0-wob;
    if(verbose>2) {printf("  rib=%d wob=%g wib=%g a=%g\n", rib, wob, wib, a); fflush(stdout);}

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      double sum1=0.0, sum2=0.0;
      for(int z=0; z<dimz; z++) for(int x=0; x<dimx; x++) {
        double d[dimy];
        for(int y=0; y<dimy; y++) sum1+=d[y]=img->m[z][y][x][t];
        filterEbox(d, dimy, wob, wib, rib, passNr);
        for(int y=0; y<dimy; y++) sum2+=img->m[z][y][x][t]=d[y];
      }
      double f=sum1/sum2;
      if(fabs(1.0-f)>1.0E-10) {
        if(verbose>1)
          printf("  frame %d: sum of voxels differs pre/post filtering %g/%g=%g\n", 1+t, sum1, sum2, f);
        for(int z=0; z<dimz; z++) for(int y=0; y<dimy; y++) for(int x=0; x<dimx; x++)
          img->m[z][y][x][t]*=f;
      }
    }
  }

  /* 
   *  Filtering in z dimension
   */
  if(zsd>0.0) {
    if(verbose>1) {printf("  filtering in z dimension\n"); fflush(stdout);}
    int rib;
    double wob, wib, s2p=zsd*zsd/(double)passNr;
    /* Calculate inner box radius */
    rib=0.5*sqrt(1.0 + 12.0*s2p) - 0.5;
    /* Alpha */
    double a=(2*rib+1)*(rib*(rib+1) - 3.0*s2p) / (6.0*(s2p - (rib+1)*(rib+1)));
    /* Outer box weight */
    wob = a/(2.0*(a+rib)+1);
    /* Inner box weight */
    wib = 1.0-wob;
    if(verbose>2) {printf("  rib=%d wob=%g wib=%g a=%g\n", rib, wob, wib, a); fflush(stdout);}

    for(int t=0; t<dimt; t++) { // Image time frames are processed as separate images
      if(verbose>5) {printf("    frame %d\n", 1+t); fflush(stdout);}
      double sum1=0.0, sum2=0.0;
      for(int y=0; y<dimy; y++) for(int x=0; x<dimx; x++) {
        double d[dimz];
        for(int z=0; z<dimz; z++) sum1+=d[z]=img->m[z][y][x][t];
        filterEbox(d, dimz, wob, wib, rib, passNr);
        for(int z=0; z<dimz; z++) sum2+=img->m[z][y][x][t]=d[z];
      }
      double f=sum1/sum2;
      if(fabs(1.0-f)>1.0E-10) {
        if(verbose>1)
          printf("  frame %d: sum of voxels differs pre/post filtering %g/%g=%g\n", 1+t, sum1, sum2, f);
        for(int z=0; z<dimz; z++) for(int y=0; y<dimy; y++) for(int x=0; x<dimx; x++)
          img->m[z][y][x][t]*=f;
      }
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** 1D extended box filtering .
    @sa imgFast1DGaussianFilter
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
 */
int filterEbox(
  /** Pointer to source data array. */
  double *a,
  /** Sample number. */
  int n,
  /** Weight of the outer box. */
  double wob,
  /** Weight of the inner box. */
  double wib,
  /** Radius of the inner box. */
  int rib,
  /** Number of filtering passes. */
  int passNr
) {
  if(a==NULL || n<3 || wob<=0.0 || wib<=0.0 || rib<0) return(1);
  double buf[n];
  for(int pass=0; pass<passNr; pass++) {
    double accum=0.0;
    for(int i=-rib; i<=rib; i++) accum+=a[filterHSSBE(n, i)];
    buf[0]=wob*(a[filterHSSBE(n, rib+1)] + a[filterHSSBE(n, -rib-1)]) + (wob+wib)*accum;
    accum=buf[0];
    for(int i=1; i<n; i++) {
      accum += wob*(a[filterHSSBE(n, i+rib+1)] - a[filterHSSBE(n, i-rib-2)])
             + wib*(a[filterHSSBE(n, i+rib)] - a[filterHSSBE(n, i-rib-1)]);
      buf[i]=accum;
    }
    for(int i=0; i<n; i++) a[i]=buf[i];
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** In-place mean 1D-filtering.
    @sa imgMeanFilter
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
*/
int fMean1DFilter(
  /** Pointer to array to be filtered. */
  float *data,
  /** Array size. */
  const int n,
  /** Filter size; should be odd number 3,5,7,... but not verified here. */
  const int s
) {
  if(data==NULL || n<1 || s<1) return(1);
  float buf[n];
  for(int i=0; i<n; i++) buf[i]=data[i];
  for(int i=0; i<n; i++) {
    data[i]=0.0;
    int a=i-(s-1)/2; if(a<0) a=0;
    int b=i+(s-1)/2; if(b>=n) b=n-1;
    int m=1+b-a;
    for(int j=a; j<=b; j++) data[i]+=buf[j];
    data[i]/=(float)m;
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Image mean filtering in 1-4 dimensions.

    @sa imgSmoothOverFrames, imgGaussianAMFilter, imgSS, fMean1DFilter
    @return If an error is encountered, function returns a non-zero value. Otherwise 0 is returned.
 */
int imgMeanFilter(
  /** Image data to be processed; data is overwritten with filtered image.
      All image frames are processed. */
  IMG *img,
  /** Number of adjacent pixels in x dimension; Enter 0 to not filter in x dimension. */
  int xn,
  /** Number of adjacent pixels in y dimension; Enter 0 to not filter in y dimension. */
  int yn,
  /** Number of adjacent pixels in z dimension; Enter 0 to not filter in z dimension. */
  int zn,
  /** Number of adjacent pixels in t dimension; Enter 0 to not filter in t dimension. */
  int tn,
  /** Verbose level; if zero, then only warnings are printed into stderr. */
  int verbose
) {
  if(verbose>0) printf("%s(*img, %d, %d, %d, %d)\n", __func__, xn, yn, zn, verbose); 

  if(img==NULL || img->dimz<1 || img->dimy<1 || img->dimx<1 || img->dimt<1) return(1);
  int dimx=img->dimx, dimy=img->dimy, dimz=img->dimz, dimt=img->dimt;
  if(dimx==1) xn=0;
  if(dimy==1) yn=0;
  if(dimz==1) zn=0;
  if(dimt==1) tn=0;
  if(!(xn>0) && !(yn>0) && !(zn>0) && !(tn>0)) {
    if(verbose>1) printf("  no filtering done.\n");
    return(0);
  }
  if((xn>0 && xn<3) || (yn>0 && yn<3) || (zn>0 && zn<3) || (tn>0 && tn<3)) {
    if(verbose>1) printf("  invalid pixel number.\n");
    return(1);
  }
  if((xn>0 && xn%2==0) || (yn>0 && yn%2==0) || (zn>0 && zn%2==0) || (tn>0 && tn%2==0)) {
    if(verbose>1) printf("  invalid pixel number.\n");
    return(1);
  }
  if((xn>0 && dimx<=xn) || (yn>0 && dimy<=yn) || (zn>0 && dimz<=zn) || (tn>0 && dimt<=tn)) {
    if(verbose>1) printf("  invalid pixel number.\n");
    return(1);
  }


  /* 
   *  Filtering in x dimension
   */
  if(xn>=3) {
    if(verbose>1) {printf("  filtering in x dimension\n"); fflush(stdout);}
    float buf[dimx];
    for(int z=0; z<dimz; z++)
      for(int y=0; y<dimy; y++)
        for(int t=0; t<dimt; t++) {
          for(int x=0; x<dimx; x++) buf[x]=img->m[z][y][x][t];
          if(!fMean1DFilter(buf, dimx, xn))
            for(int x=0; x<dimx; x++) img->m[z][y][x][t]=buf[x];
        }
  }

  /* 
   *  Filtering in y dimension
   */
  if(yn>=3) {
    if(verbose>1) {printf("  filtering in y dimension\n"); fflush(stdout);}
    float buf[dimy];
    for(int z=0; z<dimz; z++)
      for(int x=0; x<dimx; x++)
        for(int t=0; t<dimt; t++) {
          for(int y=0; y<dimy; y++) buf[y]=img->m[z][y][x][t];
          if(!fMean1DFilter(buf, dimy, yn))
            for(int y=0; y<dimy; y++) img->m[z][y][x][t]=buf[y];
        }
  }

  /* 
   *  Filtering in z dimension
   */
  if(zn>=3) {
    if(verbose>1) {printf("  filtering in z dimension\n"); fflush(stdout);}
    float buf[dimz];
    for(int y=0; y<dimy; y++)
      for(int x=0; x<dimx; x++)
        for(int t=0; t<dimt; t++) {
          for(int z=0; z<dimz; z++) buf[z]=img->m[z][y][x][t];
          if(!fMean1DFilter(buf, dimz, zn))
            for(int z=0; z<dimz; z++) img->m[z][y][x][t]=buf[z];
        }
  }

  /* 
   *  Filtering in t dimension
   */
  if(tn>=3) {
    if(verbose>1) {printf("  filtering in t dimension\n"); fflush(stdout);}
    float buf[dimt];
    for(int z=0; z<dimz; z++)
      for(int y=0; y<dimy; y++)
        for(int x=0; x<dimx; x++) {
          for(int t=0; t<dimt; t++) buf[t]=img->m[z][y][x][t];
          if(!fMean1DFilter(buf, dimt, tn))
            for(int t=0; t<dimt; t++) img->m[z][y][x][t]=buf[t];
        }
  }


  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
