/// @file imgflips.c
/// @author Vesa Oikonen
/// @brief Functions for turning IMG image volume data.
///
/*****************************************************************************/
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data horizontally (left-right).
    @sa imgFlipRight, imgFlipVertical, imgFlipPlanes, imgFlipAbove, img2cube
 */
void imgFlipHorizontal(IMG *img)
{
  int zi, yi, from, to;
  float *col_ptr;

  for(zi=0; zi<img->dimz; zi++) {
    for(yi=0; yi<img->dimy; yi++) {
      for(from=0, to=img->dimx-1; from<to; from++, to--) {
        col_ptr=img->m[zi][yi][from];
        img->m[zi][yi][from]=img->m[zi][yi][to];
        img->m[zi][yi][to]=col_ptr;
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data vertically (up-down).
    @sa imgFlipHorizontal, imgFlipRight, imgFlipPlanes, imgFlipAbove
 */
void imgFlipVertical(IMG *img)
{
  int zi, from, to;
  float **row_ptr;

  for(zi=0; zi<img->dimz; zi++) {
    for(from=0, to=img->dimy-1; from<to; from++, to--) {
      row_ptr=img->m[zi][from];
      img->m[zi][from]=img->m[zi][to];
      img->m[zi][to]=row_ptr;
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data planes (head-toes). 

    To work properly, the plane numbers must be contiguous.
    @sa imgFlipHorizontal, imgFlipVertical, imgFlipRight, imgFlipAbove
 */
void imgFlipPlanes(IMG *img)
{
  int from, to;
  float ***plane_ptr;

  for(from=0, to=img->dimz-1; from<to; from++, to--) {
    plane_ptr=img->m[from];
    img->m[from]=img->m[to];
    img->m[to]=plane_ptr;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data like viewed from right side.
    @sa imgFlipHorizontal, imgFlipVertical, imgFlipPlanes, imgFlipAbove
    @return Returns 0 if successful.
 */
int imgFlipRight(
  /** Pointer to IMG which will be flipped */
  IMG *img
) {
  int xi, yi, zi, fi, ret;
  IMG omg;

  /* Make copy of the original image */
  imgInit(&omg); ret=imgDup(img, &omg); if(ret!=0) return(100+ret);

  /* Empty the user-specified image struct */
  imgEmpty(img);
  /* Allocate it again with new dimensions */
  ret=imgAllocateWithHeader(img, omg.dimx, omg.dimy, omg.dimz, omg.dimt, &omg);
  if(ret!=0) {imgEmpty(&omg); return(200+ret);}

  /* Copy the voxel values */
  for(zi=0; zi<omg.dimz; zi++)
    for(yi=0; yi<omg.dimy; yi++)
      for(xi=0; xi<omg.dimx; xi++)
        for(fi=0; fi<omg.dimt; fi++) {
          img->m[xi][yi][zi][fi]=omg.m[zi][yi][xi][fi];
        }

  /* Switch pixel sizes */
  img->sizex=omg.sizez;
  img->sizez=omg.sizex;
  img->resolutionx=omg.resolutionz;
  img->resolutionz=omg.resolutionx;

  /* Free the memory of the copy of original image */
  imgEmpty(&omg);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data like viewed from above.
    @sa imgFlipHorizontal, imgFlipVertical, imgFlipPlanes, imgFlipRight, img2cube, imgMeanZ
    @return Returns 0 if successful.
 */
int imgFlipAbove(
  /** Pointer to IMG which will be flipped. */
  IMG *img
) {
  int xi, yi, zi, fi, ret;
  IMG omg;

  /* Make copy of the original image */
  imgInit(&omg); ret=imgDup(img, &omg); if(ret!=0) return(100+ret);

  /* Empty the user-specified image structure. */
  imgEmpty(img);
  /* Allocate it again with new dimensions */
  ret=imgAllocateWithHeader(img, omg.dimy, omg.dimz, omg.dimx, omg.dimt, &omg);
  if(ret!=0) {imgEmpty(&omg); return(200+ret);}

  /* Copy the voxel values */
  for(zi=0; zi<omg.dimz; zi++)
    for(yi=0; yi<omg.dimy; yi++)
      for(xi=0; xi<omg.dimx; xi++)
        for(fi=0; fi<omg.dimt; fi++) {
          img->m[yi][img->dimy-1-zi][xi][fi]=omg.m[zi][yi][xi][fi];
        }

  /* Switch pixel sizes */
  img->sizey=omg.sizez;
  img->sizez=omg.sizey;
  img->resolutiony=omg.resolutionz;
  img->resolutionz=omg.resolutiony;

  /* Free the memory of the copy of original image */
  imgEmpty(&omg);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate image average over z dimension (image planes).
    @sa imgFlipAbove, img2cube
    @return Returns 0 if successful.
 */
int imgMeanZ(
  /** Pointer to input IMG data structure; not modified. */
  IMG *img1,
  /** Pointer to output IMG data structure; allocated here. */
  IMG *img2
) {
  if(img1==NULL || img2==NULL) return(1);
  if(img1->dimz<1 || img1->dimy<1 || img1->dimx<1 || img1->dimt<1) return(1);

  /* If z dim is already 1, then just duplicate the image */
  if(img1->dimz==1) return(imgDup(img1, img2));

  /* Allocate memory for the output image */
  if(imgAllocateWithHeader(img2, 1, img1->dimy, img1->dimx, img1->dimt, img1)!=0)
    return(3);

  /* Sum over z dimension */
  for(int zi=0; zi<img1->dimz; zi++)
    for(int yi=0; yi<img1->dimy; yi++)
      for(int xi=0; xi<img1->dimx; xi++)
        for(int ti=0; ti<img1->dimt; ti++)
          img2->m[0][yi][xi][ti]+=img1->m[zi][yi][xi][ti];

  /* Divide by original z dimension */
  for(int yi=0; yi<img2->dimy; yi++)
    for(int xi=0; xi<img2->dimx; xi++)
      for(int ti=0; ti<img2->dimt; ti++)
        img2->m[0][yi][xi][ti]/=(float)img1->dimz;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
