/******************************************************************************
 * This file is not compiled into the library, but it contains main()
 * which is compiled to an executable, used to test the library functions. 
 *****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
/* Test function declarations */
/** Function for testing library functions related to image arithmetics */
int test_imgArithm(IMG *img1, IMG *img2, int VERBOSE);
/** Function for testing library functions related to image transforms */
int test_imgTransform(IMG *img1, IMG *img2, int zoom);

int test_imgMaskRegionLabeling(int verbose);
int test_imgMaskFloodFill(int verbose);

/* Functions for initializing test image data */
/** Function for drawing a cros on the center of the image region,
    of width 2 pixels and value 1.
*/
void drawCross(IMG *img);
/** Function for drawing an unsymmetric cross for testing image transformation
    functions.
*/
void drawUnSymmetricCross(IMG *img);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Usage: @P [options]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " -t, --test",
  "     Run all tests for library functions.",
  0};
/*****************************************************************************/

/*****************************************************************************/
/** Run unit tests to the library functions
 *  @author Vesa Oikonen
 *  @return 0 if all tests pass, otherwise >0.
 * */
int main(
  /** Nr of arguments */
  int argc,
  /** Pointer to arrays of argument string */
  char *argv[ ]
) {
  int i, help=0, version=0, verbose=1, error=0, test=0;
  int ret;
  char *cptr;

  if(argc==1) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  for(i=1; i<argc; i++) {
    if(tpcProcessStdOptions(argv[i], &help, &version, &verbose)==0) continue;
    cptr=argv[i]; if(*cptr=='-') cptr++; if(*cptr=='-') cptr++;
    if(strncasecmp(cptr, "TEST", 1)==0) {
      test=1; continue;
    } else {
      error++; break;
    }
  }
  if(error>0) {
    fprintf(stderr, "Error: specify --help for usage.\n");
    return(1);
  }
  /* Print help or version? */
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  if(test==0) return(0);

  if(verbose>0) printf("running tests for library functions...\n");
  int zoom=2, plane;
  int planes1, rows1, columns1, frames1, planes2, rows2, columns2, frames2;
  IMG img1, img2;

  /* Make space for test images */
  imgInit(&img1); imgInit(&img2);
  // Set dimensions for test images:
  planes1 = 1;
  rows1 = 32;
  columns1 = 32;
  frames1 = 1;
  planes2 = planes1;
  rows2 = rows1*zoom;
  columns2 = columns1*zoom;
  frames2 = frames1;

  if(imgAllocate(&img1,planes1,rows1,columns1,frames1)){
    printf("    \nError in image allocation: imgAllocate() @ img.c\n");
    return(1);
  }
  if(imgAllocate(&img2,planes2,rows2,columns2,frames2)){
    printf("\n    Error in image allocation: imgAllocate() @ img.c\n");
    return(1);
  }
    
  for(plane = 0; plane < img1.dimz; plane++)
    img1.planeNumber[plane] = img2.planeNumber[plane] = plane;
    
  img1.sizex=img1.sizey=img1.sizez=img2.sizex=img2.sizey=img2.sizez=1.0;
  img1.type=img2.type=IMG_TYPE_IMAGE;


  i=10;
  /* imgarithm */
  i++; if((ret=test_imgArithm(&img1, &img1, verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}

  /* imgtransform */
  i++; if((ret=test_imgTransform(&img1, &img2, zoom))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}
  imgEmpty(&img1); imgEmpty(&img2);

  /* mask */
  i++; if((ret=test_imgMaskFloodFill(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}
  i++; if((ret=test_imgMaskRegionLabeling(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}

  if(verbose>0) printf("\nAll tests passed.\n\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_imgArithm(
  IMG *img1, IMG *img2, int VERBOSE
) {
  int error_code = 0, plane, row, column, frame;
  float operand;
  char operation = 0;
  float ****cross_mask;

  printf("\n    Testing image arithmetics functions...\n");

  // Initialize the input image with an unsymmetric cross:
  drawUnSymmetricCross(img1);
  if(VERBOSE) printf("\n    Initial cross drawn. \n");
    
  // Create a mask for checking the results:
  cross_mask = malloc(img1->dimz * sizeof(float***));
  for(plane = 0; plane < img1->dimz; plane++){
    cross_mask[plane] = malloc(img1->dimy * sizeof(float**));
    for(row = 0; row < img1->dimy; row++){
      cross_mask[plane][row] = malloc(img1->dimx * sizeof(float*));
      for(column = 0; column < img1->dimx; column++)
        cross_mask[plane][row][column] = malloc(img1->dimt*sizeof(float));
    }
  }
  // Copy the image buffer to mask buffer:
  for(plane = 0; plane < img1->dimz; plane++)
    for(row = 0; row < img1->dimy; row++)
      for(column = 0; column < img1->dimx; column++) 
        for(frame = 0; frame < img1->dimt; frame++)
          cross_mask[plane][row][column][frame] = img1->m[plane][row][column][frame];
          
  if(VERBOSE) printf("\n    Mask created succesfully.\n");
    
  // Sum and check (expected result = 2*mask):
  operation = '+';
  error_code = imgArithm(img1,img2,operation,10000000,0);
  if(!error_code){
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
            if((img1->m[plane][row][column][frame] !=
                2*cross_mask[plane][row][column][frame]))
              {
                error_code = 101;
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                      plane,row,column,frame,img1->m[plane][row][column][frame]);
              }
            }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithm() sum operation failed with error code: %i\n",
          error_code);
    return(error_code);
  }
  else{
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithm() sum operation exited with: %i\n",error_code);
  }

  // Subtract and check (expected result = 0):
  operation = '-';
  error_code = imgArithm(img1,img2,operation,10000000,0);
  if(!error_code){
  for(plane = 0; plane < img1->dimz; plane++)
    for(row = 0; row < img1->dimy; row++)
      for(column = 0; column < img1->dimx; column++)
        for(frame = 0; frame < img1->dimt; frame++){
          if(img1->m[plane][row][column][frame] != 0){
            error_code = 102;
            if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                    plane,row,column,frame,img1->m[plane][row][column][frame]);
          }
        }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithm() subtract operation failed with error code: %i\n",
        error_code);
    return(error_code);
  } else {
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithm() subtract operation exited with: %i\n",
         error_code);
  }

  // Initialize the input image with an unsymmetric cross:
  drawUnSymmetricCross(img1);

  // Multiply and check (expected result = mask*mask):
  operation = 'x';
  error_code = imgArithm(img1,img1,operation,10000000,0);
  if(!error_code){
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
              if(img1->m[plane][row][column][frame] !=
                 cross_mask[plane][row][column][frame] * cross_mask[plane][row][column][frame])
              {
                error_code = 103;
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                     plane,row,column,frame,img1->m[plane][row][column][frame]);
              }
          }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithm() multiplication operation failed with error code: %i\n",
         error_code);
    return(error_code);
  } else {
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithm() multiplication operation exited with: %i\n",
         error_code);
  }
  
  // Devide and check (expected result = 0 or 1):
  if(!error_code){
    operation = '/';
    error_code = imgArithm(img1,img1,operation,10000000,0);
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
            if(cross_mask[plane][row][column][frame] != 0){
              if(img1->m[plane][row][column][frame] != 1){
                error_code = 104;
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                     plane,row,column,frame,img1->m[plane][row][column][frame]);
              }
            } else {
              if(img1->m[plane][row][column][frame] != 0){
                error_code = 105; // Error in deviding with zero.
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                     plane,row,column,frame,img1->m[plane][row][column][frame]);
              }
            }
          }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithm() division operation failed with error code: %i\n",
        error_code);
    return(error_code);
  } else{
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithm() division operation exited with: %i\n",
          error_code);
  }
  
  //// Check constant operations:

  // Initialize the input image with an unsymmetric cross:
  drawUnSymmetricCross(img1);

  // Add one and check (expected result = mask + 1):
  operation = '+';
  operand = 1.;
  error_code = imgArithmConst(img1,operand,operation,10000000,0);
  if(!error_code){
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
              if((img1->m[plane][row][column][frame] !=
                  cross_mask[plane][row][column][frame] + 1))
              {
                error_code = 106;
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                    plane,row,column,frame,img1->m[plane][row][column][frame]);
              }
          }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithmConst() sum operation failed with error code: %i\n",
           error_code);
    return(error_code);
  } else {
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithmConst() sum operation exited with: %i\n",
           error_code);
  }

  // Subtract one from previous image buffer and check (expected result = mask):
  operation = '-';
  operand = 1.;
  error_code = imgArithmConst(img1,operand,operation,10000000,0);
  if(!error_code){
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
              if(img1->m[plane][row][column][frame] != cross_mask[plane][row][column][frame]){
                error_code = 107;
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                     plane,row,column,frame,img1->m[plane][row][column][frame]);
              }
          }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithmConst() subtract operation failed with error code: %i\n",
           error_code);
    return(error_code);
  } else {
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithmConst() subtract operation exited with: %i\n",
            error_code); 
  }

  // Divide by 2 and check (expected result = mask / 2):
  operation = '/';
  operand = 2.;
  error_code = imgArithmConst(img1,operand,operation,10000000,0);
  if(error_code == 0){
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
            if(img1->m[plane][row][column][frame] != 
                 (cross_mask[plane][row][column][frame] / 2)){
                error_code = 108;
                if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                     plane,row,column,frame,img1->m[plane][row][column][frame]);
            }
          }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithmConst() division operation failed with error code: %i\n",
           error_code);
    return(error_code);
  } else{
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithmConst() division operation exited with: %i\n",
            error_code); 
  }

  // Multiply by 2 and check (expected result = mask):
  operation = 'x';
  operand = 2.;
  error_code = imgArithmConst(img1,operand,operation,10000000,0);
  if(!error_code){
    for(plane = 0; plane < img1->dimz; plane++)
      for(row = 0; row < img1->dimy; row++)
        for(column = 0; column < img1->dimx; column++)
          for(frame = 0; frame < img1->dimt; frame++){
            if(img1->m[plane][row][column][frame] != cross_mask[plane][row][column][frame]){
              error_code = 109;
              if(VERBOSE) printf("    m[%i][%i][%i][%i] = %f\n",
                     plane,row,column,frame,img1->m[plane][row][column][frame]);
            }
          }
  }
  if(error_code){
    if(VERBOSE) 
      printf("\n    Test FAILED: imgArithmConst() multiplication operation failed with error code: %i\n",
           error_code);
    return(error_code);
  } else{
    if(VERBOSE) 
      printf("\n    Test SUCCESFULL: imgArithmConst() multiplication operation exited with: %i\n",
           error_code); 
  }

  printf("\n    Test SUCCESFULL: imgArithm and imgArithmConst() operations exited with: %i\n",
            error_code); 

  return(error_code);
}
/*****************************************************************************/

/*****************************************************************************/
int test_imgTransform(
  IMG *img1, IMG *img2, int zoom
) {
  int error_code = 0;
  int plane, row, column, frame, r_limit, c_limit, cross_width;

  printf("\n    Testing image transformation functions...\n");
  int verbose=0;

  // Initialize the input image with an un-symmetric cross:
  drawUnSymmetricCross(img1);

  // Scale the first given image and write to second:
  imgScale(img1,img2,zoom,0);

  // Check the scaled cross:
  cross_width = 2*zoom;
  r_limit = (img2->dimy - cross_width)/2;
  c_limit = (img2->dimx - cross_width)/2;
  for(plane = 0; plane < img2->dimz; plane++) {
    for(row = 0; row < img2->dimy; row++) {
      for(column = 0; column < img2->dimx; column++) {
        for(frame = 0; frame < img2->dimt; frame++) {
          if((row >= r_limit) && (row <= r_limit + (cross_width - 1))) {
            if(column < c_limit) // left 
              if(img2->m[plane][row][column][frame] != 1)
                error_code = 10;
            if(column > c_limit + (cross_width - 1)) // right
              if(img2->m[plane][row][column][frame] != 3)
                error_code = 20;
            if((column >= c_limit) && (column <= c_limit + (cross_width - 1))) //centre
              if(img2->m[plane][row][column][frame] != 5)
                error_code = 30;
          } else {
            if((column >= c_limit) && (column <= c_limit + (cross_width - 1))) {
              if(row < r_limit) // top
                if(img2->m[plane][row][column][frame] != 4) {
                  if(verbose>0) printf("img2.m[%d][%d][%d][%d]: %g vs %g\n", 
                         plane, row, column, frame, img2->m[plane][row][column][frame], 4.0);
                  error_code = 40;
                }
              if(row > r_limit) // bottom
                if(img2->m[plane][row][column][frame] != 2) {
                  if(verbose>0) printf("img2.m[%d][%d][%d][%d]: %g vs %g\n", 
                         plane, row, column, frame, img2->m[plane][row][column][frame], 2.0);
                  error_code = 50;
                }
            } else{
              if(img2->m[plane][row][column][frame] != 0) {
                if(verbose>0) printf("img2.m[%d][%d][%d][%d]: %g vs %g\n", 
                       plane, row, column, frame, img2->m[plane][row][column][frame], 0.0);
                error_code = 60;
              }
            }
          }
          if(error_code>0) break;
        }
        if(error_code>0) break;
      }
      if(error_code>0) break;
    }
    if(error_code>0) break;
  }
  if(error_code) {
    printf("\n Test FAILED: imgScale() by factor %i failed with error code: %i\n", zoom, error_code);
    if(verbose>1) {
      printf("\n");
      for(int yi=0; yi<img1->dimy; yi++) {
        for(int xi=0; xi<img1->dimx; xi++) printf("%g", img1->m[plane][yi][xi][frame]);
        printf("\n");
      }
      printf("->\n");
      for(int yi=0; yi<img2->dimy; yi++) {
        for(int xi=0; xi<img2->dimx; xi++) printf("%g", img2->m[plane][yi][xi][frame]);
        printf("\n");
      }
      printf("\n");
    }
  } else {
    printf("\n    Test SUCCESFULL: imgScale() by factor %i exited with: %i\n", zoom, error_code); 
  }



  return(error_code);
}
/*****************************************************************************/

/*****************************************************************************/
/** Draw a symmetric cross on all planes and frames in the image */
void drawCross(
  IMG *img
) {
  int plane, row, column, frame, r_limit, c_limit;

  r_limit = (img->dimy - 2)/2;
  c_limit = (img->dimx - 2)/2;
  for(plane = 0; plane < img->dimz; plane++) {
    for(row = 0; row < img->dimy; row++) {
      for(column = 0; column < img->dimx; column++) {
        for(frame = 0; frame < img->dimt; frame++) {
          if((row >= r_limit) && (row <= r_limit + 1))
            img->m[plane][row][column][frame] = 1;
          if((column >= c_limit) && (column <= c_limit + 1)
              && !((row >= r_limit) && (row <= r_limit + 1)))
            img->m[plane][row][column][frame] = 1;
        }
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Draw an unsymmetric cross on all planes and frames in the image */
void drawUnSymmetricCross(
  IMG *img
) {
  int plane, row, column, frame, r_limit, c_limit;

  r_limit = (img->dimy - 2)/2;
  c_limit = (img->dimx - 2)/2;
  for(plane = 0; plane < img->dimz; plane++) {
    for(row = 0; row < img->dimy; row++) {
      for(column = 0; column < img->dimx; column++) {
        for(frame = 0; frame < img->dimt; frame++){
          if((row >= r_limit) && (row <= r_limit + 1)){
            if(column < c_limit) // left 
              img->m[plane][row][column][frame] = 1;
            if(column > c_limit + 1) // right
              img->m[plane][row][column][frame] = 3;
            if((column >= c_limit) && (column <= c_limit + 1)) //center
              img->m[plane][row][column][frame] = 5;
          } else {
            if((column >= c_limit) && (column <= c_limit + 1)){             
              if(row < r_limit) // top
                img->m[plane][row][column][frame] = 4;
              if(row > r_limit) // bottom
                img->m[plane][row][column][frame] = 2;
            } else {
              img->m[plane][row][column][frame] = 0;
            }
          }
        }
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Tests for imgMaskFloodFill() */
int test_imgMaskFloodFill(
  /** Verbose level; enter 0 to print nothing. */ 
  int verbose
) {
  if(verbose>0) printf("\n Testing imgMaskFloodFill()\n");

  if(imgMaskFloodFill(NULL, 0, 0, 0, 0, NULL, verbose)==0) return(1);

  IMG img; imgInit(&img);
  if(imgMaskFloodFill(&img, 0, 0, 0, 0, NULL, verbose)==0) return(2);

  if(imgAllocate(&img, 10, 10, 10, 1)!=0) return(3);

  if(imgMaskFloodFill(&img, 0, 0, 0, 2, NULL, verbose)==0) {imgEmpty(&img); return(11);}

  img.m[1][0][0][0]=1.0;
  img.m[2][0][0][0]=1.0;
  img.m[3][0][0][0]=1.0;

  img.m[4][4][4][0]=1.0;
  img.m[4][4][5][0]=1.0;
  img.m[4][5][5][0]=1.0;

  img.m[2][7][8][0]=1.0;
  img.m[2][8][8][0]=1.0;
  img.m[2][5][8][0]=1.0;

  if(imgMaskFloodFill(&img, 2, 0, 0, 1, NULL, verbose)==0) {imgEmpty(&img); return(21);}
  long long n=0;
  if(imgMaskFloodFill(&img, 4, 4, 5, 2, &n, verbose)!=0) {imgEmpty(&img); return(22);}
  if(n!=3) {imgEmpty(&img); return(23);}
  if(imgMaskFloodFill(&img, 4, 4, 5, 2, &n, verbose)==0) {imgEmpty(&img); return(24);}
  if(n!=0) {imgEmpty(&img); return(25);}


  imgEmpty(&img);
  if(verbose>0) printf("   ok\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Tests for imgMaskRegionLabeling() */
int test_imgMaskRegionLabeling(
  /** Verbose level; enter 0 to print nothing. */ 
  int verbose
) {
  if(verbose>0) printf("\n Testing imgMaskRegionLabeling()\n");

  if(imgMaskRegionLabeling(NULL, NULL, NULL, verbose)==0) return(1);

  IMG img; imgInit(&img);
  if(imgMaskRegionLabeling(&img, &img, NULL, verbose)==0) return(2);

  IMG img2; imgInit(&img2);
  if(imgMaskRegionLabeling(&img, &img2, NULL, verbose)==0) return(3);

  if(imgAllocate(&img, 10, 10, 10, 1)!=0) return(3);
  if(imgMaskRegionLabeling(&img, &img, NULL, verbose)==0) {imgEmpty(&img); return(4);}

  int n=0;
  if(imgMaskRegionLabeling(&img, &img2, &n, verbose)!=0) {imgEmpty(&img); return(11);}
  if(n!=0) {imgEmpty(&img); return(12);}

  img.m[1][0][0][0]=1.0;
  img.m[2][0][0][0]=1.0;
  img.m[3][0][0][0]=1.0;

  img.m[4][4][4][0]=1.0;
  img.m[4][4][5][0]=1.0;
  img.m[4][5][5][0]=1.0;

  img.m[2][7][8][0]=1.0;
  img.m[2][8][8][0]=1.0;
  img.m[2][5][8][0]=1.0;

  if(imgMaskRegionLabeling(&img, &img2, &n, verbose)!=0) {
    imgEmpty(&img); imgEmpty(&img2); return(21);}
  if(n!=4) {imgEmpty(&img); imgEmpty(&img2); return(22);}
  int ret=0;
  if(img2.m[1][0][0][0]!=2.0) ret++;
  if(img2.m[2][0][0][0]!=2.0) ret++;
  if(img2.m[3][0][0][0]!=2.0) ret++;

  if(img2.m[2][5][8][0]!=3.0) ret++;

  if(img2.m[2][7][8][0]!=4.0) ret++;
  if(img2.m[2][8][8][0]!=4.0) ret++;

  if(img2.m[4][4][4][0]!=5.0) ret++;
  if(img2.m[4][4][5][0]!=5.0) ret++;
  if(img2.m[4][5][5][0]!=5.0) ret++;

  if(ret>0) {
    for(int zi=0; zi<img2.dimz; zi++)
      for(int yi=0; yi<img2.dimy; yi++)
        for(int xi=0; xi<img2.dimx; xi++)
          if(img2.m[zi][yi][xi][0]!=0.0)
            printf("  %d,%d,%d = %g\n", zi, yi, xi, img2.m[zi][yi][xi][0]);
    imgEmpty(&img); imgEmpty(&img2); return(23);
  }


  imgEmpty(&img); imgEmpty(&img2);
  if(verbose>0) printf("   ok\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
