/******************************************************************************
 * This file is not compiled into the library, but it contains main()
 * which is compiled to an executable, used to test the library functions. 
 *****************************************************************************/

/*****************************************************************************/
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
/* Test function declarations */
int test_stump(int verbose);
int test_noise(int verbose);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Usage: @P [options]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " -t, --test",
  "     Run all tests for library functions.",
  0};
/*****************************************************************************/

/*****************************************************************************/
/** Run unit tests to the library functions
 *  @author Vesa Oikonen
 *  @return 0 if all tests pass, otherwise >0.
 * */
int main(
  /** Nr of arguments */
  int argc,
  /** Pointer to arrays of argument string */
  char *argv[ ]
) {
  int i, help=0, version=0, verbose=1, error=0, test=0;
  int ret;
  char *cptr;

  if(argc==1) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  for(i=1; i<argc; i++) {
    if(tpcProcessStdOptions(argv[i], &help, &version, &verbose)==0) continue;
    cptr=argv[i]; if(*cptr=='-') cptr++; if(*cptr=='-') cptr++;
    if(strncasecmp(cptr, "TEST", 1)==0) {
      test=1; continue;
    } else {
      error++; break;
    }
  }
  if(error>0) {
    fprintf(stderr, "Error: specify --help for usage.\n");
    return(1);
  }
  /* Print help or version? */
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  if(test==0) return(0);

  if(verbose>0) printf("running tests for library functions...\n");

  i=10;
  /* imgarithm */
  i++; if((ret=test_noise(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}
  /* imgtransform */
  i++; if((ret=test_stump(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}

  if(verbose>0) printf("\nAll tests passed.\n\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_stump(int VERBOSE)
{
  char *funcname="test_stump()";
  int error_code = 0;

  if(VERBOSE) printf("\n%s\n", funcname);

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_noise(int VERBOSE)
{
  char funcname[128];
  int error_code = 0;
  int ret;
  double sd;


  if(VERBOSE) printf("\ntest_noise()\n");
  {
  strcpy(funcname, "noiseSD4Simulation()");

  ret=noiseSD4Simulation(100.0, 0.0, 10.0, 20.4, 20.0, &sd, NULL, 0);
  if(ret!=0) error_code+=1;
  else if(sd<15.3 || sd>15.5) error_code+=1;

  ret=noiseSD4Simulation(10.0, 10.0, 10.0, 20.4, 20.0, &sd, NULL, 0);
  if(ret!=0) error_code+=2;
  else if(sd<5.7 || sd>5.8) error_code+=2;

  ret=noiseSD4Simulation(10.0, 20.0, 30.0, 20.4, 20.0, &sd, NULL, 0);
  if(ret!=0) error_code+=4;
  else if(sd<4.5 || sd>4.6) error_code+=4;

  ret=noiseSD4Simulation(12.5, 61.2, 10.0, 20.4, 20.0, &sd, NULL, 0);
  if(ret!=0) error_code+=8;
  else if(sd<15.3 || sd>15.5) error_code+=4;

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 
  }
  if(error_code) return(error_code);

  {
  strcpy(funcname, "noiseSD4SimulationFromDFT()");
  DFT dft;
  int i;

  dftInit(&dft); dftSetmem(&dft, 4, 2);
  strcpy(dft.isotope, "C-11");
  dft.timeunit=TUNIT_MIN;
  dft.timetype=DFT_TIME_STARTEND;
  dft.frameNr=4; dft.voiNr=1;
  dft.x1[0]=0.0;  dft.x2[0]=10.0;  dft.voi[0].y[0]=100.0;
  dft.x1[1]=10.0; dft.x2[1]=20.0;  dft.voi[0].y[1]=10.0;
  dft.x1[2]=20.0; dft.x2[2]=50.0;  dft.voi[0].y[2]=10.0;
  dft.x1[3]=61.2; dft.x2[3]=71.2;  dft.voi[0].y[3]=12.5;
  ret=noiseSD4SimulationFromDFT(&dft, -1, 20.0, dft.w, NULL, 0);
  if(ret!=0) error_code=1;
  else {
    if(dft.w[0]<15.3 || dft.w[0]>15.5) error_code=1;
    if(dft.w[1]<5.7  || dft.w[1]>5.8) error_code=1;
    if(dft.w[2]<4.5  || dft.w[2]>4.6) error_code=1;
    if(dft.w[3]<15.3 || dft.w[3]>15.5) error_code=1;
  }

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code);

  dft.voiNr=2;
  ret=noiseSD4SimulationFromDFT(&dft, 0, 20.0, dft.w, NULL, 0);
  if(ret!=0) error_code=2;
  else {
    if(dft.w[0]<15.3 || dft.w[0]>15.5) error_code=2;
    if(dft.w[1]<5.7  || dft.w[1]>5.8) error_code=2;
    if(dft.w[2]<4.5  || dft.w[2]>4.6) error_code=2;
    if(dft.w[3]<15.3 || dft.w[3]>15.5) error_code=2;
  }

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code);

  dft.voiNr=2;
  for(i=0; i<4; i++) {
    dft.voi[1].y[i]=2.0*dft.voi[0].y[i]; dft.voi[0].y[i]=0.0;
  }
  ret=noiseSD4SimulationFromDFT(&dft, -1, 20.0, dft.w, NULL, 0);
  if(ret!=0) {
    error_code=4;
    if(VERBOSE) printf("  ret := %d\n", ret);
  } else {
    if(dft.w[0]<15.3 || dft.w[0]>15.5) error_code=4;
    if(dft.w[1]<5.7  || dft.w[1]>5.8) error_code=4;
    if(dft.w[2]<4.5  || dft.w[2]>4.6) error_code=4;
    if(dft.w[3]<15.3 || dft.w[3]>15.5) error_code=4;
  }

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code);
  
  dftEmpty(&dft);
  }
  return(error_code);
}
/*****************************************************************************/

/*****************************************************************************/
