/// @file libtpcrec.h
/// @brief Header file for libtpcrec.
/// @author Vesa Oikonen
///
#ifndef _LIBTPCREC_H
#define _LIBTPCREC_H
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <ctype.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
/* ellipse */
enum {ELLIPSE_STATUS_UNINITIALIZED, ELLIPSE_STATUS_INITIALIZED,
  ELLIPSE_STATUS_OCCUPIED, ELLIPSE_STATUS_ERROR};

/// Drive in test mode if not 0.
extern int ELLIPSE_TEST;
/// Drive in verbose mode if not 0.
extern int ELLIPSE_VERBOSE;

/** @brief Ellipse on two dimensional plane. 
    @details Ellipse is defined on an n x n image grid by the centre coordinate
    and the length of the semiaxis. */
typedef struct {
  /// ellipse status.
  char status;
  /* Ellipse */
  /** Semiaxis of the ellipse.
      semiaxis[0]=major semiaxe, semiaxis[1]=minor semiaxe. */
  float semiaxis[2]; 
  /** Center of the ellipse. center[0]=x-coordinate, 
      center[1]=y-coordinate in cartesian coordinate system. */
  float center[2];
  /// Inclination (degrees).
  float inclination;
  /// Size of the image plane on which the ellipse is defined.
  int imageDim;
  /** Ellipse array containing n x n entries. Single entry is one
      if it is inside the ellipse and zero otherwise. Coordinates on a
      two dimensional plane are numbered from left to right and from 
      top to bottom (not cartesian). */
  int **ellipseptr;
  /// Value inside the ellipse.
  float value;
} ELLIPSE;

extern void ellipseInit(ELLIPSE *ell);
extern void ellipseEmpty(ELLIPSE *ell);
extern void ellipseInfo(ELLIPSE *ell);
extern int ellipseAllocate(ELLIPSE *ell, int imgDim);
extern int ellipseSetFromParams(
  ELLIPSE *ell, int imgDim, float *semis, float *cent, float incli, float val
);
extern int ellipseReadEllipse(FILE *fp, ELLIPSE *ell);
extern int ellipseSaveEllipse(ELLIPSE *ell, FILE *fp);
extern float ellipseGetMajor(ELLIPSE *ell);
extern float ellipseGetMinor(ELLIPSE *ell);
extern float ellipseGetCenterX(ELLIPSE *ell);
extern float ellipseGetCenterY(ELLIPSE *ell);
extern float ellipseGetInclination(ELLIPSE *ell);
extern int ellipseGetImgSize(ELLIPSE *ell);
extern int ellipseGetValue(ELLIPSE *ell);
extern int** ellipseGetArray(ELLIPSE *ell);
extern int ellipseIsInside(ELLIPSE *ell, int row, int col);
/*****************************************************************************/

/*****************************************************************************/
/* fbp */
enum {FBP_FILTER_NONE, FBP_FILTER_RAMP, FBP_FILTER_BUTTER, FBP_FILTER_HANN,
  FBP_FILTER_HAMM, FBP_FILTER_PARZEN, FBP_FILTER_SHEPP    
};

extern int fbp(
  float *sinogram, int rays, int views, int dim, int filter, float cutoff,
  float *image
);
extern int imgFBP(
  IMG *scn, IMG *img, int imgDim, float zoom, int filter, float cutoff,
  float shiftX, float shiftY, float rotation, int verbose
);
extern int fbpMakeFilter(
  float *sinFFT, float *cosFFT, float cutoff, int filter, int lenFFT, int views,
  float *fbpFilter, int verbose
);
extern void fbp_fft_bidir_complex_radix2(
  float *real, float *imag, int direct, int n, float *sine, float *cosine
);
extern void fbp_back_proj(
  float *prj, float *imgCorner, int imgDim, int view, int views,
  int rays, float offsX, float offsY, float *sinB, float *sinBrot
);
extern void fbp_back_proj_round(
  float *prj, float *imgOrigin, int imgDim, int view,
  int views, int rays, float offsX, float offsY, float bpZoom,
  float *sinB, float *sinBrot
);
/*****************************************************************************/

/*****************************************************************************/
/* mrp */
extern int imgMRP(
  IMG *scn, IMG *img, int imgDim, float zoom,
  float shiftX, float shiftY, float rotation,
  int maxIterNr, int skipPriorNr,
  float beta, int maskDim, int osSetNr,
  int verbose
);
extern void mrpUpdate(
  float *coef, float *img, float *oimg, int n
);
extern void mrpProjectionCorrection(
  float *measured, float *proj, float *correct, 
  int os_sets, int rays, int views
);
extern int mrp(
  float *sinogram, int rays, int views,
  int iter, int os_sets, int maskdim, float zoom, float beta, int skip_prior,
  int dim, float *image, int verbose
);
/*****************************************************************************/

/*****************************************************************************/
/* mrprior */
extern void do_prior(
  float *img, float beta, float *med_coef, int dim, float small, int maskdim,
  float *maxm
);
extern float med9(float *inp, int dim);
extern float med21(float *inp, int dim);
/*****************************************************************************/

/*****************************************************************************/
/* prmat */
enum {PRMAT_STATUS_UNINITIALIZED, PRMAT_STATUS_INITIALIZED,
  PRMAT_STATUS_BS_OCCUPIED, PRMAT_STATUS_LU_OCCUPIED, PRMAT_STATUS_PR_OCCUPIED,
  PRMAT_STATUS_ERROR
};
enum {PRMAT_TYPE_ECAT931, PRMAT_TYPE_GE, PRMAT_TYPE_NA};
enum {PRMAT_DMODE_01, PRMAT_DMODE_LOI, PRMAT_DMODE_EA, PRMAT_DMODE_NN};

/// If not 0 drive in test mode.
extern int PRMAT_TEST;
/// If not 0 drive in verbose mode.
extern int PRMAT_VERBOSE;

/** PRMAT is the structure for storing and processing
    the projection (system) matrix utilized in PET
    image reconstruction and Radon transform. */
typedef struct {
  /// Prmat status.
  char status;
  /// Scanner information on the prmat. 0=ECAT931 1=GE Advance.
  char type;
  /// Scanner geometrics, field viewNr.
  unsigned int viewNr;
  /// Scanner geometrics, field binNr.
  unsigned int binNr;
  /// Scanner geometrics, field imgDim.
  unsigned int imgDim;
  /// Discretisation model utilised.
  int mode; 
  /** FOV of the projection matrix.
      prmatfov[0]=major semiaxe, prmatfov[1]=minor semiaxe. */
  int *prmatfov; 
  /// Minimal factor value in the projection matrix.
  float min;
  /// Maximal factor value in the projection matrix.
  float max;
  /// The sum of all factors in the projection matrix
  float factor_sum;
  /// Scaling factor for factors (notice that factors are stored in integers).
  float scaling_factor;

  /* PRMAT LOOK-UP TABLE: for restoring lines of response intersecting a pixel.*/
  /// Number of pixels inside the fov. i.e. number of rows in the look-up table.
  unsigned int nrp;
  /** Coordinates of a pixel, number of lines of response intersecting the pixel 
     and the coordinates of coincidence lines (nrp x (lines + 2) array).
     lines[row][0] = pixel coordinates , lines[row][1] = lines hitting a pixel.
   */
  unsigned int **lines;
  /// Hidden pointer for the actual data.
  unsigned int **_linesdata;

  /* PRMAT DATA: for restoring lines of response in the projection matrix. */
  /// Dimension of rows (lines of response) in the projection matrix.
  unsigned int dimr;
  /// Square sums of factors in each row in the projection matrix.
  float *factor_sqr_sum;
  /// Number of pixels hit by a line for every line.
  unsigned int *dime;
  /** Coordinates and factors of pixels hit by a line of response.
      fact[row][pix][0]=x-coordinate && fact[row][pix][1]=y-coordinate && 
      fact[row][pix][2]=factor. */
  unsigned short int ***fact; 
  /// Hidden pointer for the actual data. 
  unsigned short int ***_factdata;
} PRMAT;

extern void prmatInit(PRMAT *mat);
extern void prmatEmpty(PRMAT *mat);
extern int prmatAllocate(PRMAT *mat, int set, unsigned int rows, unsigned int *coords);
/* GET procedures for PRMAT datatype. USE ONLY THESE FOR GETTING. */
/* GET FUNCTIONS FOR THE PARAMETERS */
extern unsigned int prmatGetNV(PRMAT *mat);
extern unsigned int prmatGetNB(PRMAT *mat);
extern unsigned int prmatGetID(PRMAT *mat);
/* GET FUNCTIONS FOR ACCESSING LOOK-UP TABLE */
extern unsigned int prmatGetPIX(PRMAT *mat);
extern unsigned int prmatGetPixCoord(PRMAT *mat,int row);
extern unsigned int prmatGetRays(PRMAT *mat,int row);
extern unsigned int prmatGetBinView(PRMAT *mat, int row, int ind);
/* GET FUNCTIONS FOR ACCESSING PROJECTION MATRIX BY A LINE OF RESPONSE.*/
extern unsigned int prmatGetRows(PRMAT *mat);
extern unsigned int prmatGetPixels(PRMAT *mat, int row);
extern unsigned int prmatGetCoord(PRMAT *mat, int row, int pix);
extern unsigned int prmatGetXCoord(PRMAT *mat, int row, int pix);
extern unsigned int prmatGetYCoord(PRMAT *mat, int row, int pix);
extern float prmatGetFactor(PRMAT *mat, int row, int pix);
extern float prmatGetMajor(PRMAT *mat);
extern float prmatGetMinor(PRMAT *mat);
extern float prmatGetMin(PRMAT *mat);
extern float prmatGetMax(PRMAT *mat);
extern float prmatGetFactorSum(PRMAT *mat);
extern float prmatGetFactorSqrSum(PRMAT *mat, int row);
/* ARCHIVING METHODS */
extern int prmatReadMatrix(char *fname, PRMAT *mat);
extern int prmatSaveMatrix(PRMAT *mat);
/*****************************************************************************/

/*****************************************************************************/
/* radon */
enum {RADON_STATUS_UNINITIALIZED, RADON_STATUS_INITIALIZED,
      RADON_STATUS_OCCUPIED, RADON_STATUS_ERROR
};
/// Drive in test mode if not 0.
extern int RADON_TEST;
/// Drive in verbose mode if not 0.
extern int RADON_VERBOSE;

/**  Radon data structure contains the parameters defining a Radon transform. */
typedef struct {
  /** Transform status. See definitions. */
  char status;
  /** Discretization model. */
  char mode;
  /** Dimension of the image plane. */
  int imgDim;
  /** Number of views (angles). */
  int viewNr;
  /** Number of bins (distances). */
  int binNr;
  /** Half of the bins. */
  int half;
  /** Bin going through the origin. */
  int centerBin;
  /** Sample distance. i.e. distance between two lines of response. */
  float sampleDist;
  /** Sine table. i.e. the tabulated values of the sine function for 
      this transform. */
  float *sines;
} RADON;

extern void radonEmpty(RADON *radtra);
extern int radonSet(RADON *radtra,int mode,int imgDim,int viewNr,int binNr);
/* Get functions for Radon data */
extern int radonGetMO(RADON *radtra);
extern int radonGetID(RADON *radtra);
extern int radonGetNV(RADON *radtra);
extern int radonGetNB(RADON *radtra);
extern float radonGetSD(RADON *radtra);
extern int radonGetHI(RADON *radtra);
extern int radonGetCB(RADON *radtra);
extern float radonGetSin(RADON *radtra, int nr);
/* THE ACTUAL TRANSFORM FUNCTIONS */
extern int radonFwdTransform(
  RADON *radtra, int set, int setNr, float *imgdata, float *scndata
);
extern int radonFwdTransformEA(
  RADON *radtra, int set, int setNr, float *imgdata, float *scndata
);
extern int radonFwdTransformSA(
  RADON *radtra, int set, int setNr, float *imgdata, float *scndata
);
extern int radonFwdTransformPRM(
  PRMAT *mat, int set, int setNr, float *imgdata, float *scndata
);
extern int radonBackTransform(
  RADON *radtra, int set, int setNr, float *scndata, float *imgdata
);
extern int radonBackTransformEA(
  RADON *radtra, int set, int setNr, float *scndata, float *imgdata
);
extern int radonBackTransformSA(
  RADON *radtra, int set, int setNr, float *imgdata, float *scndata
);
extern int radonBackTransformPRM(
  PRMAT *mat, int set, int setNr, float *scndata, float *imgdata
);
/* FUNCTIONS FOR SETTING THE FACTORS IN THE PRMAT DATA STUCTURE */
extern int radonSetBases(RADON *radtra,ELLIPSE *elli, PRMAT *mat);
extern int radonSetBasesEA(RADON *radtra,ELLIPSE *elli, PRMAT *mat);
extern int radonSetLUT(RADON *radtra, ELLIPSE *elli, PRMAT *mat);
extern int radonSetLORS(RADON *radtra,ELLIPSE *elli, PRMAT *mat);
/*****************************************************************************/

/*****************************************************************************/
/* recutil */
extern void recSinTables(
  int views, float *sinB, float *sinBrot, float rotation
);
extern void recInterpolateSinogram(
  float *srcsino, float *newsino, int srcrays, int newrays, int views
);
extern int bit_rev_int(int x, int n);
extern void set_os_set(int os_sets, int *set_seq);
extern int recGetStatistics(
  float *buf, int n, float *osum, float *omin, float *omax, int skip_zero_mins
);
/*****************************************************************************/

/*****************************************************************************/
/* reprojection */
extern int imgReprojection(IMG *img, IMG *scn, int verbose);
extern int reprojection(
  float *image, int dim, int rays, int views, float bpzoom, float *sinogram,
  int verbose
);
extern void viewReprojection(
  float *idata, float *sdata, int view, int dim, 
  int viewNr, int rayNr, float *sinB, float *sinBrot, 
  float offsX, float offsY, float bpZoom
);
extern void reprojectionAvg3(float *data, int n);
extern void reprojectionAvg5(float *data, int n);
extern void reprojectionMed3(float *data, int n);
extern void viewBackprojection(
  float *prj, float *idata, int dim, int view, int viewNr, int rayNr,
  float *sinB, float *sinBrot, float offsX, float offsY, float bpZoom
);
/*****************************************************************************/

/*****************************************************************************/
extern int trmrp(
  float *bla, float *tra, int dim, float *image,
  int iter, int sets, int rays, int views,
  int maskdim, float zoom, float beta, float axial_fov,
  float sample_distance, int skip_prior, int osl,
  float shiftX, float shiftY, float rotation,
  int verbose
);
/*****************************************************************************/

/*****************************************************************************/
#endif // _LIBTPCREC_H
