/** @file maskconj.c
 *  @brief Conjunction (AND) of two mask images.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Conjunction (AND) of two mask images in ECAT 6.3 or 7.x, NIfTI-1, or",
  "Analyze 7.5 format.",
  "Pixels with non-zero in both masks is set to 1, otherwise to 0.",
  " ",
  "Usage: @P [Options] mask1 mask2 andmask",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: maskinv, maskdila, maskeros, imgmask, imgthrs, img2dft",
  " ",
  "Keywords: image, mask, conjunction",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  char    maskfile1[FILENAME_MAX], maskfile2[FILENAME_MAX], outfile[FILENAME_MAX];
  char   *cptr=NULL;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile1[0]=maskfile2[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(maskfile1, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(maskfile2, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("maskfile1 := %s\n", maskfile1);
    printf("maskfile2 := %s\n", maskfile2);
    printf("outfile := %s\n", outfile);
    fflush(stdout);
  }


  /*
   *  Read mask files
   */
  IMG mask1; imgInit(&mask1);
  IMG mask2; imgInit(&mask2);

  if(verbose>0) {printf("reading %s\n", maskfile1); fflush(stdout);}
  ret=imgRead(maskfile1, &mask1);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask1.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(verbose>0) {printf("reading %s\n", maskfile2); fflush(stdout);}
  ret=imgRead(maskfile2, &mask2);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask2.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    imgEmpty(&mask1); return(3);
  }
  if(mask1.dimt>1 || mask2.dimt>1) {
    fprintf(stderr, "Error: mask cannot be dynamic image.\n");
    imgEmpty(&mask1); imgEmpty(&mask2); return(2);
  }
  /* Check that dimensions are compatible */
  if(mask1.dimx!=mask2.dimx || mask1.dimy!=mask2.dimy ||
     mask1.dimz!=mask2.dimz)
  {
    fprintf(stderr, "Error: different mask image dimensions.\n");
    imgEmpty(&mask1); imgEmpty(&mask2); return(4);
  }


  /*
   *  Mask conjunction
   */
  if(verbose>0) {printf("mask conjunction\n"); fflush(stdout);}
  ret=imgMaskConjunction(&mask1, &mask2);
  imgEmpty(&mask2);
  if(ret) {
    fprintf(stderr, "Error: cannot make conjunction of mask images\n");
    if(verbose>1) printf("ret := %d\n", ret);
    imgEmpty(&mask1); return(5);
  }
  /* Check whether mask has any voxels left */
  unsigned int mn;
  mn=imgMaskCount(&mask1);
  if(mn==0) {
    fprintf(stderr, "Warning: all voxels are zeroes; empty mask not saved.\n");
    imgEmpty(&mask1);
    return(0);
  }
  if(verbose>1) {
    printf("nr_of_positive_voxels := %u\n", mn);
  }


  /*
   *  Save the modified mask
   */
  if(verbose>2) printf("writing mask\n");
  ret=imgWrite(outfile, &mask1);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask1.statmsg);
    imgEmpty(&mask1); return(11);
  }
  imgEmpty(&mask1);
  if(verbose>0) printf("mask written.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
