/** @file masksize.c
 *  @brief Report the size of regions in mask image and optionally sort by size.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Report the number of pixels belonging to labelled regions in mask image",
  "in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  " ",
  "Usage: @P [Options] maskfile",
  " ",
  "Options:",
  " -sort",
  "     Region labels are modified so that label 1 will represent the largest,",
  "     label 2 the second largets, and so on.",
  "     Notice that this option will overwrite the mask file.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: masklbl, imgthrs, maskconj, imgmask, mask2pxl",
  " ",
  "Keywords: image, mask, ROI",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret, n;
  char    maskfile[FILENAME_MAX];
  int     sortLabels=0; // 0=no sorting, 1=sort
  char   *cptr=NULL;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strcasecmp(cptr, "SORT")==0) {
      sortLabels=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("maskfile := %s\n", maskfile);
    printf("sortLabels := %d\n", sortLabels);
    fflush(stdout);
  }


  /*
   *  Read mask
   */
  IMG mask; imgInit(&mask);

  if(verbose>0) {printf("reading %s\n", maskfile); fflush(stdout);}
  ret=imgRead(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(mask.dimt>1) {
    fprintf(stderr, "Error: mask cannot be dynamic image.\n");
    imgEmpty(&mask); return(2);
  }
  if(verbose>0 || sortLabels==0) {
    printf("xdim := %d\n", mask.dimx);
    printf("ydim := %d\n", mask.dimy);
    printf("zdim := %d\n", mask.dimz);
    printf("pxlNr := %d\n", mask.dimx*mask.dimy*mask.dimz);
  }

  /*
   *  Get the region labels and count the number of pixels belonging to each of them
   */
  int zi, yi, xi;
  /* Search the maximum pixel value */
  float mv=0.0;
  for(zi=0; zi<mask.dimz; zi++)
    for(yi=0; yi<mask.dimy; yi++)
      for(xi=0; xi<mask.dimx; xi++)
        if(mask.m[zi][yi][xi][0]>mv)
          mv=mask.m[zi][yi][xi][0];
  if(verbose>1) printf("maxval := %g\n", mv);
  /* Based on that, set the max number of labels */
  int maxLabel=(int)roundf(mv);
  if(verbose>1) printf("maxlabel := %d\n", maxLabel);
  if(maxLabel==0) {
    fprintf(stderr, "Error: mask file does not contain any labelled pixels.\n");
    imgEmpty(&mask); return(3);
  }
  if(maxLabel>10000) {
    fprintf(stderr, "Error: mask file contains too many labels.\n");
    imgEmpty(&mask); return(3);
  }
  /* Allocate a list for pixel number having a certain label */
  int pxlNr[maxLabel+1];
  for(int i=0; i<=maxLabel; i++) pxlNr[i]=0;
  /* Count the pixels with each label */
  for(zi=0; zi<mask.dimz; zi++)
    for(yi=0; yi<mask.dimy; yi++)
      for(xi=0; xi<mask.dimx; xi++) {
        n=(int)roundf(mask.m[zi][yi][xi][0]);
        if(n<0) n=0;
        pxlNr[n]++;
      }
  /* List the pixel numbers, if >0 */
  printf("Number of pixels per label:\n");
  int bl=0, ms=0;
  n=0;
  for(int i=0; i<=maxLabel; i++) {
    if(pxlNr[i]>0) printf("label_%d := %d\n", i, pxlNr[i]);
    if(i==0) continue;
    n+=pxlNr[i];
    if(pxlNr[i]>ms) {ms=pxlNr[i]; bl=i;}
  }
  printf("totalNr := %d\n", n);
  printf("largestRegion := %d\n", bl);
  printf("largestSize := %d\n", ms);

  /* 
   *  If user did not want sorting, then we are ready
   */
  if(sortLabels==0) {imgEmpty(&mask); return(0);}


  /*
   *  Determine new labels, based on region size
   */
  int newLabel[maxLabel+1];
  for(int i=0; i<=maxLabel; i++) newLabel[i]=0;
  /* Largest region will get label 1 */
  int currLabel=1;
  newLabel[bl]=currLabel; pxlNr[bl]=0;
  while(1) {
    /* Find the next highest */
    bl=ms=0; currLabel++;
    for(int i=1; i<=maxLabel; i++) {
      if(pxlNr[i]>ms) {ms=pxlNr[i]; bl=i;}
    }
    if(verbose>10) printf("currLabel=%d ms=%d bl=%d\n", currLabel, ms, bl);
    if(ms==0) break;
    newLabel[bl]=currLabel; pxlNr[bl]=0;
  }
  if(verbose>2) {
    printf("relabelling:\n");
    for(int i=1; i<=maxLabel; i++) printf(" %d -> %d\n", i, newLabel[i]);
  }

  /*
   *  Relabel each mask pixel
   */
  for(zi=0; zi<mask.dimz; zi++)
    for(yi=0; yi<mask.dimy; yi++)
      for(xi=0; xi<mask.dimx; xi++) {
        n=(int)roundf(mask.m[zi][yi][xi][0]);
        if(n<0) n=0;
        mask.m[zi][yi][xi][0]=(float)newLabel[n];
        pxlNr[n]++;
      }


  /*
   *  Save the modified mask
   */
  if(verbose>2) printf("writing sorted mask...\n");
  ret=imgWrite(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    imgEmpty(&mask); return(11);
  }
  imgEmpty(&mask);
  if(verbose>0) printf("sorted mask written.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
