/** @file nii_ehdr.c
 *  @brief Edit the information in NIfTI-1 header.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/
#include "niftiift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Edits the information in NIfTI-1 image header.",
  "New contents are given in an ASCII text file with one or more header fields",
  "in the same format as is the output of nii_lhdr (IFT format).",
  "Use this program only if you are familiar with NIfTI format; note that",
  "no backup is created.",
  "Note that frame times are not stored in NIfTI header but in SIF.",
  " ",
  "Usage: @P niftifile headerfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example: copy header contents from image1.nii to image2.nii",
  "     nii_lhdr image1.nii > header.dat",
  "     @P image2.nii header.dat",
  " ",
  "See also: nii_lhdr, iftlist, iftdel, iftadd, ana_lhdr",
  " ",
  "Keywords: image, NIfTI, header, IFT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       ret;
  char      hdrfile[FILENAME_MAX], iftfile[FILENAME_MAX], tmp[256];
  NIFTI_DSR dsr;
  IFT       ift1, ift2;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  hdrfile[0]=iftfile[0]=(char)0;
  iftInit(&ift1); iftInit(&ift2);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!hdrfile[0]) {strlcpy(hdrfile, argv[ai], FILENAME_MAX); continue;}
    else if(!iftfile[0]) {strlcpy(iftfile, argv[ai], FILENAME_MAX); continue;}
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!iftfile[0]) {
    fprintf(stderr, "Error: missing command-line argument.\n"); 
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("hdrfile := %s\n", hdrfile);
    printf("iftfile := %s\n", iftfile);
  }


  /*
   *  Read original header file
   */
  if(verbose>1) printf("reading %s\n", hdrfile);
  ret=niftiReadHeader(hdrfile, &dsr, verbose-2, tmp);
  if(ret) {
    fprintf(stderr, "Error: cannot read %s: %s.\n", hdrfile, tmp);
    return(2);
  }
  /* Copy header contents to IFT struct */
  ret=niftiHeaderToIFT(&dsr, &ift1, verbose-4);
  if(ret!=STATUS_OK) {
    fprintf(stderr, "Error: invalid header structure (%d).\n", ret);
    return(5);
  }
  if(verbose>3) {
    fprintf(stdout, "\nOriginal header:\n\n");
    iftWrite(&ift1, "stdout", 0);
    fprintf(stdout, "\n");
  }


  /*
   *  Read new header contents
   */
  if(verbose>1) printf("reading %s\n", iftfile);
  ret=iftRead(&ift2, iftfile, 1, 0);
  if(ret) {
    fprintf(stderr, "Error in reading %s: %s.\n", iftfile, ift2.status);
    iftEmpty(&ift1);
    return(6);
  }
  if(verbose>4) {
    fprintf(stdout, "\nNew header contents:\n\n");
    iftWrite(&ift2, "stdout", 0);
    fprintf(stdout, "\n");
  }


  /*
   *  Copy new contents into existing header struct
   */
  ret=niftiHeaderFromIFT(&dsr, &ift2, verbose-4);
  if(ret!=STATUS_OK) {
    fprintf(stderr, "Error: invalid header contents in %s.\n", iftfile);
    iftEmpty(&ift1); iftEmpty(&ift2);
    return(7);
  }
  /* Copy edited header contents to IFT struct */
  iftEmpty(&ift2);
  ret=niftiHeaderToIFT(&dsr, &ift2, verbose-4);
  if(ret!=STATUS_OK) {
    fprintf(stderr, "Error: invalid header structure (%d).\n", ret);
    return(5);
  }
  if(verbose>3) {
    fprintf(stdout, "\nNew header contents:\n\n");
    iftWrite(&ift2, "stdout", 0);
    fprintf(stdout, "\n");
  }


  /*
   *  Write edited header
   */
  if(verbose>1) printf("writing %s\n", hdrfile);
  ret=niftiWriteHeader(hdrfile, &dsr, verbose-2, tmp);
  if(ret!=STATUS_OK) {
    fprintf(stderr, "Error: cannot write file: %s.\n", tmp);
    return(11);
  }

  iftEmpty(&ift1);
  iftEmpty(&ift2);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
