/** @file res2html.c
 *  @brief Convert result files to HTML table format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/
//#define MAX_LINE_LEN 2048
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts result file to (X)HTML table format for viewing in web browsers",
  "or importing into Excel or other spreadsheet software.",
  " ",
  "Usage: @P [Options] <Result file> [HTML filename]",
  " ",
  "If HTML filename is omitted, then it is named automatically by replacing",
  "original extension with .htm.",
  " ",
  "Options:",
  " -n[osd]",
  "     SDs and CVs are not saved into HTML file.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P ug77654ki.res ug77654ki.htm",
  " ",
  "See also: reslist, rescoll, pardiff, parformat",
  " ",
  "Keywords: results, HTML, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   saveSD=1;
  RES   res;
  char *cptr, resfile[FILENAME_MAX], htmfile[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  resfile[0]=htmfile[0]=(char)0;
  resInit(&res);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!cptr[0]) continue;
    if(strncasecmp(cptr, "NOSD", 1)==0) {
      saveSD=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!resfile[0]) {strlcpy(resfile, argv[ai], FILENAME_MAX); continue;}
    else if(!htmfile[0]) {strlcpy(htmfile, argv[ai], FILENAME_MAX); continue;}
  }

  /* Is something missing? */
  if(!resfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!htmfile[0]) {
    strcpy(htmfile, resfile);
    cptr=strrchr(htmfile, '.'); if(cptr!=NULL) *cptr=(char)0;
    strcat(htmfile, ".htm");
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("resfile := %s\n", resfile);
    printf("htmfile := %s\n", htmfile);
    printf("saveSD := %d\n", saveSD);
  }
  if(verbose>4) RESULT_TEST=verbose-4; else RESULT_TEST=0;


  /*
   *  Read result file
   */
  if(verbose>1) printf("reading %s\n", resfile);
  if(resRead(resfile, &res, verbose-2)) {
    fprintf(stderr, "Error in reading '%s': %s\n", resfile, reserrmsg);
    resEmpty(&res);
    return(2);
  }


  /*
   *  Remove SDs and CVs if required
   */
  if(saveSD==0) {
    for(int ri=0; ri<res.voiNr; ri++) for(int pi=0; pi<res.parNr; pi++)
      res.voi[ri].sd[pi]=res.voi[ri].cl1[pi]=res.voi[ri].cl2[pi]=nan("");
  }


  /*
   *  Write results in HTML format
   */
  if(verbose>1) printf("writing %s\n", htmfile);
  if(resWriteHTML(&res, htmfile, verbose-1)) {
    fprintf(stderr, "Error in writing '%s': %s\n", resfile, reserrmsg);
    resEmpty(&res);
    return(11);
  }
  resEmpty(&res);
  if(verbose==1) printf("%s written.\n", htmfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
