/** @file resmatch.c
 *  @brief Verify that two result files have matching contents.
           Used in automatic software and analysis tool testing.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/******************************************************************************/
double ROUGH_TEST_LIMIT=0.01;
double AROUND_TEST_LIMIT=0.05;
/******************************************************************************/

/******************************************************************************/
static char *info[] = {
  "Verify that two result files do have matching contents;",
  "this can be used for automating software and analysis tool testing.",
  "Programs return code is 0, if files were matching, 10, if the files did",
  "not match, and 1-9 in case of error.",
  " ",
  "Usage: @P [Options] file1 file2",
  " ",
  "Options:",
  " -header=<y|N>",
  "     Header fields are verified (y) or not verified (n, default)",
  " -param[eters]=<y|N>",
  "     Parameter names are verified (y) or not verified (n, default)",
  " -regions=<y|N>",
  "     Region names are verified (y) or not verified (n, default)",
  " -res[ults]=<List of result column numbers (e.g. -res=1-2 or -res=1,3)>",
  "     By default all result values are verified. With this option,",
  "     only selected columns (or none with -res=0) are verified.",
  " -roughly",
  "     Result values are required to match roughly (<2% difference allowed).",
  " -around",
  "     Results are required to be around the same (<10% difference allowed).",
  " -abs=<value>",
  "     Absolute result differences must not exceed the specified limit value",
  " -errors=<Y|n>",
  "     Result errors (SD, CL) are verified (y, default) or not verified (n)",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: reslist, res2html, resdel, pardiff, fit2res, tacmatch, imgmatch",
  " ",
  "Keywords: results, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int pi, ri, n, test_nr=0, ret, ok=0;
  char *cptr, resfile1[FILENAME_MAX], resfile2[FILENAME_MAX];
  RES res1, res2;
  INT_list parlist;
  /* Optional tests */
  int header=0;
  int parameters=0;
  int regions=0;
  int rough_test=0;
  int test_errors=1;
  double test_limit=1.0E-15;
  double test_abs=-1.0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  resfile1[0]=resfile2[0]=(char)0;
  resInit(&res1); resInit(&res2);
  parlist.nr=0;
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* option */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "HEADER=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0) {header=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {header=0; continue;}
    } else if(strncasecmp(cptr, "PARAMETERS=", 11)==0) {
      cptr+=11;
      if(strncasecmp(cptr, "YES", 1)==0) {parameters=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {parameters=0; continue;}
    } else if(strncasecmp(cptr, "PARAM=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "YES", 1)==0) {parameters=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {parameters=0; continue;}
    } else if(strncasecmp(cptr, "REGIONS=", 8)==0) {
      cptr+=8;
      if(strncasecmp(cptr, "YES", 1)==0) {regions=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {regions=0; continue;}
    } else if(strncasecmp(cptr, "ROUGHLY", 5)==0) {
      rough_test=1; test_limit=ROUGH_TEST_LIMIT;
      if(test_abs<0.0) continue;
    } else if(strncasecmp(cptr, "AROUND", 5)==0) {
      rough_test=2; test_limit=AROUND_TEST_LIMIT; continue;
    } else if(strncasecmp(cptr, "RESULTS=", 8)==0) {
      if(parlist.nr==0) {
        ret=intExpand(cptr+8, &parlist);
        if(ret==0 && parlist.nr>0 && parlist.i[0]>=0) continue;
      }
    } else if(strncasecmp(cptr, "RES=", 4)==0) {
      if(parlist.nr==0) {
        ret=intExpand(cptr+4, &parlist);
        if(ret==0 && parlist.nr>0 && parlist.i[0]>=0) continue;
      }
    } else if(strncasecmp(cptr, "ABS=", 4)==0) {
      cptr+=4; test_abs=atof_dpi(cptr);
      if(test_abs>=0.0 && rough_test==0) continue;
    } else if(strncasecmp(cptr, "ERRORS=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0) {test_errors=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {test_errors=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    if(parlist.nr>0) free(parlist.i);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!resfile1[0]) {
      strlcpy(resfile1, argv[ai], FILENAME_MAX); continue;
    } else if(!resfile2[0]) {
      strlcpy(resfile2, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    if(parlist.nr>0) free(parlist.i);
    return(1);
  }

  /* Is something missing? */
  if(!resfile2[0]) {
    fprintf(stderr, "Error: missing result file name.\n");
    if(parlist.nr>0) free(parlist.i);
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>0) {
    printf("resfile1 := %s\n", resfile1);
    printf("resfile2 := %s\n", resfile2);
    printf("header := %d\n", header);
    printf("parameters := %d\n", parameters);
    printf("regions := %d\n", regions);
    printf("rough_test := %d\n", rough_test);
    printf("test_limit := %g\n", test_limit);
    printf("test_abs := %g\n", test_abs);
    printf("test_errors := %d\n", test_errors);
    printf("results :=");
    for(pi=0; pi<parlist.nr; pi++) printf(" %d", parlist.i[pi]);
    printf("\n");
  }
  if(verbose>10) RESULT_TEST=verbose-10;


  /*
   *  Read result files
   */
  if(verbose>0) printf("reading results\n");
  if(resRead(resfile1, &res1, verbose-2)) {
    fprintf(stderr, "Error in reading '%s': %s\n", resfile1, reserrmsg);
    resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
    return(2);
  }
  if(resRead(resfile2, &res2, verbose-2)) {
    fprintf(stderr, "Error in reading '%s': %s\n", resfile2, reserrmsg);
    resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
    return(2);
  }
  /* Check that selected columns exist in results */
  if(parlist.nr>0 && parlist.i[0]>0) {
    for(pi=0; pi<parlist.nr; pi++) if(parlist.i[pi]>res1.parNr) {
      fprintf(stderr, "Error: results do not contain required column %d\n",
        parlist.i[pi]);
      resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
      return(1);
    }
  }
  /* Report the file names */
  if(verbose>=0) {
    printf("file1 := %s\n", resfile1);
    printf("file2 := %s\n", resfile2);
  }

  /*
   *  Test the number of regions
   */
  if(verbose>0) printf("testing number of regions\n");
  if(res1.voiNr!=res2.voiNr) {
    if(verbose>=0) fprintf(stdout, "result := no match\n");
    resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
    return(10);
  }
  test_nr++;
  
  /*
   *  Test the number of parameters
   */
  if(verbose>0) printf("testing number of parameters\n");
  if(res1.parNr!=res2.parNr) {
    /* That is not a problem if the last parameters are not to be compared */
    if(parlist.nr<=0) ok=0;
    else if(parlist.nr==1 && parlist.i[0]==0) ok=1;
    else for(pi=0, ok=1; pi<parlist.nr; pi++)
      if(parlist.i[pi]>res1.parNr || parlist.i[pi]>res2.parNr) {ok=0; break;}
    if(ok!=1) {
      if(verbose>=0) fprintf(stdout, "result := no match\n");
      resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i);
      return(10);
    }
  }
  test_nr++;

  /*
   *  Check header fields
   */
  if(header) {
    if(verbose>0) printf("testing header\n");
    if((ret=resMatchHeader(&res1, &res2))!=0) {
      if(verbose>=0) 
        fprintf(stdout, "result := no match in header field (%d)\n", ret);
      if(ret!=5) { // other missmatches than studynr are considered important
        resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
        return(10);
      }
    }
    test_nr++;
  }

  /*
   *  Check region names
   */
  if(regions) {
    if(verbose>0) printf("testing regions\n");
    if(resMatchRegions(&res1, &res2)) {
      if(verbose>=0) fprintf(stdout, "result := no match in regions\n");
      resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
      return(10);
    }
    test_nr++;
  }

  /*
   *  Check result parameter names
   */
  if(parameters) {
    if(verbose>0) printf("testing regions\n");
    if(resMatchParameternames(&res1, &res2)) {
      if(verbose>=0) fprintf(stdout, "result := no match in parameter names\n");
      resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
      return(10);
    }
    test_nr++;
  }

  /*
   *  Check result parameter values
   */
  if(parlist.nr==1 && parlist.i[0]==0) {
    if(verbose>0) printf("parameter values are NOT tested\n");
  } else {
    if(verbose>0) printf("testing parameter values\n");
    for(ri=0, n=0; ri<res1.parNr; ri++) {
      if(parlist.nr>0) { /* should this column be verified? */
        for(pi=ok=0; pi<parlist.nr; pi++) if(ri==parlist.i[pi]-1) {ok=1; break;}
        if(ok==0) continue;
      }
      n++;
      if(verbose>2) printf("  test parameter values for col %d\n", ri+1);
      if(test_abs<0.0)
        ret=resMatchParameters(&res1, &res2, ri, test_limit, test_errors);
      else
        ret=resMatchParametersAbs(&res1, &res2, ri, test_abs, test_errors);
      if(ret!=0) {
        if(verbose>1) printf("  ret:=%d\n", ret);
        if(verbose>=0) fprintf(stdout, "result := no match in parameter values (column %d)\n", ri+1);
        if(verbose>2) {
          for(int i=0; i<res1.voiNr; i++)
            printf("  %g vs %g\n", res1.voi[i].parameter[ri], res2.voi[i].parameter[ri]);
        }
        resEmpty(&res1); resEmpty(&res2); if(parlist.nr>0) free(parlist.i); 
        return(10);
      }
    }
    if(n>0) test_nr++;
  }

  /* Report the success */
  if(verbose>=0) {
    fprintf(stdout, "result :=  match was found\n");
    fprintf(stdout, "nr_of_tests :=  %d\n", test_nr);
  }

  resEmpty(&res1); resEmpty(&res2);
  if(parlist.nr>0) free(parlist.i);
  return(0);
}
/******************************************************************************/

/******************************************************************************/
/// @endcond

