/** @file ainterp.c
 *  @brief Interpolates TACs to an optimal time scale for simulations.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Interpolation of TAC to an optimal time scale for simulations.",
  " ",
  "Usage: @P [options] tacfile endtime newfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Endtime is the length of the inter/extrapolated data. To interpolate to",
  "the end of input data, set it to 0.",
  " ",
  "See also: interpol, simframe, tactime, tacframe, avgbolus, dftavg, fit2dat",
  " ",
  "Keywords: TAC, simulation, interpolation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        ret;
  char       datfile[FILENAME_MAX], outfile[FILENAME_MAX];
  DFT        dft, dft2;
  double     endtime=-1.0;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dftInit(&dft); dftInit(&dft2);
  datfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!datfile[0]) {
      strcpy(datfile, argv[ai]); continue;
    } else if(endtime<0.0) {
      endtime=atof_dpi(argv[ai]); if(endtime<0.0) endtime=0.0; continue;
    } else if(!outfile[0]) {
      strcpy(outfile, argv[ai]); continue;
    } else {
      fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
      return(1);
    }
  }  

  /* Is something missing? */
  if(!outfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("datfile := %s\n", datfile);
    printf("outfile := %s\n", outfile);
    printf("endtime := %g\n", endtime);
  }

  /*
   *  Read TAC data
   */
  if(verbose>1) printf("reading %s\n", datfile);
  if(dftRead(datfile, &dft)) {
    fprintf(stderr, "Error in reading '%s': %s\n", datfile, dfterrmsg);
    dftEmpty(&dft);
    return(2);
  }


  /*
   *  Interpolate
   */
  if(verbose>1) printf("interpolating\n");
  /* set the endtime if necessary */
  if(endtime<=0.0) {
    if(dft.timetype==DFT_TIME_STARTEND) endtime=dft.x2[dft.frameNr-1];
    else endtime=dft.x[dft.frameNr-1];
  }
  /* interpolate */
  ret=dftAutointerpolate(&dft, &dft2, endtime, verbose-2);
  if(ret) {
    fprintf(stderr, "Error %d in interpolation.\n", ret);
    dftEmpty(&dft); dftEmpty(&dft2); return(3);
  }
  dft2.timetype=dft.timetype;
  dftEmpty(&dft);
  

  /*
   *  Save TACs
   */
  if(verbose>1) printf("saving %s\n", outfile);
  /* Write DFT file */
  ret=dftWrite(&dft2, outfile); dftEmpty(&dft2);
  if(ret) {
    fprintf(stderr, "Error in writing '%s': %s\n", outfile, dfterrmsg);
    return(11);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
