#!/bin/bash
#: Title      : test_heartcor
#: Date       : 2019-08-26
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../heartcor$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f observ.dft ] || [ ! -f correct_corrected.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f observ0.dft ] || [ ! -f correct_corrected0.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f true.dft ] || [ ! -f correct_simulated.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f oldpatlak.dft ] || [ ! -f correct_oldpatlak_newform.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f f18-fdg_arteriaplasma.fit ] || [ ! -f heart.fra ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f errorfree.dft ] || [ ! -f blood.dat ] || [ ! -f tissue.dat ]; then
  printf "\n Constructing test data\n"
  fit2dat -A=60 f18-fdg_arteriaplasma.fit blood.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren blood.dat 1 cav
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub --silent blood.dat 0.31 1.28 0.24 0 0 0 myo.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren myo.dat 1 myo
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub --silent blood.dat 0.20 0.80 0.30 0 0 0 reg2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren reg2.dat 1 reg2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub --silent blood.dat 0.40 1.60 0.10 0 0 0 reg3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren reg3.dat 1 reg3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr --silent tissue.dat myo.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent tissue.dat blood.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent tissue.dat reg2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent tissue.dat reg3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy tissue.dat fdg100
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tissue.dat heart.fra errorfree.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg errorfree.svg errorfree.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -stupidoption errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM errorfree.dft cav myo 52 10 10 output.dft extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.1.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM  cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM errorfree.dft  myo 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM errorfree.dft cav  52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM errorfree.dft cav myo  10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM errorfree.dft cav myo 52 10  output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM errorfree.dft cav myo 52 10 10  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM 52 10 10 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n 0.2.0 \n"
printf " Test case: Correct input, no options. \n"
printf " Expected result 1: Values of variables are correct. \n"
printf " Expected result 2: Output file is created. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt dfile "errorfree.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt rfile "output.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt cav_roi "cav"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt myo_roi "myo"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt R "26"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt d "10"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 stdout.txt s "4.246609"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt simulation "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0.1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

if [ ! -f output.dft ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 0.2.1 \n"
printf " Test case: Option -simulate. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f stdout.txt
$PROGRAM -d2 -simulation errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt simulation "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM -d2 -sim errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt simulation "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.2 \n"
printf " Test case: Option -Vb with valid and nonvalid fraction. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 -Vb=0.11 errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0.11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 -Vb=0,11 errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0.11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 -Vb=0.0 errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 -Vb=1.0 errorfree.dft cav myo 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-Vb=1.0'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-------------------------------------------------------------------\n"

printf "\n 0.3.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM nonexisting.dat 2 1 52 10 7 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.3.1 \n"
printf " Test case: Wrong input file format. \n"
printf " Expected result: Error. \n\n"

printf "abc defg\nbubudum\nmeh\n" > bad.dat
rm -f stdout.txt output.dat
$PROGRAM bad.dat 2 1 52 10 7 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-------------------------------------------------------------------\n"

printf "\n 0.4.0 \n"
printf " Test case: ROIs given with numbers that exist. \n"
printf " Expected result: Correct TACs selected. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft 2 1 52 10 10 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt cav_roi "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt myo_roi "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt cavity "cav"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt myocardium "myo"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.4.1 \n"
printf " Test case: ROIs given with numbers that do not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft 0 1 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "ROI '0' not found in errorfree.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft 2 8 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "ROI '8' not found in errorfree.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.4.2 \n"
printf " Test case: ROIs given as names that do not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft vac myo 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "ROI 'vac' not found in errorfree.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft cav moy 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "ROI 'moy' not found in errorfree.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.4.3 \n"
printf " Test case: Same ROI given as cavity and myocardium. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft myo myo 52 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "same ROI selected for cavity and myocardium."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n 0.5.0 \n"
printf " Test case: Invalid diameter given. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft 2 1 0 10 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid diameter '0'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.5.1 \n"
printf " Test case: Invalid thickness given. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft 2 1 52 0 10 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid thickness '0'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.5.2 \n"
printf " Test case: Invalid resolution given. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 errorfree.dft 2 1 52 10 0 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid resolution '0'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Simulated data \n"
printf "===================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Spillover errors are simulated. \n"
printf " Test case 2: Spillover errors are corrected again. \n"
printf " Expected result: Correct output is written. \n\n"

rm -f stdout.txt output.dft output2.dft
$PROGRAM --silent -simulate errorfree.dft cav myo 52 10 10 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg simulated_errors.svg -s errorfree.dft -l output.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.1 errorfree.dft output.dft
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM --silent output.dft cav myo 52 10 10 output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 errorfree.dft output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.1.0 \n"
printf " Test case: Correction with different relative concentration levels. \n"
printf " Expected result: Correct output is written. \n\n"

rm -f stdout.txt output.dft
$PROGRAM observ.dft 2 1 52 10 7 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_corrected.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.1.1 \n"
printf " Test case: Different relative concentration levels, Vb=0. \n"
printf " Expected result: Correct output is written. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -Vb=0 observ0.dft 2 1 52 10 7 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_corrected0.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.0 \n"
printf " Test case: Simulation with different relative concentration levels. \n"
printf " Expected result: Correct output is written. \n\n"

rm -f output.dft
$PROGRAM -simulate true.dft 2 1 52 10 7 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_simulated.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.0 \n"
printf " Test case: Option -Vb not given, using default Vb. \n"
printf " Expected result: Correct output is written. \n\n"

rm -f stdout.txt output.dft
$PROGRAM -d2 observ.dft 2 1 52 10 7 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0.1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.01 -rel=0.1 correct_corrected.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Just compute the correction parameters. \n"
printf " Expected result: Correct output is written in stdout. \n\n"

rm -f stdout.txt
$PROGRAM none 2 1 52 10 7 none &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 stdout.txt FMB 0.000000
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.0001 stdout.txt FMM 0.907434
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.0001 stdout.txt FBB 1.000000
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.0001 stdout.txt FBM 0.146283
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.0.0 \n"
printf " Test case: Current Patlak and data pre-correction. \n"
printf " Expected result: The same as with old Patlak with in-built correction. \n\n"

rm -f output.dft output.res
$PROGRAM oldpatlak.dft cav myo 52 10 10 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
patlak -sd=n output.dft blood.dat 10 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resdel output.res cav
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -roughly correct_oldpatlak_newform.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

