/** @file fit2res.c
 *  @brief Convert FIT files into result file format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts FIT files (1) to RES files (2).", 
  " ",
  "Usage: @P [Options] fitfile(s)",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "References:",
  "1. http://www.turkupetcentre.net/petanalysis/format_tpc_fit.html",
  "2. http://www.turkupetcentre.net/petanalysis/format_tpc_res.html",
  " ",
  "See also: fit2dat, rescoll, resdiff, res2html, reslist, resmatch, parformat",
  " ",
  "Keywords: FIT, RES, simulation, tool, results",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret, fileNr=0, ffi=0;
  char   fitfile[FILENAME_MAX], resfile[FILENAME_MAX];
  char  *cptr, tmp[FILENAME_MAX];
  RES    res;
  FIT    fit;



  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  fitfile[0]=resfile[0]=(char)0;
  fitInit(&fit); resInit(&res);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", fileNr);
  }
  if(verbose>3) RESULT_TEST=verbose-3; else RESULT_TEST=0;
  if(verbose>3) MATHFUNC_TEST=verbose-3; else MATHFUNC_TEST=0;



  /*
   *  Convert one file at a time
   */
  fileNr=0;
  for(ai=ffi; ai<argc; ai++) {

    strlcpy(fitfile, argv[ai], FILENAME_MAX);
    if(verbose>0 && fileNr>1) fprintf(stdout, "%s :\n", fitfile);

    /*
     *  Construct result filename
     */
    strlcpy(resfile, fitfile, FILENAME_MAX);
    cptr=strrchr(resfile, '.'); if(cptr!=NULL) *cptr=(char)0;
    strlcat(resfile, ".res", FILENAME_MAX);
    if(verbose>2) printf("converting %s -> %s\n", fitfile, resfile);

    /*
     *  Read FIT data
     */
    if(verbose>2) printf("reading '%s'\n", fitfile);
    if((ret=fitRead(fitfile, &fit, verbose-2))) {
      fprintf(stderr, "Error in reading fit file: %s\n", fiterrmsg);
      continue;
    }

    /*
     *  Copy FIT data into RES structure
     */
    if(verbose>2) printf("converting\n");
    ret=fitToResult(&fit, &res, tmp);
    if(ret!=0) {
      fprintf(stderr, "Error: %s; not converted\n", tmp);
      fitEmpty(&fit); resEmpty(&res);
      continue;
    }


    /*
     *  Write result file
     */
    if(verbose>0) fprintf(stdout, "  writing %s\n", resfile);
    ret=resWrite(&res, resfile, verbose-3);
    if(ret) {
      fprintf(stderr, "Error in writing result file: %s\n", reserrmsg);
      fitEmpty(&fit); resEmpty(&res);
      continue;
    }

    resEmpty(&res); fitEmpty(&fit);

    fileNr++;
  } // next file


  if(fileNr<1) {
    fprintf(stderr, "Error: no fit file was successfully converted.\n");
    return(9);
  }
  if(verbose>0) printf("converted %d fit file(s).\n", fileNr);

  return(0);
}

/*****************************************************************************/

/*****************************************************************************/
/// @endcond
