/** @file fitedit.c
 *  @brief Edit parameter value in FIT file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Edit parameter value in FIT file 1, saving edited contents in file 2.",
  " ",
  "Usage: @P [Options] fitfile1 par_nr tac_nr newvalue tacfile2",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: fit2res, fit2dat, metabcor, fit_ppf",
  " ",
  "Keywords: FIT, tool, input, simulation, metabolite correction",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret, pn=-1, tn=-1;
  char  *cptr, file1[FILENAME_MAX], file2[FILENAME_MAX];
  FIT    fit;
  double v=nan("");


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  file1[0]=file2[0]=(char)0;
  fitInit(&fit);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!file1[0]) {
      strcpy(file1, argv[ai]); continue;
    } else if(pn<0) {
      pn=atoi(argv[ai]); if(pn>0) continue;
    } else if(tn<0) {
      tn=atoi(argv[ai]); if(tn>0) continue;
    } else if(isnan(v)) {
      v=atof_dpi(argv[ai]); continue;
    } else if(!file2[0]) {
      strcpy(file2, argv[ai]); continue;
    }
    fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!file2[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("file1 := %s\n", file1);
    printf("file2 := %s\n", file2);
    printf("parameter := %d\n", pn);
    printf("TAC := %d\n", tn);
    printf("new_value := %g\n", v);
  }
  if(verbose>3) MATHFUNC_TEST=verbose-3; else MATHFUNC_TEST=0;

  /*
   *  Read fit
   */
  if(verbose>2) printf("Reading fitfile\n");
  if((ret=fitRead(file1, &fit, verbose-2))) {
    fprintf(stderr, "Error %d in reading '%s': %s\n", ret, file1, fiterrmsg);
    return(4);
  }
  if(verbose>10) fitPrint(&fit);
  if(verbose>1) {
    printf("fit_tac_nr := %d\n", fit.voiNr);
  }

  /* Check that requested parameter exists */
  if(tn>fit.voiNr) {
    fprintf(stderr, "Error: FIT file does not contain TAC %d\n", tn);
    fitEmpty(&fit); return(5);
  }
  if(pn>fit.voi[tn-1].parNr) {
    fprintf(stderr, "Error: FIT TAC does not contain parameter %d\n", pn);
    fitEmpty(&fit); return(6);
  }
  
  /* Change the parameter value */
  if(verbose>0) printf("  %g -> %g\n", fit.voi[tn-1].p[pn-1], v);
  fit.voi[tn-1].p[pn-1]=v;
  
  /*
   *  Save edit fit
   */
  if(verbose>1) printf("saving edited fit in %s\n", file2);
  ret=fitWrite(&fit, file2); if(ret) {
    fprintf(stderr, "Error (%d) in writing '%s': %s\n", ret, file2, fiterrmsg);
    fitEmpty(&fit); return(11);
  }
  if(verbose>0) printf("edited fit written in %s\n", file2);

  /* Free memory */
  fitEmpty(&fit);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
