#!/bin/bash
#: Title      : test_fit_exp
#: Date       : 2019-06-19
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_exp$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f frontal.dft ] || [ ! -f frontal2.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Fit default function (3-exp) to a PET TAC. \n"
printf " Test case 2: Save parameters in FIT format. \n"
printf " Test case 3: Use fit2dat to calculate the fitted TAC. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dft output_100.svg

$PROGRAM -svg=output_100.svg frontal.dft output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=frontal.dft output.fit output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output_100.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.05 -rel=5 frontal.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Save parameters in RES format. \n"
printf " Expected result: File is created, but parameters may not be the same. \n\n"

rm -f output2.res

$PROGRAM frontal.dft output2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -res=1 -parameters=y -regions=y output.res output2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 1.1.0 \n"
printf " Test case 1: Fit 2-exp function to a PET TAC with only descending part. \n"
printf " Test case 2: Save parameters in FIT format. \n"
printf " Test case 3: Use fit2dat to calculate the fitted TAC. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dft output_110.svg

$PROGRAM -2e -svg=output_110.svg frontal2.dft output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=frontal2.dft output.fit output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.05 -rel=6 frontal2.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case 1: Fit 1-exp function to a PET TAC with only descending part. \n"
printf " Test case 2: Save parameters in FIT format. \n"
printf " Test case 3: Use fit2dat to calculate the fitted TAC. \n"
printf " Expected result: Reasonable fitting result with no errors. \n\n"

rm -f output.fit output.dft output_120.svg

$PROGRAM -1e -svg=output_120.svg frontal2.dft output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=frontal2.dft output.fit output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=1 -rel=20 frontal2.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.3.0 \n"
printf " Test case 1: User-given fit time range. \n"
printf " Test case 2: Fit 1-, 2-, and 3-exp functions. \n"
printf " Test case 3: Use option -wf. \n"
printf " Test case 4: Use fit2dat to calculate the fitted TAC. \n"
printf " Expected result: Reasonable fitting result with no errors. \n\n"

rm -f output.fit output?.dft output_130.svg
taccut frontal.dft 9.9 40.1 short.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

printf "\n 1e fitting \n\n"
$PROGRAM -1e -starttime=10 -endtime=40 -wf -svg=e1.svg frontal.dft output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=short.dft output.fit output1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.1 -rel=5 short.dft output1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2e fitting \n\n"
$PROGRAM -2e -starttime=10 -endtime=40 -wf -svg=e2.svg frontal.dft output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=short.dft output.fit output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.05 -rel=2 short.dft output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3e fitting \n\n"
$PROGRAM -3e -starttime=10 -endtime=40 -wf -svg=e3.svg frontal.dft output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=short.dft output.fit output3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.02 -rel=1 short.dft output3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg output_130.svg -s frontal.dft -l output1.dft output2.dft output3.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.4.0 \n"
printf " Test case 1: Fit is started from peak time. \n"
printf " Test case 2: Saved in RES format. \n"
printf " Expected result: Fit range is set correctly. \n\n"

rm -f output_140.res output_140.svg
$PROGRAM -2 -starttime=peak -endtime=30 -wf -svg=output_140.svg frontal.dft output_140.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output_140.res "Data range" "1.5 - 30"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.4.1 \n"
printf " Test case 1: Fit is started from peak time. \n"
printf " Test case 2: TAC times in seconds. \n"
printf " Expected result: Fit range is set correctly. \n\n"

tacformat -f=pmod --silent frontal.dft frontal.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec --silent frontal.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
rm -f output_141.res output_141.svg

$PROGRAM -2 -starttime=peak -wf -svg=output_141.svg frontal.tac output_141.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output_141.res "Data range" "90 - 3240"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.5.0 \n"
printf " Test case 1: TAC file contains more than one TAC. \n"
printf " Test case 2: Option -expneg is used. \n"
printf " Expected result: All TACs are fitted and parameters saved. \n\n"

rm -f multitac.dft multitac.fit output.dft
taccalc frontal.dft x 1.5 multitac.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd multitac.dft frontal.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren multitac.dft 1 frontl_1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren multitac.dft 2 frontl_2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacstudy multitac.dft koe2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut multitac.dft 4.9 9999 shorter.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -3 -starttime=5 -wf -expneg -svg=output_150.svg multitac.dft multitac.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=shorter.dft multitac.fit output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.02 -rel=3 shorter.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

