#!/bin/bash
#: Title      : test_fit_sigm
#: Date       : 2014-09-25
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../fit_sigm$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

if [ ! -f typical.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f typical_n1.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f data841a.dat ] || [ ! -f data841c.dat ] || [ ! -f data841d.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f data844a.dat ] || [ ! -f data844c.dat ] || [ ! -f data844d.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f data_w.dat ] || [ ! -f data_now.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f data_na.dat ] || [ ! -f data_naw.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


if [ ! -f typical.dat ]; then
  printf "\n making typical.dat \n"
  fit2dat -x=0.1,0.2,0.5,1,2,3,4,5,6,8,10 typical.fit typical.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f typical_n1.dat ]; then
  printf "\n making typical_n1.dat \n"
  fit2dat -x=0.1,0.2,0.5,1,2,3,4,5,6,8,10 typical_n1.fit typical_n1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f typical_s3.dat ]; then
  printf "\n making typical_s3.dat \n"
  fit2dat -x=0.1,1,8 typical.fit typical_s3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f typical_s4.dat ]; then
  printf "\n making typical_s4.dat \n"
  fit2dat -x=0.1,0.5,3,10 typical.fit typical_s4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Executed with typical data and with default settings. \n"
printf " Test case 2: Options and arguments are correctly set. \n"
printf " Expected result: Good fitting result with no errors. \n"
printf " Options and variables have correct values in test mode. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d2 typical.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=typical.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.5 -rel=1 typical.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval stdout.txt datfile "typical.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt fitfile "output.fit"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt outfile ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt svgfile ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt type "841"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt MRL_check "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Existing FIT file is saved as backup file. \n"
printf " Expected result: Original FIT file is saved with extension .bak \n\n"

rm -f output.fit.bak

$PROGRAM typical.dat output.fit 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.fit ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.fit.bak ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n===============================================================\n\n"

printf "\n 1.1.0 \n"
printf " Test case: Executed with typical data with n constrained to 1. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -n=1 typical_n1.dat output.fit 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=typical_n1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.5 -rel=1 typical_n1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n===============================================================\n\n"

printf "\n 1.2.0 \n"
printf " Test case: Data file may contain more than one TAC. \n"
printf " Expected result: All curves are fitted. \n\n"

rm -f output.fit output.dat
tacadd -ovr temp.dat typical.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -force temp.dat typical_n1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -svg=test_1_2_0.svg temp.dat output.fit 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=temp.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -abs=0.5 -rel=1 temp.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n===============================================================\n"
printf "   Extreme tests with difficult data \n"
printf "===============================================================\n\n"

printf "\n 2.0.0 \n"
printf " Test case: Gentle sigmoid curve with default settings. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -svg=test_2_0_0.svg data841a.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data841a.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.5 -rel=1 data841a.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Steep sigmoid curve with default settings. \n"
printf " Expected result: Reasonable fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -svg=test_2_0_1.svg data841c.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data841c.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.5 -rel=5 data841c.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.2 \n"
printf " Test case: Steady curve with default settings. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -svg=test_2_0_2.svg data841d.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data841d.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.1 -rel=0.1 data841d.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: Gentle sigmoid curve with background fitting. \n"
printf " Test case 2: Option -b for fitting background. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b -svg=test_2_1_0.svg data844a.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844a.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.2 -rel=0.5 data844a.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Steep sigmoid curve with background fitting. \n"
printf " Expected result: Reasonable fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b -svg=test_2_1_1.svg data844c.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844c.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.5 -rel=5 data844c.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.2 \n"
printf " Test case: Steady curve with background fitting. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b -svg=test_2_1_2.svg data844d.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844d.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.1 -rel=0.5 data844d.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 2.2.0 \n"
printf " Test case 1: Gentle sigmoid curve with fixed background. \n"
printf " Test case 2: Option -b=value for fixing background. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b=0.2 -svg=test_2_2_0.svg data844a.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844a.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.2 -rel=0.5 data844a.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case 1: Steep sigmoid curve with fixed background. \n"
printf " Test case 2: Option -b=value with decimal comma. \n"
printf " Expected result: Reasonable fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b=0,3 -svg=test_2_2_1.svg data844c.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844c.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.5 -rel=5 data844c.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.2 \n"
printf " Test case: Steady curve with fixed background. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b=0.5 -svg=test_2_2_2.svg data844d.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844d.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.1 -rel=0.5 data844d.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 2.3.0 \n"
printf " Test case: Typical curve but extra parameter fitted. \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -b -svg=test_extra.svg typical.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=typical.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.2 -rel=0.5 typical.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n\n===============================================================\n"
printf "   Testing option for constraining parameter n \n"
printf "===============================================================\n\n"


printf "\n 3.0.0 \n"
printf " Test case: Option -n=1 for fixing parameter n \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -n=1 -svg=test_3_0_0.svg data841e.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data841e.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.1 -rel=0.5 data841e.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Option -n=1 for fixing parameter n together with \n"
printf "            backgound fitting \n"
printf " Expected result: Good fitting result with no errors. \n\n"

rm -f output.fit output.dat

$PROGRAM -n=1 -b -svg=test_3_1_0.svg data844e.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data844e.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.1 -rel=0.5 data844e.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n===============================================================\n"
printf "   Testing weights, weighting options, and missing values \n"
printf "===============================================================\n\n"


printf "\n 4.0.0 \n"
printf " Test case: File contains weights, no weighting options applied. \n"
printf " Expected result: Weights are read correctly as seen in test mode. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 data_w.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 2, 3, 4, 5, 6, 7, 7"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case: File contains weights, which can be overwritten with option -w1. \n"
printf " Expected result: Weights are set correctly as seen in test mode. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 -w1 data_w.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.1.0 \n"
printf " Test case: File does not contains weights, which are automatically set to 1. \n"
printf " Expected result: Weights are set correctly as seen in test mode. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 -w1 data_now.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.1 \n"
printf " Test case: File does not contains weights, which are set with option -wf. \n"
printf " Expected result: Weights are set correctly as seen in test mode. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 -wf data_now.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "0.461538, 0.692308, 0.923077, 1.15385, 1.38462, 1.38462"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 4.2.0 \n"
printf " Test case: File contains missing value but no weights. Weights are not set. \n"
printf " Expected result: Missing value gets zero weight. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 data_na.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 1, 1, 0, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[2] "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.2.1 \n"
printf " Test case: File contains missing value but no weights. Weights are set. \n"
printf " Expected result: Missing value gets zero weight. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 -w1 data_na.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 1, 1, 0, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[2] "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.2.2 \n"
printf " Test case: File contains missing value and weights. Weights are not set. \n"
printf " Expected result: Missing value gets zero weight. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 data_naw.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 2, 3, 0, 5, 6, 7, 8"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[2] "1, 2, 3, 4, 5, 6, 7, 8"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.2.3 \n"
printf " Test case: File contains missing value and weights. Weights are set. \n"
printf " Expected result: Missing value gets zero weight. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -d4 -w1 data_naw.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[1] "1, 1, 1, 0, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights[2] "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n\n===============================================================\n"
printf "   Tests for error handling \n"
printf "===============================================================\n\n"


printf "\n 5.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -stupidoption typical.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.1.0 \n"
printf " Test case: Obligatory argument is missing. \n"
printf " Expected result: error code returned. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.1.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM typical.dat output.fit extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.2.0 \n"
printf " Test case: Nonexisting data file. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM nonexistingfile.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.3.0 \n"
printf " Test case: Wrong data file format. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

echo bleah meh > temp.dat
echo more rubbish >> temp.dat
rm -f output.fit output.dat stdout.txt

$PROGRAM temp.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'temp.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 5.4.0 \n"
printf " Test case: Data file contains no more samples than there are \n"
printf " free parameters to fit. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM typical_s3.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples for fitting in typical_s3.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM typical_s4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -b typical_s4.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples for fitting in typical_s4.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 5.4.1 \n"
printf " Test case 1: Catenated data file contains no more unique samples \n"
printf "              than there are free parameters to fit. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

rm -f output.fit output.dat stdout.txt temp.dat
dftcat -both typical_s3.dat typical_s3.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples for fitting in temp.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n===============================================================\n"
printf "   Testing MRL \n"
printf "===============================================================\n\n"


printf "\n 6.0.0 \n"
printf " Test case: Data which can be fitted well passes MRL test. \n"
printf " Expected result: No errors or warnings reported. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -MRL -d2 typical.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt MRL_check "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.0 \n"
printf " Test case: Data which can not be fitted well does not pass MRL test. \n"
printf "            Option to stop with error if MRL test not passed is turned on. \n"
printf " Expected result: Appropriate error message and error code returned. \n\n"

cp -f typical.dat temp.dat 
printf "12 0.1\n" >> temp.dat
printf "14 0.1\n" >> temp.dat
rm -f output.fit output.dat stdout.txt
cat temp.dat

$PROGRAM -MRL -d2  -svg=badfit.svg temp.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "bad fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.1 \n"
printf " Test case: Data which can not be fitted well does not pass MRL test. \n"
printf "            Option to stop with error if MRL test not passed is NOT turned on. \n"
printf " Expected result: Appropriate warning message returned, but no errors. \n\n"

$PROGRAM -svg=badfit.svg temp.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "bad fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n===============================================================\n"
printf "   Testing optional output data \n"
printf "===============================================================\n\n"


printf "\n 7.0.0 \n"
printf " Test case: SVG plot can be saved optionally. \n"
printf " Expected result: SVG file is created. \n\n"

rm -f output.fit output.dat stdout.txt output.svg

$PROGRAM -svg=output.svg typical.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.svg ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 7.1.0 \n"
printf " Test case: Fitted TAC can be saved optionally. \n"
printf " Expected result: TAC file is created, and contents match the one \n"
printf "                  computed with fit2dat \n\n"

rm -f output.fit output.dat stdout.txt output2.dat

$PROGRAM -fit=output2.dat typical.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output2.dat ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=typical.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=y -y=y -abs=0.01 -rel=0.1 output.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 7.2.0 \n"
printf " Test case: Fitted parameters can be saved in result format. \n"
printf " Expected result: RES file is created, and contents are close to correct. \n\n"

rm -f output.fit output.res

$PROGRAM -res=output.res typical.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.res ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -header=y -param=y -regions=y -abs=0.000001 typical.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

