/** @file dftdblf.c
 *  @brief Doubles the TAC sample number by making each sample/frame
           into two by linear interpolation.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
#include "libtpcmodel.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Doubles the TAC sample number by making each sample/frame into two",
  "by means of linear interpolation. This is mainly for simulation",
  "and software testing purposes.",
  " ",
  "Usage: @P tacfile outputfile",
  " ",
  "Options:",
  " -mid",
  "     Mid frame times are used even if frame start and end times are",
  "     available.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: interpol, tacsetx, simframe, tacframe, dftinteg, taccat",
  " ",
  "Keywords: TAC, simulation, time frame",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  int     mid_always=0;
  char    dfile[FILENAME_MAX], rfile[FILENAME_MAX], *cptr;
  DFT     dft, dft2;
  

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dfile[0]=rfile[0]=(char)0;
  dftInit(&dft); dftInit(&dft2);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "MID", 3)==0) {
      mid_always=1; continue;
    }
    fprintf(stderr, "Error: unknown option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!dfile[0]) {strlcpy(dfile, argv[ai], FILENAME_MAX); continue;}
    else if(!rfile[0]) {strlcpy(rfile, argv[ai], FILENAME_MAX); continue;}
    fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!rfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("dfile := %s\n", dfile);
    printf("rfile := %s\n", rfile);
    printf("mid_always := %d\n", mid_always);
  }

  /*
   *  Read data file
   */
  if(verbose>1) printf("reading %s\n", dfile);
  if(dftRead(dfile, &dft)) {
    fprintf(stderr, "Error in reading '%s': %s\n", dfile, dfterrmsg);
    return(2);
  }
  /* If using frame mid times was required, then do it */
  if(mid_always) {
    dft.timetype=DFT_TIME_MIDDLE;
    if(verbose>1) printf("timetype is set to 0.\n");
  }

  /*
   *  Double the frame number
   */
  if(verbose>1) printf("doubling frames\n");
  ret=dftDoubleFrames(&dft, &dft2);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot double the frame number (%d).\n", ret);
    dftEmpty(&dft); dftEmpty(&dft2); return(8);
  } 
  dftEmpty(&dft);

  /*
   *  Save TACs
   */
  if(verbose>1) printf("writing %s\n", rfile);
  ret=dftWrite(&dft2, rfile); dftEmpty(&dft2);
  if(ret) {
    fprintf(stderr, "Error in writing '%s': %s\n", rfile, dfterrmsg);
    return(11);
  }
  if(verbose>0) printf("  %s written.\n", rfile);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
