/** @file upet2e7.c
 *  @brief Convert Inveon/microPET image to ECAT 7 image volume format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Conversion of Siemens Inveon/MicroPET CT or PET image to ECAT 7",
  "image volume format. Please note that a .NET image format conversion",
  "tool with GUI (ImageConverter) is also available.",
  "MicroPET image consists of two files with extension .img and img.hdr;",
  "both are required in conversion.",
  "Note that also Analyze and dual file format NIfTI images consist",
  "of equally named two files.",
  " ",
  "Usage: @P [Options] <MicroPET image> [ECAT filename]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P ae72.pet.img ae72.v",
  " ",
  "See also: upet2sif, nii2ecat, ana2ecat, flat2img, img2flat",
  " ",
  "Keywords: image, format conversion, MicroPET, Inveon",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  main()
 */
int main(int argc, char *argv[])
{
  int      ai, help=0, version=0, verbose=1;
  char    *cptr;
  char     ecatfile[FILENAME_MAX], upetname[FILENAME_MAX];
  int      ret;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  upetname[0]=ecatfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(upetname, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(ecatfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!upetname[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print options */
  if(verbose>1) {
    printf("upetname := %s\n", upetname);
    if(ecatfile[0]) printf("ecatfile := %s\n", ecatfile);
    fflush(stdout);
  }

  /*
   *  Make sure that we have the header and image files
   */
  char upetheader[FILENAME_MAX], upetimage[FILENAME_MAX];
  ret=upetExists(upetname, upetheader, upetimage, verbose-1);
  if(ret<1) {
    fprintf(stderr, "Error: %s does not exist or is not Inveon/MicroPET image.\n",
      upetname);
    return(2);
  } else if(ret<2) {
    fprintf(stderr, "Error: Inveon/MicroPET image %s not found.\n",
      upetname);
    return(2);
  }
  if(verbose>1) {
    printf("upetheader := %s\n", upetheader);
    printf("upetimage := %s\n", upetimage);
    fflush(stdout);
  }
  /* Make ECAT filename, if not given by user */
  if(!ecatfile[0]) {
	strcpy(ecatfile, upetimage);
    cptr=strrchr(ecatfile, '.');
    if(cptr!=NULL) {
      if(strncasecmp(cptr, ".IMG", 4)==0) *cptr=(char)0;
    }
    strcat(ecatfile, ".v");
    if(verbose>1) printf("ecatfile := %s\n", ecatfile);
  }

  /*
   *  Convert image
   */
  if(verbose>0) {
    fprintf(stdout, "  converting %s ...\n", upetname);
    fflush(stdout);
  }
  ret=imgMicropetToEcat7(upetname, ecatfile, verbose-1);
  if(ret!=STATUS_OK) {
    fprintf(stderr, "Error: %s\n", imgStatus(ret));
    return(11);
  }
  if(verbose>0) {
    fprintf(stdout, "  Image saved in ECAT format in %s\n", ecatfile);
    fflush(stdout);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
