/** @file upet2sif.c
 *  @brief Extract SIF from Inveon/microPET image header.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Constructs a scan information file (SIF) from Siemens Inveon/microPET",
  "image header.",
  "MicroPET image consists of two files with extension .img and .img.hdr;",
  "only header file is needed here.",
  " ",
  "Usage: @P [Options] <MicroPET header file> <SIF filename>",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P ae72.pet.img.hdr ae72.sif",
  " ",
  "See also: upet2e7, tacframe, tacweigh, eframe, sifcat, sifisot",
  " ",
  "Keywords: SIF, image, time, MicroPET, Inveon",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  main()
 */
int main(int argc, char *argv[])
{
  int      ai, help=0, version=0, verbose=1;
  char    *cptr, upetheader[FILENAME_MAX];
  char     siffile[FILENAME_MAX], upetname[FILENAME_MAX];



  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  upetname[0]=siffile[0]=upetheader[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(upetname, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(siffile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!siffile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print options */
  if(verbose>1) {
    printf("upetname := %s\n", upetname);
    printf("siffile := %s\n", siffile);
    fflush(stdout);
  }

  /*
   *  Make sure that we have the header file name
   */
  if(upetExists(upetname, upetheader, NULL, verbose-1)<1) {
    fprintf(stderr, "Error: %s does not exist or is not Inveon/MicroPET header.\n",
      upetname);
    return(2);
  }


  /*
   *  Open Micropet Header
   */
  FILE *fp;
  if((fp=fopen(upetheader, "r"))==NULL) {
    fprintf(stderr, "Error: cannot open %s for reading.\n", upetheader);
    return(3);
  }

  /*
   *  Read SIF
   */
  SIF sif; sifInit(&sif);
  if(verbose>1) printf("reading scan information for SIF\n");
  if(imgGetMicropetSIF(fp, &sif)!=0) {
    fprintf(stderr, "Error: non-supported header file %s.\n", upetheader);
    fclose(fp);
    return(4);
  }
  /* Close header file */
  fclose(fp);

  /* Set study number */
  studynr_from_fname2(upetname, sif.studynr, 0);
  //studynr_in_fname(upetname, sif.studynr);

  /*
   *  Write SIF
   */
  if(sifWrite(&sif, siffile)!=0) {
    fprintf(stderr, "Error: cannot write %s.\n", siffile);
    return(11);
  }
  if(verbose>0) fprintf(stdout, "SIF saved in %s\n", siffile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
