/** @file absszero.c
    @brief Remove initial bumps in on-line blood sampler data file.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "tpcabss.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Delete initial \"bumps\" in Scanditronics, GEMS, or Allogg on-line blood",
  "sampler data by setting the activities during the specified sample time",
  "range (in seconds) to zero.",
  "Notice that the original data file is overwritten by default.",
  " ",
  "Usage: @P [options] abssfile starttime endtime",
  " ",
  "Options:",
  " -o=<filename>",
  "     Name for corrected file; by default the original file is overwritten.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P us1328.bld 0 40",
  " ",
  "See also: abssbkg, absscal, absstime, abssfch, abssexam, fitdelay",
  " ",
  "Keywords: input, blood, background, ABSS",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  char    abssfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char   *cptr;
  double  startTime=nan("");
  double  endTime=nan("");
  int     ret;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  abssfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "O=", 2)==0 && strlen(cptr)>2) {
      strlcpy(outfile, cptr+2, FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(abssfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    if(atofCheck(argv[ai], &startTime)) {
      fprintf(stderr, "Error: invalid time range.\n");
      return(1);
    }
    ai++;
  }
  if(ai<argc) {
    if(atofCheck(argv[ai], &endTime) || endTime<=startTime) {
      fprintf(stderr, "Error: invalid time range.\n");
      return(1);
    }
    ai++;
  }
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Did we get all the information that we need? */
  if(isnan(endTime)) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) strcpy(outfile, abssfile);


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("abssfile := %s\n", abssfile);
    printf("startTime := %g\n", startTime);
    printf("endTime := %g\n", endTime);
    printf("outfile := %s\n", outfile);
    fflush(stdout);
  }

  /*
   *  Read ABSS file
   */
  if(verbose>1) printf("reading %s\n", abssfile);
  TAC abss; tacInit(&abss);
  ret=tacRead(&abss, abssfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&abss); return(2);
  }
  if(verbose>1) {
    printf("sampleNr := %d\n", abss.sampleNr);
  }
  if(verbose>2) {
    printf("fileformat := %d\n", abss.format);
    printf("tacNr := %d\n", abss.tacNr);
  }
  if(abss.format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss.format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss.format!=TAC_FORMAT_ABSS_GEMS &&
     abss.format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    fprintf(stderr, "Error: not valid ABSS format.\n");
    tacFree(&abss); return(2);
  }
  if(verbose>100) abssWrite(&abss, stdout, NULL);
  if(abss.tacNr<1 || abss.sampleNr<1) {
    fprintf(stderr, "Error: not valid ABSS format.\n");
    tacFree(&abss); return(2);
  }


  /* 
   *  Remove the background from the data
   */
  if(verbose>1) {printf("removing the background\n"); fflush(stdout);}
  int i, j, n=0;
  for(i=0; i<abss.sampleNr; i++) {
    if(abss.x[i]<startTime || abss.x[i]>endTime) continue;
    for(j=0; j<abss.tacNr; j++) abss.c[j].y[i]=0.0;
    n++;
  }
  if(verbose>2) {printf("%d sample(s) set to zero.\n", n); fflush(stdout);}
  if(n==0) {
    fprintf(stderr, "Warning: no samples inside the time range.\n");
    if(verbose>0) {printf("Note: file not modified.\n"); fflush(stdout);}
    tacFree(&abss); return(0);
  }

  if(verbose>1) {
    printf("writing corrected data file in %s\n", outfile); fflush(stdout);}
  FILE *fp;
  fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing.\n");
    tacFree(&abss); return(11);
  }
  ret=abssWrite(&abss, fp, &status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&abss); return(12);
  }
  if(verbose>0) {
    printf("corrected data written in %s\n", outfile); 
    fflush(stdout);
  }

  tacFree(&abss);  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
