/** @file csvtrps.c
    @brief Transpose data in CSV table.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Transpose data in a CSV or TSV file.",
  "Table cells or lines starting with '#' are removed.",
  " ",
  "Usage: @P [options] csvfile [outputfile]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P data.csv transposed.csv",
  " ",
  "See also: rmcmnts, csvrmcol, tacformat, parformat",
  " ",
  "Keywords: CSV, TAC",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/*
 *  Main
 */
int main(int argc, char *argv[])
{
  int ai, help=0, version=0, verbose=1;
  char csvfile[FILENAME_MAX], outfile[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  csvfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(csvfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]); return(1);}

  if(!csvfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) strcpy(outfile, csvfile);

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("csvfile := %s\n", csvfile);
    printf("outfile := %s\n", outfile);
    fflush(stdout);
  }


  /*
   *  Read the CSV (or TSV) file
   */
  CSV csv; csvInit(&csv);
  if(verbose>1) fprintf(stdout, "reading %s\n", csvfile);
  {
    FILE *fp;
    fp=fopen(csvfile, "r");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file %s\n", csvfile);
      return(2);
    }
    if(csvRead(&csv, fp, &status)!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      fclose(fp); return(2);
    }
    fclose(fp);
  }
  if(verbose>2) {
    printf("colNr := %d\n", csv.col_nr);
    printf("rowNr := %d\n", csv.row_nr);
    printf("separator := '%c'\n", csv.separator);
  }
  if(csv.col_nr<1 || csv.row_nr<1) {
    fprintf(stderr, "Error: invalid data.\n");
    csvFree(&csv); return(2);
  }
  if(csv.col_nr==1 && csv.row_nr==1) {
    if(verbose>0) fprintf(stderr, "Notice: file has only one cell.\n");
  }


  /*
   *  Remove comments and Transpose
   */
  if(csvRemoveComments(&csv)!=TPCERROR_OK) {
    fprintf(stderr, "Error: invalid data.\n");
    csvFree(&csv); return(3);
  }
  if(csvRemoveEmptyRows(&csv)!=TPCERROR_OK) {
    fprintf(stderr, "Error: invalid data.\n");
    csvFree(&csv); return(3);
  }
  if(csvTranspose(&csv)!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot transpose data.\n");
    csvFree(&csv); return(4);
  }


  /*
   *  Save the edited file 
   */
  if(verbose>1) printf("writing %s\n", outfile);
  {
    FILE *fp; fp=fopen(outfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      csvFree(&csv); return(11);
    }
    int ret=csvWrite(&csv, 0, fp, &status);
    fclose(fp); csvFree(&csv);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
    if(verbose>0) printf("edited data saved in %s\n", outfile);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
