/** @file dcmlhdr.c
 *  @brief List the DICOM header contents.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the header contents of a DICOM file.",
  " ",
  "Usage: @P [options] [> outputfile]",
  " ",
  "Options:",
  " -group=<tag group>",
  "     List only contents with specified tag group.",
  " -element=<tag element>",
  "     List only contents with specified tag element.",
  " -tag=<Y|n>",
  "     List (y, default) or do not list (n) the element tag.",
  " -tagname=<Y|n>",
  "     List (y, default) or do not list (n) the tag description.",
  " -VR=<y|N>",
  "     List (y) or do not list (n, default) the element VR.",
  " -VL=<y|N>",
  "     List (y) or do not list (n, default) the element VL.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P -group=0002 -element=0010 f1.dcm",
  " ",
  "See also: dcmdict, dcmframe, dcmmlist, dcmxform, lmhdr, ana_lhdr, iftlist",
  " ",
  "Keywords: DICOM, header",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char *cptr;
  int   group=-1, element=-1;
  int   listVR=0; // 0=no, 1=yes
  int   listVL=0; // 0=no, 1=yes
  int   listTag=1; // 0=no, 1=yes
  int   listTagname=1; // 0=no, 1=yes
  int   listValue=1; // 0=no, 1=yes
  char  dcmfile[FILENAME_MAX];
 

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dcmfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "GROUP=", 6)==0) {
      unsigned int ui=0;
      if(sscanf(cptr+6, "%x", &ui)==1) {group=(int)ui; continue;}
    } else if(strncasecmp(cptr, "ELEMENT=", 8)==0) {
      unsigned int ui=0;
      if(sscanf(cptr+8, "%x", &ui)==1) {element=(int)ui; continue;}
    } else if(strcasecmp(cptr, "VR")==0) {
      listVR=1; continue;
    } else if(strncasecmp(cptr, "VR=", 3)==0) {
      if((listVR=tpcYesNo(cptr+3))>=0) continue;
    } else if(strncasecmp(cptr, "VL=", 3)==0) {
      if((listVL=tpcYesNo(cptr+3))>=0) continue;
    } else if(strncasecmp(cptr, "TAG=", 4)==0) {
      if((listTag=tpcYesNo(cptr+4))>=0) continue;
    } else if(strncasecmp(cptr, "TAGNAME=", 8)==0) {
      if((listTagname=tpcYesNo(cptr+8))>=0) continue;
    } else if(strncasecmp(cptr, "VALUE=", 6)==0) {
      if((listValue=tpcYesNo(cptr+6))>=0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(dcmfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Did we get all the information that we need? */
  if(!dcmfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("dcmfile := %s\n", dcmfile);
    printf("tag := (");
    if(group<0) printf("0xXXXX,"); 
    else printf("0x%04x,", (unsigned short int)group);
    if(element<0) printf("0xXXXX)\n"); 
    else printf("0x%04x)\n", (unsigned short int)element);
    printf("listVR := %d\n", listVR);
    printf("listVL := %d\n", listVL);
    printf("listTag := %d\n", listTag);
    printf("listTagname := %d\n", listTagname);
    printf("listValue := %d\n", listValue);
  }

  /* Check that we're requested to print something */
  if(!listVR && !listVL && !listTag && !listTagname && !listValue) {
    fprintf(stderr, "Error: all prints were turned off.\n");
    return(1);
  }

  /*
   *  Read file
   */
  DCMFILE dcm; dcmfileInit(&dcm);
  int ret;
  ret=dcmFileRead(dcmfile, &dcm, 1, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    dcmfileFree(&dcm); return(2);
  }
  if(verbose>0) {
    printf("max_tree_depth := %u\n", dcmfileMaxDepth(&dcm));
  }

  /* List the contents */
  unsigned int itemNr=0;
  unsigned int listedNr=0;
  DCMITEM *d1=dcm.item;
  while(d1!=NULL) {
    unsigned int parent1Shown=0;
    /* If user specified tag group and/or element, then check the match */
    if((group<0 || group==d1->tag.group) && 
       (element<0 || element==d1->tag.element))
    {
      /* Print the tag */
      if(listTag) printf("(%04x,%04x) ", d1->tag.group, d1->tag.element);
      /* Print the tag description */
      if(listTagname) printf("%s ", dcmDictIndexDescr(dcmDictFindTag(&d1->tag)));
      /* Print the equal sign for IFT format */
      if((listTag || listTagname) && (listVR || listVL || listValue)) printf(":= ");
      /* Print the VR */
      if(listVR) printf("%s ", dcmVRName(d1->vr));
      /* Print the VL */
      if(listVL && d1->vl!=0xFFFFFFFF) printf("%u ", d1->vl);
      /* Print the value */
      if(listValue) {
        char *buf=dcmValueString(d1); printf("%s", buf); free(buf);
      }
      /* Print the linefeed */
      printf("\n");
      parent1Shown=1;
    }
    /* If this element has children, then print those */
    if(d1->child_item!=NULL) {
      DCMITEM *d2=d1->child_item;
      while(d2!=NULL) {
        unsigned int parent2Shown=0;
        /* If user specified tag group and/or element, then check the match */
        if((group<0 || group==d2->tag.group) && 
           (element<0 || element==d2->tag.element))
        {
          /* Print parent information if not yet printed */
          if(!parent1Shown) {
            printf("(%04x,%04x) ", d1->tag.group, d1->tag.element);
            printf("%s ", dcmDictIndexDescr(dcmDictFindTag(&d1->tag)));
            printf("\n");
            parent1Shown=1;
          }
          /* Print the indentation */
          printf("  ");
          /* Print the tag */
          if(listTag) printf("(%04x,%04x) ", d2->tag.group, d2->tag.element);
          /* Print the tag description */
          if(listTagname) printf("%s ", dcmDictIndexDescr(dcmDictFindTag(&d2->tag)));
          /* Print the equal sign for IFT format */
          if((listTag || listTagname) && (listVR || listVL || listValue)) printf(":= ");
          /* Print the VR */
          if(listVR) printf("%s ", dcmVRName(d2->vr));
          /* Print the VL */
          if(listVL && d2->vl!=0xFFFFFFFF) printf("%u ", d2->vl);
          /* Print the value */
          if(listValue) {
            char *buf=dcmValueString(d2); printf("%s", buf); free(buf);
          }
          /* Print the linefeed */
          printf("\n");
          parent2Shown=1;
        }
        /* If this element has children, then print those */
        if(d2->child_item!=NULL) {
          DCMITEM *d3=d2->child_item;
          while(d3!=NULL) {
            /* If user specified tag group and/or element, then check the match */
            if((group>=0 && group!=d3->tag.group) || 
               (element>=0 && element!=d3->tag.element))
            {
              d3=d3->next_item; itemNr++; continue;
            }
            /* Print parent information if not yet printed */
            if(!parent1Shown) {
              printf("(%04x,%04x) ", d1->tag.group, d1->tag.element);
              printf("%s ", dcmDictIndexDescr(dcmDictFindTag(&d1->tag)));
              printf("\n");
              parent1Shown=1;
            }
            if(!parent2Shown) {
              printf("  (%04x,%04x) ", d2->tag.group, d2->tag.element);
              printf("%s ", dcmDictIndexDescr(dcmDictFindTag(&d2->tag)));
              printf("\n");
              parent2Shown=1;
            }
            /* Print the indentation */
            printf("    ");
            /* Print the tag */
            if(listTag) printf("(%04x,%04x) ", d3->tag.group, d3->tag.element);
            /* Print the tag description */
            if(listTagname) printf("%s ", dcmDictIndexDescr(dcmDictFindTag(&d3->tag)));
            /* Print the equal sign for IFT format */
            if((listTag || listTagname) && (listVR || listValue)) printf(":= ");
            /* Print the VR */
            if(listVR) printf("%s ", dcmVRName(d3->vr));
            /* Print the VL */
            if(listVL && d3->vl!=0xFFFFFFFF) printf("%u ", d3->vl);
            /* Print the value */
            if(listValue) {
              char *buf=dcmValueString(d3); printf("%s", buf); free(buf);
            }
            /* Print the linefeed */
            printf("\n");
            d3=d3->next_item;
            itemNr++; listedNr++;
          }
        }
        d2=d2->next_item;
        itemNr++; listedNr++;
      }
    }
    d1=d1->next_item;
    itemNr++; listedNr++;
  }
  dcmfileFree(&dcm);


  if(verbose>1) {
    printf("elementNr := %d\n", itemNr);
    if(listedNr!=itemNr || verbose>2)
      printf("selected_tags := %d\n", listedNr);
  }
  if(itemNr==0) {
    fprintf(stderr, "Error: invalid header in %s\n", dcmfile);
    return(3);
  }
  if(listedNr==0) {
    fprintf(stderr, "Error: required tags not found.\n");
    return(4);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
