/** @file execlong.c
 *  @brief Execute shell command based on IFT file contents.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Execute shell command based on the contents of given IFT file.",
  "IFT file may or may not contain key names; any key names are ignored.",
  "First line must contain the program/command name.",
  "Comment lines are ignored.",
  " ",
  "Usage: @P [options] filename",
  " ",
  "Options:",
  " -dry",
  "     Just print the command line in stdout.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P command.ift",
  " ",
  "Program uses system() function; its return value is not standardized,",
  "therefore do not trust the return value of this function either.",
  " ",
  "See also: iftadd, iftdel, iftisval, iftvalc",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  char *cptr, iftfile[FILENAME_MAX];
  int dryrun=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "DRY")==0) {
      dryrun=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(iftfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    fprintf(stderr, "Error: too many arguments.\n");
    return(1);
  }

  /* Is something missing? */
  if(!iftfile[0]) {
    fprintf(stderr, "Error: missing command parameter file; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("iftfile := %s\n", iftfile);
    printf("dryrun := %d\n", dryrun);
  }

  /* Check that command processor is available, if not a dry run */
  if(dryrun==0 && system(NULL)==0) {
    fprintf(stderr, "Error: no command processor available.\n");
    return(10);
  }


  /*
   *  Read IFT file
   */
  if(verbose>1) printf("reading %s\n", iftfile);
  IFT ift;
  iftInit(&ift);
  FILE *fp=fopen(iftfile, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    return(2);
  }
  ret=iftRead(&ift, fp, 0, 0, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift); return(2);
  }
  if(verbose>2) printf("list size: %d item(s)\n", ift.keyNr);

  /* Calculate the length of command-line */
  int i, len=0;
  for(i=0; i<ift.keyNr; i++) {
    if(i>0) len++;
    len+=strlen(ift.item[i].value);
  }
  if(i<1) {
    fprintf(stderr, "Error: invalid file contents.\n");
    iftFree(&ift); return(3);
  }


  /* Join the values */
  char cmdline[len+1]; cmdline[0]=(char)0;
  for(i=0; i<ift.keyNr; i++) {
    if(i>0) strcat(cmdline, " ");
    strcat(cmdline, ift.item[i].value);
  }
  iftFree(&ift);

  /* If dry run was requested, then print it and exit */
  if(dryrun!=0) {
    fflush(stdout);
    fprintf(stdout, "%s\n", cmdline);
    fflush(stdout);
    return(0);
  }

  /* In verbose mode print it too */
  if(verbose>1) {
    fflush(stdout);
    fprintf(stdout, "command := %s\n", cmdline);
    fflush(stdout);
  }

  /* Otherwise try to execute the command */
  ret=system(cmdline); // return value is not standardized, so do not trust it too much
  // in case of error it may even be that we never get this far
  if(ret!=0 || verbose>2) printf("  %d returned.\n", ret);
  fflush(stdout);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
