/** @file iftedit.c
 *  @brief Change the value of specified key in Interfile-type file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Change the value of specified key in Interfile-type file or files that can",
  "contain similar header data (such as TAC and parameter files).",
  "Any additional lines with the same key name are removed.",
  "If new value is not given, then previous key and value are deleted.",
  "If key does not exist in the file, then the key with value are added.",
  " ",
  "Usage: @P [Options] filename key [value]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P rabbit12.par studynr tg2r12",
  " ",
  "See also: iftlist, iftren, iftvalc, iftadd, iftdel, iftisval",
  " ",
  "Keywords: header, parameter, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret;
  char  iftfile[FILENAME_MAX], keyname[FILENAME_MAX], keyvalue[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftfile[0]=keyname[0]=keyvalue[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //char *cptr=argv[ai]+1;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(iftfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(keyname, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(keyvalue, argv[ai++], FILENAME_MAX);
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing? */
  if(!keyname[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n"); fflush(stdout);
    printf("iftfile := %s\n", iftfile);
    printf("keyname := %s\n", keyname);
    printf("keyvalue := %s\n", keyvalue);
  }


  /*
   *  Read IFT file
   */
  IFT ift; iftInit(&ift);

  if(verbose>1) {printf("reading %s\n", iftfile); fflush(stdout);}
  FILE *fp=fopen(iftfile, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    return(2);
  }
  ret=iftRead(&ift, fp, 0, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(2);
  }
  if(verbose>2) {printf("list size: %d item(s)\n", ift.keyNr); fflush(stdout);}
  int foundNr=iftFindNrOfKeys(&ift, keyname);
  if(verbose>1) {printf("%d match(es) found.\n", foundNr); fflush(stdout);}
  if(foundNr==0 && !keyvalue[0]) {
    fprintf(stderr, "Note: file contains no key '%s'\n", keyname);
    iftFree(&ift); return(0);
  }
  if(foundNr>1) {
    if(verbose>0) {printf("deleting duplicate keys\n"); fflush(stdout);}
    iftDeleteDuplicateKeys(&ift, &status);
  }

  /*
   *  Find the key from IFT
   */
  int li;
  /* Search for the first occurrence of keyname */
  li=iftFindKey(&ift, keyname, 0);
  if(li<0) {
    if(verbose>2) {printf("adding new key\n"); fflush(stdout);}
    if(iftPut(&ift, keyname, keyvalue, 1, &status)!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      iftFree(&ift); return(3);
    }
    if(verbose>0) {printf("added new key with given value\n"); fflush(stdout);}
    /* saving later */
  } else if(keyvalue[0]) {
    if(verbose>3) printf("key_index := %d\n", li);
    if(verbose>0) printf("resetting value '%s' -> '%s'\n", ift.item[li].value, keyvalue);
    if(iftReplaceValue(&ift, li, keyvalue, &status)!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      iftFree(&ift); return(3);
    }
    if(verbose>0) {printf("added new key with given value\n"); fflush(stdout);}
  } else {
    if(verbose>3) printf("key_index := %d\n", li);
    if(verbose>0) printf("removing '%s','%s'\n", ift.item[li].key, ift.item[li].value);
    ret=iftDelete(&ift, li);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      iftFree(&ift); return(3);
    }
  }


  /*
   *  Write the modified IFT contents.
   */
  if(verbose>1) printf("writing modified IFT in %s\n", iftfile);
  fp=fopen(iftfile, "w"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    iftFree(&ift); return(11);
  }
  ret=iftWrite(&ift, fp, &status); fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(12);
  }
  iftFree(&ift);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
