/** @file rmcmnts.c
 *  @brief Remove comment lines from ASCII files.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Remove comment lines from data files.",
  " ",
  "Usage: @P [Options] datafile(s)",
  " ",
  "Options:",
  " -semicolon=<y|N>",
  "     Semicolon is used to mark comment line (y) or not (n, default).",
  " -sharp=<Y|n>",
  "     Sharp ('#') is used to mark comment line (y, default) or not (n).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: iftlist, taclist, iftdel, csv2ift, parformat",
  " ",
  "Keywords: TAC, CSV, IFT, software testing, reporting",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char *cptr, datfile[FILENAME_MAX];
  int   ffi=0, fileNr=0;
  int   cmt_sharp=1;
  int   cmt_semicolon=0;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  datfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "SEMICOLON=", 10)==0) {
      cptr+=10;
      if(strncasecmp(cptr, "YES", 1)==0) {cmt_semicolon=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0) {cmt_semicolon=0; continue;}
    } else if(strncasecmp(cptr, "SHARP=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "YES", 1)==0) {cmt_sharp=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0) {cmt_sharp=0; continue;}
    } 
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Check that we got the filenames */
  if(fileNr<1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  /* Check that at least some comments are to be removed */
  if(!cmt_semicolon && !cmt_sharp) {
    fprintf(stderr, "Warning: no comments were selected to be removed.\n");
    return(0);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("fileNr := %d\n", fileNr);
    printf("cmt_semicolon := %d\n", cmt_semicolon);
    printf("cmt_sharp := %d\n", cmt_sharp);
  }


  /*
   *  Process one file at a time
   */
  FILE *fp;
  size_t fsize;
  char *data, *data2, *line;
  int i, j, n;
  fileNr=0;
  for(ai=ffi; ai<argc; ai++) {

    strlcpy(datfile, argv[ai], FILENAME_MAX);
    if(verbose>0) printf("%s\n", datfile);

    /* Open the file */
    fp=fopen(datfile, "r");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file %s\n", datfile);
      return(2);
    }

    /* Get the size of the ASCII part of the file */
    fsize=asciiFileSize(fp, NULL);
    if(verbose>2) printf("  filesize := %u\n", (unsigned int)fsize);
    /* If ASCII part is too small, then lets consider that an error */
    if(fsize<1) {
      fprintf(stderr, "Warning: not processed %s\n", datfile);
      fclose(fp); 
      continue;
    }
    /* If ASCII part is too large, then lets consider that an error */
    if(fsize>5000000) {
      fprintf(stderr, "Warning: not processed %s\n", datfile);
      fclose(fp); 
      continue;
    }

    /* Read that to a string */
    data=asciiFileRead(fp, NULL, fsize+1); rewind(fp);
    if(data==NULL) {
      fprintf(stderr, "Error: cannot read file %s\n", datfile);
      fclose(fp); return(3);
    }
    if(verbose>10) printf("  ASCII file read\n");
    fclose(fp);

    /* Allocate memory for new data */
    data2=calloc(fsize+1, sizeof(char));
    if(data2==NULL) {
      fprintf(stderr, "Error: cannot allocate memory.\n");
      free(data); return(4);
    }

    /* Read one line at a time from the string */
    cptr=data; i=0; n=0;
    while((line=strTokenDup(cptr, "\n\r", &j))!=NULL) {
      /* If line starts with comment then jump over it */
      if(cmt_sharp && line[0]=='#') {free(line); cptr+=j; n++; continue;}
      if(cmt_semicolon && line[0]==';') {free(line); cptr+=j; n++; continue;}
      strcat(data2, line); strcat(data2, "\n");
      free(line); cptr+=j; i++;
    }
    free(data);
    if(verbose>0) printf("  %d line(s) removed.\n", n);
    if(verbose>1) printf("  %d line(s) retained.\n", i);
    /* If no changes done, then continue with next file */
    if(n==0) {
      free(data2); continue;
    }
    /* Otherwise overwrite the original file */ 
    fp=fopen(datfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot write file %s\n", datfile);
      free(data2); return(11);
    }
    n=fprintf(fp, "%s", data2);
    free(data2); fclose(fp);
    if(n<2) {
      fprintf(stderr, "Error: cannot write file %s\n", datfile);
      return(12);
    }
    fileNr++;

  } /* next file */

  if(verbose>0) {
    printf("comments removed from %d file(s).\n", fileNr);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
