/** @file bpr2cpr.c
 *  @brief Converts blood-to-plasma ratio curve to RBC-to-plasma ratio curve.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
#include "tpcli.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts blood-to-plasma (or optionally plasma-to-blood) ratio curve to",
  "RBC-to-plasma (blood cell-to-plasma) curve using haematocrit (HCT),",
  "based on equation",
  " ",
  "  Blood = HCT*RBC + (1-HCT)*Plasma ",
  " ",
  "HCT is normally between 0.40-0.51 in men and 0.36-0.47 in women.",
  " ",
  "Usage: @P [Options] bprfile HCT cprfile",
  " ",
  "Options:",
  " -pbr",
  "     Conversion is applied to plasma-to-blood ratio data.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: b2rbc, p2blood, b2plasma, taccalc, fit_bpr",
  " ",
  "Keywords: input, modelling, simulation, RBC, plasma, blood",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret;
  char  *cptr, bprfile[FILENAME_MAX], cprfile[FILENAME_MAX];
  TAC    tac;
  double HCT=-1.0;
  int    inmode=0; // Input is 0=BPR, or 1=PBR


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  bprfile[0]=cprfile[0]=(char)0;
  tacInit(&tac);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strcasecmp(cptr, "BPR")==0) {
      inmode=0; continue;
    } else if(strcasecmp(cptr, "PBR")==0) {
      inmode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!bprfile[0]) {
      strlcpy(bprfile, argv[ai], FILENAME_MAX); continue;
    } else if(HCT<0.0) {
      if(!atofCheck(argv[ai], &HCT)) {
        if(HCT>1.0) HCT/=100.0;
        if(HCT>0.0 && HCT<=1.0) continue;
      }
      fprintf(stderr, "Error: invalid HCT.\n"); 
      return(1);
    } else if(!cprfile[0]) {
      strlcpy(cprfile, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!cprfile[0]) { 
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("bprfile := %s\n", bprfile);
    printf("cprfile := %s\n", cprfile);
    printf("HCT := %g\n", HCT);
    printf("inmode := %d\n", inmode);
  }


  /*
   *  Read blood-to-plasma ratio
   */
  if(verbose) printf("reading %s\n", bprfile);
  ret=tacRead(&tac, bprfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
  }
  if(tac.tacNr>1) {
    fprintf(stderr, "Warning: only first ratio curve is used.\n");
    tac.tacNr=1;
  }

  /* Sort by sample times */
  tacSortByTime(&tac, NULL);


  /* Calculate RBC-to-plasma curve */
  for(int i=0; i<tac.sampleNr; i++) {
    if(isnan(tac.c[0].y[i])) continue;
    if(inmode==1) { // convert plasma/blood to blood/plasma
      if(tac.c[0].y[i]<1.0E-100) tac.c[0].y[i]=0.0;
      else tac.c[0].y[i]=1.0/tac.c[0].y[i];
    }
    tac.c[0].y[i]-=(1.0-HCT);
    tac.c[0].y[i]/=HCT;
  }

  /* Save RBC-to-plasma ratio */
  if(verbose>1) printf("writing %s\n", cprfile);
  FILE *fp; fp=fopen(cprfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", cprfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }
  if(verbose>=0) printf("RBC-to-plasma curve written in %s\n", cprfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
