/** @file bf_radiowater.c
 *  @brief BFM for radiowater.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "tpctacmod.h"
#include "tpccm.h"
/*****************************************************************************/
#include "tpcbfm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate set of basis functions for generic radiowater model.
 *  @todo Add tests.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int bfmRadiowater(
  /** Pointer to blood input TAC (not modified). If frame start and end times
      are available (->isframes!=0), then BTAC integral is calculated and 
      used as input. */
  TAC *input,
  /** Pointer to PET TTAC data, which must contain the sample (frame) times. */
  TAC *tissue,
  /** Nr of basis functions to calculate */
  int bfNr,
  /** Minimum of k2 (sec-1 or min-1, corresponding to TAC time units) */
  const double k2min,
  /** Maximum of k2 (sec-1 or min-1, corresponding to TAC time units) */
  const double k2max,
  /** Pointer to output TAC, to be allocated and filled with basis functions 
   *  in here. */
  TAC *bf,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(input==NULL || tissue==NULL || bf==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(input->sampleNr<1 || input->tacNr<1 || tissue->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(input->sampleNr<3 || bfNr<2) {
    if(verbose>1) printf("invalid sample or BF number\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_TOO_FEW);
    return TPCERROR_TOO_FEW;
  }
  if(k2min<1.0E-10 || k2min>=k2max) { // range cannot be computed otherwise
    if(verbose>1) printf("invalid k2 range\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  int ret;

  /* Check units */
  if(status==NULL || !status->forgiving) {
    if(!unitIsTime(input->tunit) || input->tunit!=tissue->tunit) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNKNOWN_UNIT);
      return TPCERROR_UNKNOWN_UNIT;
    }
  }

  /* Allocate meory for basis functions */
  if(verbose>1) printf("allocating memory for basis functions\n");
  tacFree(bf);
  ret=tacAllocate(bf, tissue->sampleNr, bfNr);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }
  /* Copy and set information fields */
  bf->tacNr=bfNr; bf->sampleNr=tissue->sampleNr;
  bf->format=TAC_FORMAT_TSV_UK;
  bf->isframe=tissue->isframe; tacXCopy(tissue, bf, 0, tissue->sampleNr-1);
  bf->tunit=tissue->tunit; bf->cunit=tissue->cunit;
  for(int bi=0; bi<bf->tacNr; bi++) sprintf(bf->c[bi].name, "B%5.5d", bi+1);
  /* Compute the range of k2 values to size fields */
  {
    if(verbose>1) printf("computing k2 values\n");
    double a, b, c;
    a=log10(k2min); b=log10(k2max); c=(b-a)/(double)(bfNr-1);
    for(int bi=0; bi<bf->tacNr; bi++) {
      bf->c[bi].size=pow(10.0, (double)bi*c+a);
    }
  }
  
  /* Allocate memory for simulated TAC */
  double *sim;
  sim=(double*)malloc(input->sampleNr*sizeof(double));
  if(sim==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    tacFree(bf); return TPCERROR_OUT_OF_MEMORY;
  }

  /* Simulate the basis functions at input time points, and interpolate to
     tissue sample times */
  double *cbi=NULL;
  if(input->isframe!=0) {
    // Time frames are available:
    // Integrate BTAC to frame middle times and use it as input */ 
    if(verbose>1) printf("using BTAC integral as simulation input\n");
    cbi=(double*)malloc(input->sampleNr*sizeof(double));
    if(cbi==NULL) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
      tacFree(bf); free(sim); return TPCERROR_OUT_OF_MEMORY;
    }
    ret=liIntegratePET(input->x1, input->x2, input->c[0].y, input->sampleNr,
                       cbi, NULL, verbose-10);
    if(ret) {
      if(verbose>1) printf("liIntegratePET() = %d\n", ret);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      tacFree(bf); free(sim); free(cbi); return TPCERROR_INVALID_VALUE;
    }
  }
  if(verbose>1) printf("computing basis functions\n");
  ret=0;
  for(int bi=0; bi<bf->tacNr && !ret; bi++) {
    if(input->isframe==0) { // time frames not available
      ret=simC1(input->x, input->c[0].y, input->sampleNr,
                1.0, bf->c[bi].size, sim);
    } else { // time frames are available
      ret=simC1_i(input->x, cbi, input->sampleNr,
                  1.0, bf->c[bi].size, sim);
    }
    if(ret) break;
    if(verbose>100) {
      printf("\nk2 := %g\n", bf->c[bi].size);
      printf("simulated TAC:\n");
      for(int i=0; i<input->sampleNr; i++)
        printf("  %12.6f  %12.3f\n", input->x[i], sim[i]);
    }
    /* interpolate to PET time frames */
    if(tissue->isframe)
      ret=liInterpolateForPET(input->x, sim, input->sampleNr, 
               tissue->x1, tissue->x2, bf->c[bi].y, NULL, NULL, bf->sampleNr, 
               4, 1, verbose-8);
    else
      ret=liInterpolate(input->x, sim, input->sampleNr, tissue->x,
               bf->c[bi].y, NULL, NULL, bf->sampleNr,
               4, 1, verbose-8);
    if(ret) break;
  } // next basis function
  free(sim); free(cbi);
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
