/** @file sim3cmp.c
 *  @brief Simulation of 3-tissue CM with compartments 2 and 3 in parallel.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpccm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using 1-3 tissue compartment model (2nd and 3rd compartments in parallel) 
    and plasma TAC, at plasma TAC times.
     
    @details
    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta, ctb 
    and/or ctc can be given; if compartmental TACs are not required, NULL pointer can be given
    instead.
  
    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
   
    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
    @sa simC3s, simC1, simC1DI, simC2, simC2l, simC3vp, simC3vs, simSamples,
     parExampleTTACs, parExamplePerfectBolus
 */
int simC3p(
  /** Array of time values. */
  double *t,
  /** Array of arterial activities. */
  double *ca,
  /** Number of values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double k4,
  /** Rate constant of the model. */
  const double k5,
  /** Rate constant of the model. */
  const double k6,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL. */
  double *ctc
) {
  int i;
  double dt2, r, s, u, v, w;
  double cai, ca_last, t_last;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Check parameters */
  if(k1<0.0) return 3;

  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      r=1.0+k4*dt2;
      s=1.0+k6*dt2;
      u=ct1i_last+dt2*ct1_last;
      v=ct2i_last+dt2*ct2_last;
      w=ct3i_last+dt2*ct3_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai - (k2 + (k3/r) + (k5/s))*u + (k4/r)*v + (k6/s)*w )
            / ( 1.0 + dt2*(k2 + (k3/r) + (k5/s)) );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*v) / r;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct1i - k6*w) / s;
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* copy values to argument arrays; set very small values to zero */
    ct[i]=ct1+ct2+ct3;
    if(cta!=NULL) cta[i]=ct1;
    if(ctb!=NULL) ctb[i]=ct2;
    if(ctc!=NULL) ctc[i]=ct3;
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using 1-3 tissue compartment model (2nd and 3rd compartments in parallel) 
    and plasma TAC, at plasma TAC times, considering also arterial and venous vasculature.

    @details
    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta, ctb, ctc, 
    ctab, and/or ctvb can be given; if compartmental TACs are not required, NULL pointer can be 
    given instead.

    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
    If blood flow is set to 0, function assumes that f>>k1, and Cvb=Cab.,

    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
    @sa simC3vs, simC3p, simC3s, simC3vpKLoss, simC4DIvp, simSamples, simDispersion, simDelay,
     parExampleTTACs, parExamplePerfectBolus
 */
int simC3vp(
  /** Array of time values. */
  double *t,
  /** Array of arterial plasma activities. */
  double *ca,
  /** Array of arterial blood activities. */
  double *cb,
  /** Number of values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double k4,
  /** Rate constant of the model. */
  const double k5,
  /** Rate constant of the model. */
  const double k6,
  /** Blood flow; if 0, function assumes that f>>k1, and Cvb=Cab. */
  const double f,
  /** Vascular volume fraction. */
  const double vb,
  /** Arterial fraction of vascular volume. */
  const double fa,
  /** Vascular volume modelling: 
      set to 0 to use Cpet = Vb*Cb + (1-Vb)*Ct,
      or set to 1 to use Cpet = Vb*Cb + Ct. */
  const int vvm,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *cpet,
  /** Pointer for 1st compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL. */
  double *ctc,
  /** Pointer for arterial TAC in tissue, or NULL. */
  double *ctab,
  /** Pointer for venous TAC in tissue, or NULL. */
  double *ctvb
) {
  int i;
  double dt2, r, s, u, v, w, va, vv;
  double cai, ca_last, t_last, dct, cvb;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(cpet==NULL) return 2;

  /* Check parameters */
  if(k1<0.0) return 3;
  if(vb<0.0 || vb>=1.0) return 4;
  if(fa<=0.0 || fa>1.0) return 5;
  va=fa*vb; vv=(1.0-fa)*vb;

  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      r=1.0+k4*dt2;
      s=1.0+k6*dt2;
      u=ct1i_last+dt2*ct1_last;
      v=ct2i_last+dt2*ct2_last;
      w=ct3i_last+dt2*ct3_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai - (k2 + (k3/r) + (k5/s))*u + (k4/r)*v + (k6/s)*w )
            / ( 1.0 + dt2*(k2 + (k3/r) + (k5/s)) );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*v) / r;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct1i - k6*w) / s;
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* Venous curve */
    if(f>0.) {dct = k1*ca[i] - k2*ct1; cvb = cb[i] - dct/f;} else cvb=cb[i];
    /* copy values to argument arrays */
    if(vvm==0) cpet[i]= va*cb[i] + vv*cvb + (1.0-vb)*(ct1+ct2+ct3);
    else cpet[i]= va*cb[i] + vv*cvb + (ct1+ct2+ct3);
    if(cta!=NULL) {cta[i]=ct1; if(vvm==0) cta[i]*=(1.0-vb);}
    if(ctb!=NULL) {ctb[i]=ct2; if(vvm==0) ctb[i]*=(1.0-vb);}
    if(ctc!=NULL) {ctc[i]=ct3; if(vvm==0) ctc[i]*=(1.0-vb);}
    if(ctab!=NULL) {ctab[i]=va*cb[i];}
    if(ctvb!=NULL) {ctvb[i]=vv*cvb;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
