/** @file simkloss.c
 *  @brief Simulation of compartment models with loss rate constant.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpccm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using 3 tissue compartmental model with two parallel compartments, and 
    plasma TAC, at plasma TAC sample times, considering also arterial and venous vasculature.
    The efflux from 3rd tissue compartment (C) goes directly to blood at rate kLoss.
 
    @details 
    Memory for cpet must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta, ctb,
    ctab, and/or ctvb can be given; if compartmental TACs are not required, NULL pointer can be 
    given instead.
 
    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
 
    If blood flow is set to 0, function assumes that f>>k1, and Cvb=Cab.,

    @sa simC2vl, simC2l, simC3vs, simC4DIvp, parExamplePerfectBolus

    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
 */
int simC3vpKLoss(
  /** Array of sample times. */
  double *t,
  /** Array of arterial plasma activities. */
  double *ca,
  /** Array of arterial blood activities. */
  double *cb,
  /** Number of sample values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double k4,
  /** Rate constant of the model. */
  const double k5,
  /** Rate constant of the model. */
  const double k6,
  /** Rate constant of the model. */
  const double kLoss,
  /** Blood flow; if 0, function assumes that f>>k1, and Cvb=Cab. */
  const double f,
  /** Vascular volume fraction. */
  const double vb,
  /** Arterial fraction of vascular volume. */
  const double fa,
  /** Vascular volume modelling: 
      set to 0 to use Cpet = Vb*Cb + (1-Vb)*Ct,
      or set to 1 to use Cpet = Vb*Cb + Ct. */
  const int vvm,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *cpet,
  /** Pointer for 1st tissue compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL. */
  double *ctc,
  /** Pointer for arterial TAC in tissue, or NULL. */
  double *ctab,
  /** Pointer for venous TAC in tissue, or NULL. */
  double *ctvb
) {
  int i;
  double dt2, b, c, d, w, z, u, va, vv;
  double cai, ca_last, t_last, dct, cvb;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;
 
 
  /* Check for data */
  if(nr<2) return 1;
  if(cpet==NULL) return 2;
 
  /* Check parameters */
  if(k1<0.0) return 3;
  if(vb<0.0 || vb>=1.0) return 4;
  if(fa<=0.0 || fa>1.0) return 5;
  va=fa*vb; vv=(1.0-fa)*vb;
 
  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      w = 1.0 + k4*dt2;
      z = 1.0 + dt2*(k6 + kLoss); 
      u = k2 + k3 + k5 - k3*k4*dt2/w - k5*k6*dt2/z;
      b = ct1i_last+dt2*ct1_last;
      c = ct2i_last+dt2*ct2_last;
      d = ct3i_last+dt2*ct3_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai - u*b + k4*c/w + k6*d/z ) / ( 1.0 + dt2*u);
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*c) / w;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct1i - (k6 + kLoss)*d) / z;
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* Venous curve */
    if(f>0.) {dct = k1*ca[i] - k2*ct1 - kLoss*ct3; cvb = cb[i] - dct/f;}
    else cvb=cb[i];
    /* copy values to argument arrays */
    if(vvm==0) cpet[i]= va*cb[i] + vv*cvb + (1.0-vb)*(ct1+ct2+ct3);
    else cpet[i]= va*cb[i] + vv*cvb + (ct1+ct2+ct3);
    if(cta!=NULL) {cta[i]=ct1; if(vvm==0) cta[i]*=(1.0-vb);}
    if(ctb!=NULL) {ctb[i]=ct2; if(vvm==0) ctb[i]*=(1.0-vb);}
    if(ctc!=NULL) {ctc[i]=ct3; if(vvm==0) ctc[i]*=(1.0-vb);}
    if(ctab!=NULL) {ctab[i]=va*cb[i];}
    if(ctvb!=NULL) {ctvb[i]=vv*cvb;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }
  
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using 2 tissue compartment model (in series) and plasma TAC, at plasma 
    TAC times. In contrary to the common model, kLoss represents a direct loss rate from
    the 2nd tissue compartment to venous plasma.

    @details
    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta and ctb 
    can be given; if compartmental TACs are not required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.

    @sa simC2, simC2vl, simC3vs, simC3vpKLoss

    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
 */
int simC2l(
  /** Array of time values. */
  double *t,
  /** Array of arterial activities. */
  double *ca,
  /** Number of values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double kLoss,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb
) {
  int i;
  double b, c, dt2;
  double cai, ca_last, t_last;
  double ct1, ct1_last, ct2, ct2_last;
  double ct1i, ct1i_last, ct2i, ct2i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Check actual parameter number */
  if(k1<0.0) return 3;

  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  cai=ca_last=0.0;
  ct1_last=ct2_last=ct1i_last=ct2i_last=ct1=ct2=ct1i=ct2i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* partial results */
      b=ct1i_last+dt2*ct1_last;
      c=ct2i_last+dt2*ct2_last;
      /* 1st tissue compartment and its integral */
      ct1 = (k1*cai - (k2+k3)*b) / (1.0 + (k2+k3)*dt2 );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - kLoss*c) / (1.0 + kLoss*dt2);
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
    }
    /* copy values to argument arrays */
    ct[i]=ct1+ct2;
    if(cta!=NULL) cta[i]=ct1;
    if(ctb!=NULL) ctb[i]=ct2;
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using 2 tissue compartment model and plasma TAC, at plasma TAC times,
    considering also arterial and venous vasculature.
    The efflux from 2nd tissue compartment (at rate kL) goes directly to blood.

    @details
    Memory for cpet must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta, ctb, 
    ctab, and/or ctvb can be given; if compartmental TACs are not required, NULL pointer can be
    given instead.

    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
    If blood flow is set to 0, function assumes that f>>k1, and Cvb=Cab.

    @sa simC2, simC2l, simC3vs, simC3vpKLoss, simC3DIvs

    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
 */
int simC2vl(
  /** Array of time values. */
  double *t,
  /** Array of arterial plasma activities. */
  double *ca,
  /** Array of arterial blood activities. */
  double *cb,
  /** Number of values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double kL,
  /** Blood flow; if 0, function assumes that f>>k1, and Cvb=Cab. */
  const double f,
  /** Vascular volume fraction. */
  const double vb,
  /** Arterial fraction of vascular volume. */
  const double fa,
  /** Vascular volume modelling: 
      set to 0 to use Cpet = Vb*Cb + (1-Vb)*Ct,
      or set to 1 to use Cpet = Vb*Cb + Ct. */
  const int vvm,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *cpet,
  /** Pointer for 1st compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb,
  /** Pointer for arterial TAC in tissue, or NULL. */
  double *ctab,
  /** Pointer for venous TAC in tissue, or NULL. */
  double *ctvb
) {
  int i;
  double dt2, b, c, va, vv;
  double cai, ca_last, t_last, dct, cvb;
  double ct1, ct1_last, ct2, ct2_last;
  double ct1i, ct1i_last, ct2i, ct2i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(cpet==NULL) return 2;

  /* Check parameters */
  if(k1<0.0) return 3;
  if(vb<0.0 || vb>=1.0) return 4;
  if(fa<=0.0 || fa>1.0) return 5;
  va=fa*vb; vv=(1.0-fa)*vb;

  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  cai=ca_last=0.0;
  ct1_last=ct2_last=ct1i_last=ct2i_last=ct1=ct2=ct1i=ct2i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      b=ct1i_last+dt2*ct1_last;
      c=ct2i_last+dt2*ct2_last;
      /* 1st tissue compartment and its integral */
      ct1 = (k1*cai - (k2+k3)*b) / (1.0 + (k2+k3)*dt2 );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - kL*c) / (1.0 + kL*dt2);
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
    }
    /* Venous curve */
    if(f>0.) {dct = k1*ca[i] - k2*ct1 - kL*ct2; cvb = cb[i] - dct/f;}
    else cvb=cb[i];

    /* copy values to argument arrays */
    if(vvm==0) cpet[i]= va*cb[i] + vv*cvb + (1.0-vb)*(ct1+ct2);
    else cpet[i]= va*cb[i] + vv*cvb + (ct1+ct2);
    if(cta!=NULL) {cta[i]=ct1; if(vvm==0) cta[i]*=(1.0-vb);}
    if(ctb!=NULL) {ctb[i]=ct2; if(vvm==0) ctb[i]*=(1.0-vb);}
    if(ctab!=NULL) {ctab[i]=va*cb[i];}
    if(ctvb!=NULL) {ctvb[i]=vv*cvb;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
