/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpccsv.h"
/*****************************************************************************/

/*****************************************************************************/
int test_csvList(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(csvList(NULL, NULL)==TPCERROR_OK) return(1);

  int ret;
  CSV csv1, csv2;
  FILE *fp=NULL;
  char fname[]="list.tsv";
  
  csvInit(&csv1); csvInit(&csv2);
  if(csvAllocate(&csv1, 30)!=TPCERROR_OK) return(100);
  csv1.col_nr=5; csv1.row_nr=6;
  csv1.separator=';';
  int i;
  i=0; csv1.c[i].row=1; csv1.c[i].col=0; csv1.c[i].content=strdup("cell21");
  i=1; csv1.c[i].row=2; csv1.c[i].col=1; csv1.c[i].content=strdup("cell32");
  i=2; csv1.c[i].row=3; csv1.c[i].col=2; csv1.c[i].content=strdup("cell43");
  i=3; csv1.c[i].row=4; csv1.c[i].col=3; csv1.c[i].content=strdup("cell54");
  csv1.nr=1+i;

  if(verbose>1) printf("listing test data in stdout\n");
  ret=csvList(&csv1, stdout);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); return(102);}

  if(verbose>1) {printf("open file for writing test data\n"); fflush(stdout);}
  fp=fopen(fname, "w"); if(fp==NULL) {csvFree(&csv1); return(111);}
  if(verbose>1) {printf("writing test data\n"); fflush(stdout);}
  ret=csvList(&csv1, fp); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); return(112);}
  if(verbose>1) {printf("reading test data\n"); fflush(stdout);}
  fp=fopen(fname, "r"); if(fp==NULL) {csvFree(&csv1); return(113);}
  ret=csvRead(&csv2, fp, status); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); csvFree(&csv2); return(114);}
  if(verbose>3) ret=csvWrite(&csv2, 0, stdout, NULL);
  if(csv2.nr!=12) {csvFree(&csv1); csvFree(&csv2); return(115);}
  if(strcmp(csv2.c[0].content, "2") || strcmp(csv2.c[1].content, "1") || strcmp(csv2.c[2].content, "cell21")) {
    csvFree(&csv1); csvFree(&csv2); return(116);}
  if(strcmp(csv2.c[9].content, "5") || strcmp(csv2.c[10].content, "4") || strcmp(csv2.c[11].content, "cell54")) {
    csvFree(&csv1); csvFree(&csv2); return(117);}
  if(verbose>2) printf("  ->ok\n");

  csvFree(&csv1); csvFree(&csv2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_csvWrite(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  /* Mainly tested in test_csvRead() */

  if(csvWrite(NULL, 0, NULL, NULL)==TPCERROR_OK) return(1);

  int ret;
  CSV csv;
  FILE *fp=NULL;
  char fname[]="test.tsv";
  
  csvInit(&csv);
  csv.separator='\t';
  ret=csvPutString(&csv, "C1R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C2R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C3R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C4R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C1R2", 1);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C2R2", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C3R2", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C4R2", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C1R3", 1);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C2R3", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C3R3", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv, "C4R3", 0);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 2;}

  if(verbose>1) printf("\nwriting to fp==NULL\n");
  ret=csvWrite(&csv, 0, fp, NULL);
  if(ret==TPCERROR_OK) {csvFree(&csv); return 3;}
  
  if(verbose>1) {
    if(verbose>1) printf("\nwriting to stdout\n");
    ret=csvWrite(&csv, 1, stdout, NULL);
    if(ret!=TPCERROR_OK) {csvFree(&csv); return 10;}
  }

  if(verbose>1) printf("\nwriting test data\n");
  fp=fopen(fname, "w"); if(fp==NULL) {csvFree(&csv); return 20;}
  ret=csvWrite(&csv, 0, fp, NULL); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 21;}

  if(verbose>1) printf("\nwriting test data\n");
  fp=fopen(fname, "w"); if(fp==NULL) {csvFree(&csv); return 30;}
  ret=csvWrite(&csv, 0, fp, NULL); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 31;}

  csvFree(&csv);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_csvRead(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  CSV csv1, csv2;
  char fname[]="test.csv";
  FILE *fp=NULL;
  TPCSTATUS statusl;

  csvInit(&csv1); csvInit(&csv2); statusInit(&statusl);
  statusl.verbose=verbose-3;
  
  /* Try to read from NULL */
  if(verbose>1) printf("reading from NULL should fail\n");
  ret=csvRead(&csv1, fp, &statusl);
  if(ret==TPCERROR_OK) {return 1;}

  /* Create test contents */
  if(verbose>1) printf("creating test data\n");
  csv1.separator=',';
  ret=csvPutString(&csv1, "C1R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C2R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C3R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C4R1", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C1R2", 1);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C2R2", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C3R2", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C4R2", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C1R3", 1);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C2R3", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C3R3", 0);
  if(ret==TPCERROR_OK) ret=csvPutString(&csv1, "C4R3", 0);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); return 2;}
  
  /* Write it to file */
  if(verbose>1) printf("writing test data\n");
  fp=fopen(fname, "w"); if(fp==NULL) {csvFree(&csv1); return 2;}
  fprintf(fp, "# Test CSV file\n");
  ret=csvWrite(&csv1, 0, fp, &statusl); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); return 2;}
   
  /* Read it into another CSV struct */
  if(verbose>1) printf("reading test data\n");
  fp=fopen(fname, "r"); if(fp==NULL) {csvFree(&csv1); return 3;}
  ret=csvRead(&csv2, fp, &statusl); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); csvFree(&csv2); return 3;}
  if(verbose>3) ret=csvWrite(&csv2, 0, stdout, NULL);

  /* Compare CSV structures */ 
  if(verbose>1) printf("comparing test data\n");
  if(csv1.nr!=csv2.nr) {csvFree(&csv1); csvFree(&csv2); return 4;}
  for(int i=0; i<csv1.nr; i++) {
    if(strcmp(csv1.c[i].content, csv2.c[i].content)!=0)
      {csvFree(&csv1); csvFree(&csv2); return 5;}
    if(csv1.c[i].row!=csv2.c[i].row || csv1.c[i].col!=csv2.c[i].col)
      {csvFree(&csv1); csvFree(&csv2); return 6;}
  }
  if(csv1.separator!=csv2.separator)
    {csvFree(&csv1); csvFree(&csv2); return 7;}
  if(verbose>2) printf("  ->ok\n\n");

  csvFree(&csv2);

  /*
   *  Repeat with space as field delimiter
   */
  csv1.separator=' ';
  char fname2[]="test2.csv";

  /* Write it to file */
  if(verbose>1) printf("open file for writing test data\n");
  fp=fopen(fname2, "w"); if(fp==NULL) {csvFree(&csv1); return 12;}
  if(verbose>1) printf("writing test data\n");
  ret=csvWrite(&csv1, 0, fp, &statusl); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); return 12;}
   
  /* Read it into another CSV struct */
  if(verbose>1) printf("reading test data\n");
  fp=fopen(fname2, "r"); if(fp==NULL) {csvFree(&csv1); return 13;}
  ret=csvRead(&csv2, fp, &statusl); fclose(fp);
  if(ret!=TPCERROR_OK) {csvFree(&csv1); csvFree(&csv2); return 13;}
  if(verbose>3) ret=csvWrite(&csv2, 0, stdout, NULL);

  /* Compare CSV structures */ 
  if(verbose>1) printf("comparing test data\n");
  //printf("%d vs %d\n", csv1.nr, csv2.nr);
  if(csv1.nr!=csv2.nr) {csvFree(&csv1); csvFree(&csv2); return 14;}
  for(int i=0; i<csv1.nr; i++) {
    if(strcmp(csv1.c[i].content, csv2.c[i].content)!=0)
      {csvFree(&csv1); csvFree(&csv2); return 15;}
    if(csv1.c[i].row!=csv2.c[i].row || csv1.c[i].col!=csv2.c[i].col)
      {csvFree(&csv1); csvFree(&csv2); return 16;}
  }
  if(csv2.separator!=' ') {csvFree(&csv1); csvFree(&csv2); return 17;}
  if(verbose>2) printf("  ->ok\n");

  csvFree(&csv1); csvFree(&csv2);

  /*
   *  Regularized writing
   */
  {
    if(verbose>1) printf("\nTesting regularized writing\n");
    char fname3[]="test3a.csv";
    if(csvAllocate(&csv1, 30)!=TPCERROR_OK) return(100);
    csv1.col_nr=5; csv1.row_nr=6;
    csv1.separator=',';
    int i;
    i=0; csv1.c[i].row=1; csv1.c[i].col=0; csv1.c[i].content=strdup("cell21");
    i=1; csv1.c[i].row=2; csv1.c[i].col=1; csv1.c[i].content=strdup("cell32");
    i=2; csv1.c[i].row=3; csv1.c[i].col=2; csv1.c[i].content=strdup("cell43");
    i=3; csv1.c[i].row=4; csv1.c[i].col=3; csv1.c[i].content=strdup("cell54");
    csv1.nr=1+i;
    if(verbose>1) printf("open file for writing test data\n");
    fp=fopen(fname3, "w"); if(fp==NULL) {csvFree(&csv1); return(101);}
    if(verbose>1) printf("writing test data\n");
    ret=csvWrite(&csv1, 1, fp, &statusl); fclose(fp);
    if(ret!=TPCERROR_OK) {csvFree(&csv1); return(102);}
    if(verbose>1) printf("reading test data\n");
    fp=fopen(fname3, "r"); if(fp==NULL) {csvFree(&csv1); return(103);}
    ret=csvRead(&csv2, fp, &statusl); fclose(fp);
    if(ret!=TPCERROR_OK) {csvFree(&csv1); csvFree(&csv2); return(104);}
    if(verbose>3) ret=csvWrite(&csv2, 1, stdout, NULL);
    if(csv2.nr!=30) {csvFree(&csv1); csvFree(&csv2); return(105);}
    if(strcmp(csvCell(&csv1, 4, 3), csvCell(&csv2, 4, 3))) {csvFree(&csv1); csvFree(&csv2); return(106);}
    if(verbose>2) printf("  ->ok\n");
    csvFree(&csv2);

    if(verbose>1) printf("\nTesting non-regularized writing with the same data\n");
    char fname4[]="test3b.csv";
    if(verbose>1) printf("open file for writing test data\n");
    fp=fopen(fname4, "w"); if(fp==NULL) {csvFree(&csv1); return(111);}
    if(verbose>1) printf("writing test data\n");
    ret=csvWrite(&csv1, 0, fp, &statusl); fclose(fp);
    if(ret!=TPCERROR_OK) {csvFree(&csv1); return(112);}
    if(verbose>1) printf("reading test data\n");
    fp=fopen(fname4, "r"); if(fp==NULL) {csvFree(&csv1); return(113);}
    ret=csvRead(&csv2, fp, &statusl); fclose(fp);
    if(ret!=TPCERROR_OK) {csvFree(&csv1); csvFree(&csv2); return(114);}
    if(verbose>3) ret=csvWrite(&csv2, 1, stdout, NULL);
    if(csv2.nr!=10) {csvFree(&csv1); csvFree(&csv2); return(115);}
    if(strcmp(csvCell(&csv1, 4, 3), csvCell(&csv2, 3, 3))) {csvFree(&csv1); csvFree(&csv2); return(116);}
    if(verbose>2) printf("  ->ok\n");

    csvFree(&csv1); csvFree(&csv2); 
  }

  statusFree(&statusl);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_csvPutLine(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  char str[256];
  int ret, n, i;
  CSV csv;
  TPCSTATUS statusl;

  csvInit(&csv);
  statusInit(&statusl);
  statusl.verbose=verbose-3;

  /* Empty */
  strcpy(str, ""); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>2) printf(" -> ret=%d\n", ret);
  if(ret==TPCERROR_OK) {csvFree(&csv); return 1;}
  csvFree(&csv);  

  /* Easy */
  strcpy(str, "one,two,three"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 2;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=3; if(csv.nr!=n) {csvFree(&csv); return 2;}
  if(strcmp(csv.c[0].content, "one")!=0) {csvFree(&csv); return 2;}
  if(strcmp(csv.c[1].content, "two")!=0) {csvFree(&csv); return 2;}
  if(strcmp(csv.c[2].content, "three")!=0) {csvFree(&csv); return 2;}
  csvFree(&csv);  

  /* Starting with delimiter */
  strcpy(str, ",One,Two,Three"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 3;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=4; if(csv.nr!=n) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[0].content, "")!=0) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[1].content, "One")!=0) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[2].content, "Two")!=0) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[3].content, "Three")!=0) {csvFree(&csv); return 3;}
  csvFree(&csv);  

  /* Ending with delimiter */
  strcpy(str, "1,2,3,"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 4;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=4; if(csv.nr!=n) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[0].content, "1")!=0) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[1].content, "2")!=0) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[2].content, "3")!=0) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[3].content, "")!=0) {csvFree(&csv); return 4;}
  csvFree(&csv);  

  /* Starting and ending with delimiter */
  strcpy(str, ",a,b,c,"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 5;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=5; if(csv.nr!=n) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[0].content, "")!=0) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[1].content, "a")!=0) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[2].content, "b")!=0) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[3].content, "c")!=0) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[4].content, "")!=0) {csvFree(&csv); return 5;}
  csvFree(&csv);

  /* Just a delimiter */
  strcpy(str, ";"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=';';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 6;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=2; if(csv.nr!=n) {csvFree(&csv); return 6;}
  for(i=0; i<n; i++)
    if(strcmp(csv.c[i].content, "")!=0) {csvFree(&csv); return 6;}
  csvFree(&csv);

  /* Just delimiters */
  strcpy(str, ";;;;;;;;"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=';';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 7;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=9; if(csv.nr!=n) {csvFree(&csv); return 7;}
  for(i=0; i<n; i++)
    if(strcmp(csv.c[i].content, "")!=0) {csvFree(&csv); return 7;}
  csvFree(&csv);  

  /* All inside quotes */
  strcpy(str, "\";;;;;;;;\""); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=';';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 11;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=1; if(csv.nr!=n) {csvFree(&csv); return 11;}
  if(strcmp(csv.c[0].content, "\";;;;;;;;\"")!=0) {csvFree(&csv); return 11;}
  csvFree(&csv);  

  /* Quote */
  strcpy(str, "'"); if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=';';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 12;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=1; if(csv.nr!=n) {csvFree(&csv); return 12;}
  if(strcmp(csv.c[0].content, "'")!=0) {csvFree(&csv); return 12;}
  csvFree(&csv);  

  /* Normal quotes */
  strcpy(str, "\"Time[min]\",\"roi 1\",\"roi 2\",\"roi 3\"");
  if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 13;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=4; if(csv.nr!=n) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[0].content, "\"Time[min]\"")!=0) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[1].content, "\"roi 1\"")!=0) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[2].content, "\"roi 2\"")!=0) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[3].content, "\"roi 3\"")!=0) {csvFree(&csv); return 13;}
  csvFree(&csv);  

  /* Normal quotes with empty fields */
  strcpy(str, ";\"Time[min]\";\"roi 1\";;\"roi 3\";");
  if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=';';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 14;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=6; if(csv.nr!=n) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[0].content, "")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[1].content, "\"Time[min]\"")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[2].content, "\"roi 1\"")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[3].content, "")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[4].content, "\"roi 3\"")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[5].content, "")!=0) {csvFree(&csv); return 14;}
  csvFree(&csv);  

  /* Delimiter inside quotes */
  strcpy(str, ",1,2,'3,0',4,");
  if(verbose>1) printf("\nstr := '%s'\n", str);
  csv.separator=',';
  ret=csvPutLine(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 15;}
  csv.separator=';'; if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=6; if(csv.nr!=n) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[0].content, "")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[1].content, "1")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[2].content, "2")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[3].content, "'3,0'")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[4].content, "4")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[5].content, "")!=0) {csvFree(&csv); return 15;}
  csvFree(&csv);  
  
  statusFree(&statusl);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_csvPutLineWithSpaces(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  char str[256];
  int ret, n;
  CSV csv;
  TPCSTATUS statusl;

  csvInit(&csv);
  csv.separator=','; // should work despite this
  statusInit(&statusl);
  statusl.verbose=verbose-3;

  /* Empty */
  strcpy(str, ""); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl);
  if(verbose>2) printf(" -> ret=%d\n", ret);
  if(ret==TPCERROR_OK) {csvFree(&csv); return 1;}
  csvFree(&csv);  

  /* Easy */
  strcpy(str, "one two three"); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl);
  if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 2;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=3; if(csv.nr!=n) {csvFree(&csv); return 2;}
  if(strcmp(csv.c[0].content, "one")!=0) {csvFree(&csv); return 2;}
  if(strcmp(csv.c[1].content, "two")!=0) {csvFree(&csv); return 2;}
  if(strcmp(csv.c[2].content, "three")!=0) {csvFree(&csv); return 2;}
  csvFree(&csv);  

  /* Starting with delimiter */
  strcpy(str, " One  Two  Three"); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl);
  if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 3;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=3; if(csv.nr!=n) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[0].content, "One")!=0) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[1].content, "Two")!=0) {csvFree(&csv); return 3;}
  if(strcmp(csv.c[2].content, "Three")!=0) {csvFree(&csv); return 3;}
  csvFree(&csv);  

  /* Ending with delimiter */
  strcpy(str, "1   2 3    "); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl);
  if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 4;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=3; if(csv.nr!=n) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[0].content, "1")!=0) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[1].content, "2")!=0) {csvFree(&csv); return 4;}
  if(strcmp(csv.c[2].content, "3")!=0) {csvFree(&csv); return 4;}
  csvFree(&csv);  

  /* Starting and ending with delimiter */
  strcpy(str, "  a  b     c "); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl);
  if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 5;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=3; if(csv.nr!=n) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[0].content, "a")!=0) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[1].content, "b")!=0) {csvFree(&csv); return 5;}
  if(strcmp(csv.c[2].content, "c")!=0) {csvFree(&csv); return 5;}
  csvFree(&csv);

  /* Just a delimiter */
  strcpy(str, " "); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl);
  if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 6;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=0; if(csv.nr!=n) {csvFree(&csv); return 6;}
//  for(i=0; i<n; i++)
//    if(strcmp(csv.c[i].content, "")!=0) {csvFree(&csv); return 6;}
  csvFree(&csv);

  /* Just delimiters */
  strcpy(str, "         "); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 7;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=0; if(csv.nr!=n) {csvFree(&csv); return 7;}
//  for(i=0; i<n; i++)
//    if(strcmp(csv.c[i].content, "")!=0) {csvFree(&csv); return 7;}
  csvFree(&csv);

  /* All inside quotes */
  strcpy(str, "\"       \""); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 11;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=1; if(csv.nr!=n) {csvFree(&csv); return 11;}
  if(strcmp(csv.c[0].content, "\"       \"")!=0) {csvFree(&csv); return 11;}
  csvFree(&csv);

  /* Quote */
  strcpy(str, "'"); if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 12;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=1; if(csv.nr!=n) {csvFree(&csv); return 12;}
  if(strcmp(csv.c[0].content, "'")!=0) {csvFree(&csv); return 12;}
  csvFree(&csv);

  /* Normal quotes */
  strcpy(str, "\"Time[min]\" \"roi 1\"  \"roi 2\"  \"roi 3\"");
  if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 13;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=4; if(csv.nr!=n) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[0].content, "\"Time[min]\"")!=0) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[1].content, "\"roi 1\"")!=0) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[2].content, "\"roi 2\"")!=0) {csvFree(&csv); return 13;}
  if(strcmp(csv.c[3].content, "\"roi 3\"")!=0) {csvFree(&csv); return 13;}
  csvFree(&csv);

  /* Normal quotes with delimiters before and after */
  strcpy(str, "     \"Time[min]\"     \"roi 1\"    \"roi 3\"   ");
  if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 14;}
  if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=3; if(csv.nr!=n) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[0].content, "\"Time[min]\"")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[1].content, "\"roi 1\"")!=0) {csvFree(&csv); return 14;}
  if(strcmp(csv.c[2].content, "\"roi 3\"")!=0) {csvFree(&csv); return 14;}
  csvFree(&csv);

  /* Delimiter inside quotes */
  strcpy(str, " 1 2 '3,0' 4 ");
  if(verbose>1) printf("\nstr := '%s'\n", str);
  ret=csvPutLineWithSpaces(&csv, str, &statusl); if(verbose>3) printf(" -> ret=%d\n", ret);
  if(ret!=TPCERROR_OK) {csvFree(&csv); return 15;}
  csv.separator=';'; if(verbose>2) csvWrite(&csv, 0, stdout, &statusl);
  n=4; if(csv.nr!=n) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[0].content, "1")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[1].content, "2")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[2].content, "'3,0'")!=0) {csvFree(&csv); return 15;}
  if(strcmp(csv.c[3].content, "4")!=0) {csvFree(&csv); return 15;}
  csvFree(&csv);
  
  statusFree(&statusl);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_csvCleanSpaces(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  CSV csv;
  csvInit(&csv);

  csvPutString(&csv, "ok", 0);
  csvPutString(&csv, " with space ", 0);
  csvCleanSpaces(&csv);
  if(csv.nr!=2 || strcmp(csv.c[0].content, "ok") || strcmp(csv.c[1].content, "with space")) {
    csvFree(&csv); return 1;
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
