/** @file dcmdata.c
    @brief Processing DICOM data structs.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the DCMFILE struct before any use. 
    @sa dcmfileFree
 */
void dcmfileInit(
  /** Pointer to DCMFILE. */
  DCMFILE *d
) {
  if(d==NULL) return;
  d->filename[0]=(char)0;
  d->fp=(FILE*)NULL;
  d->truid=DCM_TRUID_UNKNOWN;
  d->item=(DCMITEM*)NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Recursively free memory allocated for DCMITEM items and their children items.
    @sa dcmfileFree
    @author Vesa Oikonen
 */
void dcmitemFree(
  /** Pointer to DCMITEM. */
  DCMITEM *d
) {
  if(d==NULL) return;
  /* find the last item in the list */
  DCMITEM *ip=d; while(ip->next_item!=NULL) ip=ip->next_item;
  while(ip!=NULL) {
    /* Free items child and their children */
    if(ip->child_item!=NULL) dcmitemFree(ip->child_item);
    /* Free this item and move to previous item */
    if(ip->prev_item!=NULL) {
      ip=ip->prev_item; 
      free(ip->next_item->rd); free(ip->next_item); 
      ip->next_item=NULL;
    } else {
      free(ip->rd); free(ip); ip=NULL;
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for DCMFILE data. All contents are destroyed.
    @pre Before first use initialize the struct with dcmfileInit().
    @sa dcmfileInit
    @author Vesa Oikonen
 */
void dcmfileFree(
  /** Pointer to DCMFILE. */
  DCMFILE *d
) {
  if(d==NULL) return;
  dcmitemFree(d->item);
  dcmfileInit(d);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the maximum depth of DCMITEM tree.
    @return Returns the number of levels _under_ specified item, not including
     given item itself.
    @sa dcmitemParentNr, dcmfileMaxDepth
 */
unsigned short int dcmitemMaxDepth(
  /** Pointer to DCMITEM item. */
  DCMITEM *d
) {
  if(d==NULL || d->child_item==NULL) return(0);
  unsigned short int m=0, n=0;
  DCMITEM *cd=d->child_item;
  /* go through all children */
  while(cd!=NULL) {
    n=dcmitemMaxDepth(cd); if(n>m) m=n;
    cd=cd->next_item;
  }
  return(m+1);
}
/** Get the maximum depth of DCMFILE items tree.
    @return Returns the number of item levels under specified DCMFILE, or
     zero in case of an error or if there are no items.
    @sa dcmitemParentNr, dcmitemMaxDepth
 */
unsigned short int dcmfileMaxDepth(
  /** Pointer to DCMFILE item. */
  DCMFILE *df
) {
  if(df==NULL || df->item==NULL) return(0);
  unsigned short int m=0, n=0;
  DCMITEM *sd=df->item;
  while(sd!=NULL) { // go through all sisters
    n=dcmitemMaxDepth(sd); if(n>m) m=n;
    sd=sd->next_item;
  }
  return(m+1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check how deep in DCMITEM tree this item is.
    @return Returns the number of parents this item has.
    @sa dcmitemMaxDepth, dcmfileMaxDepth
 */
unsigned short int dcmitemParentNr(
  /** Pointer to DCMITEM. */
  DCMITEM *d
) {
  if(d==NULL) return(0);
  unsigned short int n=0;
  DCMITEM *pd=d->parent_item;
  while(pd!=NULL) {n++; pd=pd->parent_item;}
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Pre-process the DICOM element value into format suitable for printing.
    @note Use only for printing information for the user.
    @return Returns pointer to locally allocated null-terminated string.
    @post Free the pointer after use.
    @sa dcmitemGetInt, dcmitemGetReal, dcmFindTag
 */ 
char *dcmValueString(
  /** Pointer to item containing the value to print. */
  DCMITEM *d
) {
  if(d==NULL) return((char*)NULL);

  /* For sequence, return string 'na' */
  if(d->vr==DCM_VR_SQ) {
    char *s=malloc(3); strcpy(s, "na"); // do not write char *s="na";
    return(s);
  }

  /* If there is no value, then return string 'empty', or
     'na', if value just was not stored (pixel data) */
  if(d->vl==0) {
    char *s=malloc(6); strcpy(s, "empty"); 
    return(s);
  } else if(d->rd==NULL) {
    char *s=malloc(3); strcpy(s, "na"); 
    return(s);
  }

  unsigned int len;
  if(d->vl==0xFFFFFFFF) len=(unsigned int)dcmVRVLength(d->vr); else len=d->vl;

  /* String values */
  if(d->vr==DCM_VR_CS || d->vr==DCM_VR_DS || d->vr==DCM_VR_IS || 
     d->vr==DCM_VR_LO || d->vr==DCM_VR_LT || d->vr==DCM_VR_PN || 
     d->vr==DCM_VR_SH || d->vr==DCM_VR_ST)
  {
    char *s=malloc(len+1);
    memcpy(s, d->rd, len); s[len]=(char)0;
    return(s);
  }

  /* More string values */
  if(d->vr==DCM_VR_AS || d->vr==DCM_VR_PN || 
     d->vr==DCM_VR_DA || d->vr==DCM_VR_DT || d->vr==DCM_VR_TM || 
     d->vr==DCM_VR_UT || d->vr==DCM_VR_AE || 
     d->vr==DCM_VR_UI || d->vr==DCM_VR_UR)
  {
    char *s=malloc(len+1);
    memcpy(s, d->rd, len); s[len]=(char)0;
    return(s);
  }

  /** Attribute tag */
  if(d->vr==DCM_VR_AT) {
    DCMTAG tag;
    memcpy(&tag.group, d->rd, 2);
    memcpy(&tag.element, d->rd+2, 2);
    if(!endianLittle()) {
      swap(&tag.group, &tag.group, 2);
      swap(&tag.element, &tag.element, 2);
    }
    char *s=malloc(14);
    sprintf(s, "0x%04x,0x%04x", tag.group, tag.element);
    return(s);
  }

  /** Float */
  if(d->vr==DCM_VR_FL) {
    float f;
    memcpy(&f, d->rd, 4);
    if(!endianLittle()) swap(&f, &f, 4);
    char *s=malloc(16);
    sprintf(s, "%g", f);
    return(s);
  }

  /** Double */
  if(d->vr==DCM_VR_FD) {
    char *s=malloc(32);
    /*if(d->vl==2) {
      printf("\nFD VL=%d\n", d->vl);
      float f;
      memcpy(&f, d->rd+4, 4);
      if(!endianLittle()) swap(&f, &f, 4);
      sprintf(s, "%g", f);
    } else {
      double f;
      memcpy(&f, d->rd, 8);
      if(!endianLittle()) swap(&f, &f, 8);
      sprintf(s, "%g", f);
    }
    */
    double f;
    memcpy(&f, d->rd, 8);
    if(!endianLittle()) swap(&f, &f, 8);
    sprintf(s, "%g", f);
    return(s);
  }

  /** Unsigned 32-bit int */
  if(d->vr==DCM_VR_UL) {
    unsigned int i;
    memcpy(&i, d->rd, 4);
    if(!endianLittle()) swap(&i, &i, 4);
    char *s=malloc(16);
    sprintf(s, "%u", i);
    return(s);
  }

  /** Unsigned 16-bit int */
  if(d->vr==DCM_VR_US) {
    unsigned short int i;
    memcpy(&i, d->rd, 2);
    if(!endianLittle()) swap(&i, &i, 2);
    char *s=malloc(8);
    sprintf(s, "%u", i);
    return(s);
  }

  /** Signed 32-bit int */
  if(d->vr==DCM_VR_SL) {
    int i;
    memcpy(&i, d->rd, 4);
    if(!endianLittle()) swap(&i, &i, 4);
    char *s=malloc(16);
    sprintf(s, "%d", i);
    return(s);
  }

  /** Signed 16-bit int */
  if(d->vr==DCM_VR_SS) {
    short int i;
    memcpy(&i, d->rd, 2);
    if(!endianLittle()) swap(&i, &i, 2);
    char *s=malloc(8);
    sprintf(s, "%d", i);
    return(s);
  }

/* Not (yet) printed:
  DCM_VR_OB,        ///< DICOM other byte string, even bytes, endian insensitive.
  DCM_VR_OD,        ///< DICOM other double (64-bit) stream, endian sensitive.
  DCM_VR_OF,        ///< DICOM other float (32-bit) stream, endian sensitive.
  DCM_VR_OL,        ///< DICOM other long (32-bit) stream, endian sensitive.
  DCM_VR_OW,        ///< DICOM other word (16-bit) stream, even bytes, endian sensitive.
  DCM_VR_UC,        ///< DICOM unlimited characters.
  DCM_VR_UN,        ///< DICOM unknown, any valid length of another VR.
  DCM_VR_INVALID    ///< Invalid DICOM value representation.
*/
  char *s=malloc(3); strcpy(s, "na"); // do not write char *s="na";
  return(s);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read integer value from given DICOM item.
    @details VR must be either UL, US, SL, SS, or IS; otherwise 0 is returned.
    @return Returns the value as long int, in order to cope with originally unsigned integers.
    @sa dcmitemGetInt, dcmValueString, dcmFindTag
 */
long int dcmitemGetInt(
  /** Pointer to item. */ 
  DCMITEM *d
) {
  if(d==NULL || d->rd==NULL) return(0);
  long int li=0;
  if(d->vr==DCM_VR_UL) { // unsigned 32-bit int
    unsigned int i;
    memcpy(&i, d->rd, 4); if(!endianLittle()) swap(&i, &i, 4);
    li=(long int)i;
  } else if(d->vr==DCM_VR_US) { // unsigned 16-bit int
    unsigned short int i;
    memcpy(&i, d->rd, 2); if(!endianLittle()) swap(&i, &i, 2);
    li=(long int)i;
  } else if(d->vr==DCM_VR_SL) { // signed 32-bit int
    int i;
    memcpy(&i, d->rd, 4); if(!endianLittle()) swap(&i, &i, 4);
    li=(long int)i;
  } else if(d->vr==DCM_VR_SS) { // signed 16-bit int
    short int i;
    memcpy(&i, d->rd, 2); if(!endianLittle()) swap(&i, &i, 2);
    li=(long int)i;
  } else if(d->vr==DCM_VR_IS) { // integer string
    li=atol(d->rd);
  }
  return(li);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read floating point value from given DICOM item.
    @details VR must be either FL, FD, DS, UL, US, SL, SS, or IS; 
     otherwise 0 is returned.
    @return Returns the value as double.
    @sa dcmitemGetInt, dcmValueString, dcmFindTag
 */
double dcmitemGetReal(
  /** Pointer to item. */ 
  DCMITEM *d
) {
  if(d==NULL || d->rd==NULL) return(0);
  double r=0.0;
  if(d->vr==DCM_VR_FL) { // 32-bit float
    float f;
    memcpy(&f, d->rd, 4); if(!endianLittle()) swap(&f, &f, 4);
    r=(double)f;
  } else if(d->vr==DCM_VR_FD) { // 64-bit double
    double f;
    memcpy(&f, d->rd, 8); if(!endianLittle()) swap(&f, &f, 8);
    r=f;
  } else if(d->vr==DCM_VR_DS) { // decimal string
    r=atof(d->rd);
  } else if(d->vr==DCM_VR_UL) { // unsigned 32-bit int
    unsigned int i;
    memcpy(&i, d->rd, 4); if(!endianLittle()) swap(&i, &i, 4);
    r=(double)i;
  } else if(d->vr==DCM_VR_US) { // unsigned 16-bit int
    unsigned short int i;
    memcpy(&i, d->rd, 2); if(!endianLittle()) swap(&i, &i, 2);
    r=(double)i;
  } else if(d->vr==DCM_VR_SL) { // signed 32-bit int
    int i;
    memcpy(&i, d->rd, 4); if(!endianLittle()) swap(&i, &i, 4);
    r=(double)i;
  } else if(d->vr==DCM_VR_SS) { // signed 16-bit int
    short int i;
    memcpy(&i, d->rd, 2); if(!endianLittle()) swap(&i, &i, 2);
    r=(double)i;
  } else if(d->vr==DCM_VR_IS) { // integer string
    r=(double)atol(d->rd);
  }
  return(r);
}
/*****************************************************************************/

/*****************************************************************************/
/** Search for specified tag in DCMITEM data tree, from left to right, down and up.

    @return Returns pointer to next item with the tag, or NULL if not found.
 */
DCMITEM *dcmFindTag(
  /** Pointer to current DICOM item. */
  DCMITEM *d,
  /** Omit this item from the search. */
  const short int omit,
  /** Pointer to the DICOM tag that is searched for. */
  DCMTAG *tag,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(d==NULL || tag==NULL) return(NULL);
  if(verbose>0) printf("%s((%04X,%04X),%d)\n", __func__, tag->group, tag->element, omit);
  DCMITEM *iptr;
  if(omit==0) iptr=d; else iptr=d->next_item;
  while(iptr!=NULL) {
    if(verbose>2)
      printf(" checking tag(%04X,%04X)...\n", iptr->tag.group, iptr->tag.element);
    if(iptr->tag.group==tag->group && iptr->tag.element==tag->element) {
      if(verbose>2) printf("  found!\n");
      break;
    }
    /* Check if this item has children */
    if(iptr->child_item!=NULL) {
      if(verbose>2) printf("  going to search inside children...\n");
      DCMITEM *rptr=dcmFindDownTag(iptr->child_item, 0, tag, verbose);
      if(rptr!=NULL) return(rptr);
      if(verbose>3) printf("  nothing found in any of the children\n");
    }
    iptr=iptr->next_item;
  }
  /* Stop if we found tag */
  if(iptr!=NULL) return(iptr);
  /* Stop if we do not have parent */
  if(d->parent_item==NULL) return(NULL);
  /* Search from the parent, but not including the parent */
  if(verbose>2) printf("  going to search the parents sister...\n");
  return(dcmFindTag(d->parent_item, 1, tag, verbose));

#if(0)
  if(d->parent_item==NULL) return(NULL);

  /* Search from the parent */
  if(verbose>2) printf("  going to search inside parent...\n");
  return(dcmFindTag(d->parent_item, 1, tag, verbose));
#endif
}
/*****************************************************************************/

/*****************************************************************************/
/** Search for specified tag in DCMITEM data tree, but only downward.

    @return Returns pointer to next item with the tag, or NULL if not found.
 */
DCMITEM *dcmFindDownTag(
  /** Pointer to current DICOM item. */
  DCMITEM *d,
  /** Omit this item from the search. */
  const short int omit,
  /** Pointer to the DICOM tag that is searched for. */
  DCMTAG *tag,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(d==NULL || tag==NULL) return(NULL);
  if(verbose>0) printf("%s((%04X,%04X),%d)\n", __func__, tag->group, tag->element, omit);
  DCMITEM *iptr;
  if(omit==0) iptr=d; else iptr=d->next_item;
  while(iptr!=NULL) {
    if(verbose>2)
      printf(" checking tag(%04X,%04X)...\n", iptr->tag.group, iptr->tag.element);
    if(iptr->tag.group==tag->group && iptr->tag.element==tag->element) {
      if(verbose>2) printf("  found!\n");
      break;
    }
    /* Check if this item has children */
    if(iptr->child_item!=NULL) {
      if(verbose>2) printf("  going to search inside children...\n");
      DCMITEM *rptr=dcmFindDownTag(iptr->child_item, 0, tag, verbose);
      if(rptr!=NULL) return(rptr);
      if(verbose>3) printf("  nothing found in any of the children\n");
    }
    iptr=iptr->next_item;
  }
  /* Stop if we found tag */
  if(iptr!=NULL) return(iptr);
  /* Stop anyway */
  return(NULL);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print contents of given DICOM item into stdout. */
void dcmitemPrint(
  /** Pointer to item. */
  DCMITEM *d
) {
  if(d==NULL) {printf("(null)\n"); fflush(stdout); return;}
  printf("tag(%04X,%04X)", d->tag.group, d->tag.element); fflush(stdout);
  printf(" VR=%s", dcmVRName(d->vr)); fflush(stdout);
  if(d->vl==0xFFFFFFFF) printf(" VL=%08X", d->vl); else printf(" VL=%u", d->vl);
  fflush(stdout);
  char *buf=dcmValueString(d); printf(" '%s'", buf); free(buf);
  printf("\n"); fflush(stdout);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set DICOM Tag group and element. */
void dcmTagSet(
  /** Pointer to Tag to set. */
  DCMTAG *tag,
  /** Tag Group */
  unsigned short int group,
  /** Tag Element */
  unsigned short int element
) {
  tag->group=group;
  tag->element=element;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add an item to DCMFILE data struct.
   @sa dcmfileInit, dcmfileFree
   @return 0 if successful, otherwise >0.
*/
int dcmAddItem(
  /** Pointer to DCMFILE. */
  DCMFILE *dcm,
  /** Pointer to a previous item in DCMFILE, into which to link this item;
      enter NULL to add as next item to the highest level. */
  DCMITEM *d,
  /** Add as child; 1=yes, 0=no. */
  short int aschild,
  /** Tag */
  DCMTAG tag,
  /** VR */
  dcmvr vr,
  /** VL; enter 0xFFFFFFFF to use VR's default length. */
  unsigned int vl,
  /** Pointer to the item value as byte array */
  char *rd,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) {
    printf("%s(dcm, (%04X,%04X))", __func__, tag.group, tag.element);
    if(d==NULL) printf(", null"); else printf(", ptr");
    printf(", %d, %s, 0x%08X, rd", aschild, dcmVRName(vr), vl);
    printf(")\n");
  }
  if(dcm==NULL) return(1);
  if(vr==DCM_VR_INVALID) return(2);

  /* Check that caller has not given previous item pointer when there are none in DCMFILE */
  if(d!=NULL && dcm->item==NULL) return(3);
  /* Check that caller has not given previous item pointer that already has child */
  if(d!=NULL && aschild && d->child_item!=NULL) return(4);

  /* Check whether we currently support the Transfer UID */
  if(dcm->truid!=DCM_TRUID_LEE) return(5);

  /* Allocate memory for the new element; do not free it here, since it will be part of
     DCMFILE struct. */
  if(verbose>1) printf("  allocating memory for the item\n");
  DCMITEM *item=(DCMITEM*)malloc(sizeof(DCMITEM));
  if(item==NULL) return(11);
  item->next_item=item->child_item=(DCMITEM*)NULL;
  item->fp=dcm->fp; item->truid=dcm->truid;
  item->rd=(char*)NULL;

  /* Set item tag, VR, and VL */
  if(verbose>1) printf("  setting item contents\n");
  item->tag.group=tag.group;
  item->tag.element=tag.element;
  item->vr=vr;
  item->vl=vl;
  /* Allocate memory for item value */
  size_t s;
  if(vl==0xFFFFFFFF) s=dcmVRVLength(vr); else s=vl;
  if(s>0) {
    if(item->vl==0xFFFFFFFF) item->vl=(unsigned int)s;
    if(verbose>1) printf("  allocating %u bytes for the item value\n", (unsigned int)s);
    item->rd=(char*)calloc(s, sizeof(char));
    if(item->rd==NULL) {free(item); return(21);}
  } else {
    if(verbose>1) printf("zero size for item value\n");
    if(rd==NULL) {
      if(verbose>1) printf("... which is ok since value is empty, too.\n");
    } else {
      if(verbose>0) printf("... which is not ok because we have value to store.\n");
      if(item->rd==NULL) {free(item); return(22);}
    }
  }
  /* Copy the item value */
  if(rd!=NULL && s>0) {
    if(verbose>1) printf("  copying the item value\n");
    /* Special treatment for strings, because those tend to be shorter than told */
    if(vr==DCM_VR_LO || vr==DCM_VR_LT || vr==DCM_VR_PN || vr==DCM_VR_SH || vr==DCM_VR_UI || vr==DCM_VR_UR) 
    {
      unsigned int len=strnlen(rd, s);
      if(len<s) strlcpy(item->rd, rd, s);
      else memcpy(item->rd, rd, s);
    } else if(vr==DCM_VR_DS || vr==DCM_VR_IS) 
    {
      unsigned int len=strnlen(rd, s);
      if(len<s) strlcpy(item->rd, rd, s);
      else memcpy(item->rd, rd, s);
    } else {
      memcpy(item->rd, rd, s);
    }
  }


  /* If we have the item to link to, then do the linking */
  if(verbose>1) printf("  link the item.\n");
  if(d!=NULL) {
    if(aschild) {
      d->child_item=item;
      item->parent_item=d;
      item->prev_item=(DCMITEM*)NULL;
    } else if(d->next_item==NULL) {
      d->next_item=item;
      item->prev_item=d;
      item->parent_item=d->parent_item;
    } else {
      /* find the last item in the list */
      DCMITEM *ip=d; while(ip->next_item!=NULL) ip=ip->next_item;
      ip->next_item=item;
      item->prev_item=ip;
      item->parent_item=ip->parent_item;
    }
  } else if(dcm->item==NULL) {
    /* This is truly the first item ever */
    dcm->item=item;
    item->prev_item=item->parent_item=(DCMITEM*)NULL;
  } else {
    /* Caller lets us find the item to link to */
    DCMITEM *ip=dcm->item; while(ip->next_item!=NULL) ip=ip->next_item;
    ip->next_item=item;
    item->prev_item=ip;
    item->parent_item=ip->parent_item;
  }

  if(verbose>2) dcmitemPrint(item);

  if(verbose>1) printf("  all done.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Search the range of integer values for specified tag in DCMITEM data tree.

   @return 0 if successful, otherwise >0.
 */
int dcmTagIntRange(
  /** Pointer to top DICOM item. */
  DCMITEM *d,
  /** Pointer to the DICOM tag that is searched for. */
  DCMTAG *tag,
  /** Pointer to int where min value is written; NULL if not needed. */
  int *mi,
  /** Pointer to int where max value is written; NULL if not needed. */
  int *ma,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(d==NULL || tag==NULL) return(1);
  if(verbose>0) printf("%s(%04X,%04X)\n", __func__, tag->group, tag->element);

  /* Find the first occurrence of tag */
  DCMITEM *iptr=dcmFindTag(d, 0, tag, 0);
  if(iptr==NULL) {
    if(verbose>1) printf("  tag not found.\n");
    return(2);
  }
  int iv=dcmitemGetInt(iptr); if(verbose>2) printf("  value := %d\n", iv);
  int mini=iv;
  int maxi=iv;
  /* Find the rest of occurrences */
  while(iptr!=NULL) {
    iptr=dcmFindTag(iptr, 1, tag, 0);
    if(iptr!=NULL) {
      iv=dcmitemGetInt(iptr); if(verbose>2) printf("  value := %d\n", iv);
      if(iv>maxi) maxi=iv; else if(iv<mini) mini=iv;
    }
  }
  if(verbose>1) {
    printf("  minimum_value := %d\n", mini);
    printf("  maximum_value := %d\n", maxi);
  }
  if(mi!=NULL) *mi=mini;
  if(ma!=NULL) *ma=maxi;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
