/** @file dcmfile.c
    @brief DICOM file and directory functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check that two DICOM files belong to the same image.
    @details Defaults to yes, if the needed tags are missing.
    @return Returns 1 if they are part of the same image, 0 if not.*/
int dcmSameImage(
  /** Pointer to the first DICOM file data (pixel data not necessary). */
  const DCMFILE *d1,
  /** Pointer to the second DICOM file data (pixel data not necessary). */
  const DCMFILE *d2,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) printf("%s()\n", __func__);
  if(d1==NULL || d2==NULL) return(0);

  DCMTAG tag;
  DCMITEM *iptr;

  /* Check the similarity of Series Instance UID */
  char siuid1[64], siuid2[64];
  siuid1[0]=siuid2[0]=(char)0;
  dcmTagSet(&tag, 0x0020, 0x000E);
  iptr=dcmFindTag(d1->item, 0, &tag, 0);
  if(iptr!=NULL) strlcpy(siuid1, iptr->rd, 64);
  iptr=dcmFindTag(d2->item, 0, &tag, 0);
  if(iptr!=NULL) strlcpy(siuid2, iptr->rd, 64);
  if(verbose>1) {
    printf("Series Instance UID [1] := %s\n", siuid1);
    printf("Series Instance UID [2] := %s\n", siuid2);
  }
  if(strcmp(siuid1, siuid2)) return(0);

  /* Check the similarity of Study ID */
  char sid1[16], sid2[16];
  sid1[0]=sid2[0]=(char)0;
  dcmTagSet(&tag, 0x0020, 0x0010);
  iptr=dcmFindTag(d1->item, 0, &tag, 0);
  if(iptr!=NULL) strlcpy(sid1, iptr->rd, 16);
  iptr=dcmFindTag(d2->item, 0, &tag, 0);
  if(iptr!=NULL) strlcpy(sid2, iptr->rd, 16);
  if(verbose>1) {
    printf("Study ID [1] := %s\n", sid1);
    printf("Study ID [2] := %s\n", sid2);
  }
  if(strcmp(sid1, sid2)) return(0);

  /* Check the series number */
  char snr1[12], snr2[12];
  snr1[0]=snr2[0]=(char)0;
  dcmTagSet(&tag, 0x0020, 0x0011);
  iptr=dcmFindTag(d1->item, 0, &tag, 0);
  if(iptr!=NULL) strlcpy(snr1, iptr->rd, 12);
  iptr=dcmFindTag(d2->item, 0, &tag, 0);
  if(iptr!=NULL) strlcpy(snr2, iptr->rd, 12);
  if(verbose>1) {
    printf("Series Number [1] := %s\n", snr1);
    printf("Series Number [2] := %s\n", snr2);
  }
  if(strcmp(snr1, snr2)) return(0);

  /* The rest of checks should not be necessary... */
  if(verbose>2) {printf("  ... still looks the same...\n"); fflush(stdout);}

  /* Check the number of frames (time slices) */
  unsigned short int frameNr1=0, frameNr2=0;
  dcmTagSet(&tag, 0x0054, 0x0101);
  iptr=dcmFindTag(d1->item, 0, &tag, 0);
  if(iptr!=NULL) frameNr1=(unsigned short int)dcmitemGetInt(iptr);
  iptr=dcmFindTag(d2->item, 0, &tag, 0);
  if(iptr!=NULL) frameNr2=(unsigned short int)dcmitemGetInt(iptr);
  if(verbose>1) {
    printf("frameNr [1] := %u\n", frameNr1);
    printf("frameNr [2] := %u\n", frameNr2);
  }
  if(frameNr1!=frameNr2) return(0);
  if(verbose>2) {printf("  ... still looks the same...\n"); fflush(stdout);}

  /* Check the number of slices */
  unsigned short int sliceNr1=0, sliceNr2=0;
  dcmTagSet(&tag, 0x0054, 0x0081);
  iptr=dcmFindTag(d1->item, 0, &tag, 0);
  if(iptr!=NULL) sliceNr1=(unsigned short int)dcmitemGetInt(iptr);
  iptr=dcmFindTag(d2->item, 0, &tag, 0);
  if(iptr!=NULL) sliceNr2=(unsigned short int)dcmitemGetInt(iptr);
  if(verbose>1) {
    printf("sliceNr [1] := %u\n", sliceNr1);
    printf("sliceNr [2] := %u\n", sliceNr2);
  }
  if(sliceNr1!=sliceNr2) return(0);

  if(verbose>2) {printf("  ... the same.\n"); fflush(stdout);}
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief List DICOM files belonging to one image.
    @details User can give DICOM image as a folder containing image file(s),
     or as a name of one of the files of the image. This function lists the 
     files (with paths) belonging to the same image.
    @return Code tpcerror, TPCERROR_OK (0) when successful.
 */
int dcmFileList(
  /** Pointer to the name of DICOM directory or file. */
  const char *filename,
  /** Pointer to initiated IFT structure, into which file names will be stored;
      any previous contents will be deleted. */
  IFT *ift,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(filename==NULL || strnlen(filename, 10)<1 || ift==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>1) printf("%s('%s', ...)\n", __func__, filename);

  /* Make the list */
  unsigned short int fileNr=0;
  if(pathExist(filename)) {
    /* We got a directory */
    if(verbose>2) printf("%s is a directory\n", filename);
    /* List all files in it */
    iftFree(ift);
    fileNr=pathFileList(filename, ift);
    if(fileNr==0) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
      return(TPCERROR_NO_FILE);
    }
    /* Remove files that are not DICOM */
    int i=ift->keyNr-1;
    do {
      if(!dcmVerifyMagic(ift->item[i].value, NULL)) iftDelete(ift, i);
      i--;
    } while(i>=0);
    fileNr=ift->keyNr; if(verbose>3) printf("fileNr := %d\n", fileNr);
    if(fileNr==0) {
      iftFree(ift);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
    /* Check that all remaining files belong to the same image */
    DCMFILE dcm1; dcmfileInit(&dcm1);
    i=0; int ret;
    ret=dcmFileRead(ift->item[i].value, &dcm1, 1, status);
    if(ret!=TPCERROR_OK) {iftFree(ift); return(ret);}
    DCMFILE dcm2; dcmfileInit(&dcm2);
    for(i=1; i<ift->keyNr; i++) {
      ret=dcmFileRead(ift->item[i].value, &dcm2, 1, NULL /*status*/);
      if(ret!=TPCERROR_OK) {
        iftFree(ift); dcmfileFree(&dcm1);
        statusSet(status, __func__, __FILE__, __LINE__, ret);
        return(ret);
      }
      if(!dcmSameImage(&dcm1, &dcm2, verbose-10)) {
        dcmfileFree(&dcm2); iftFree(ift);
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_DATA);
        return(TPCERROR_INCOMPATIBLE_DATA);
      }
      dcmfileFree(&dcm2);
    }
    dcmfileFree(&dcm1);
  } else if(fileExist(filename)) {
    /* We got a file */
    if(verbose>2) printf("%s is a file\n", filename);
    /* Verify that it is DICOM */
    if(!dcmVerifyMagic(filename, NULL)) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
    if(verbose>2) {printf("... and DICOM\n"); fflush(stdout);}
    /* Read the specified file */
    DCMFILE dcm1; dcmfileInit(&dcm1);
    int ret;
    ret=dcmFileRead(filename, &dcm1, 1, status);
    if(ret!=TPCERROR_OK) {iftFree(ift); return(ret);}
    /* Check whether Per-frame Functional Groups Sequence is set */
    DCMTAG tag; dcmTagSet(&tag, 0x5200, 0x9230);
    if(dcmFindTag(dcm1.item, 0, &tag, 0)!=NULL) {
      dcmfileFree(&dcm1);
      if(verbose>1)
        printf("Per-frame Functional Groups Sequence is available: assuming single-file.\n");
      iftFree(ift); fileNr=1;
      iftPut(ift, "", filename, 0, NULL); 
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
      return(TPCERROR_OK);
    }

    /* Since there is no RELIABLE way to tell how many files make up the whole
       image, we have to list files in the same path */
    char *pathname=strdup(filename); filenameRmFile(pathname);
    if(verbose>3) printf("%s is its path\n", pathname);
    iftFree(ift); 
    fileNr=pathFileList(pathname, ift); free(pathname);
    if(verbose>3) printf("fileNr := %d\n", fileNr);
    if(fileNr==0) {
      iftFree(ift);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
      return(TPCERROR_INVALID_FILENAME);
    }
    /* Remove files that are not DICOM */
    int i=ift->keyNr-1;
    do {
      if(!dcmVerifyMagic(ift->item[i].value, NULL)) iftDelete(ift, i);
      i--;
    } while(i>=0);
    fileNr=ift->keyNr; if(verbose>3) printf("fileNr := %d\n", fileNr);
    if(fileNr==0) {
      iftFree(ift);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
    /* Remove files that do not belong to the same series as the specified file */
    DCMFILE dcm2; dcmfileInit(&dcm2);
    i=ift->keyNr-1;
    do {
      ret=dcmFileRead(ift->item[i].value, &dcm2, 1, NULL /*status*/);
      if(ret!=TPCERROR_OK || !dcmSameImage(&dcm1, &dcm2, verbose-10)) {
        if(verbose>5) {printf("  not same, removing from list\n"); fflush(stdout);}
        iftDelete(ift, i); 
      }
      dcmfileFree(&dcm2);
      i--;
    } while(i>=0);
    dcmfileFree(&dcm1);
    fileNr=ift->keyNr; if(verbose>3) printf("fileNr := %d\n", fileNr);
    if(fileNr==0) {
      iftFree(ift);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return(TPCERROR_UNSUPPORTED);
    }
  } else {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
    return(TPCERROR_NO_FILE);
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
