/** @file dcmio.c
    @brief IO functions for DICOM files.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verify that given file (either file name or file pointer) appears to be
    DICOM file, based on the magic number.
    @sa dcmReadFile, dcmReadTransferSyntaxUID
    @return Returns 1 if DICOM magic number can be found, or 0 if not.
    @author Vesa Oikonen
 */
int dcmVerifyMagic(
  /** Name of file to open and to verify for the magic number; enter NULL to
      use the file pointer (next argument) instead. */
  const char *filename,
  /** File pointer of file to check, opened with fp=fopen(filename, "rb"); 
      enter NULL to open file (previous argument) locally. 
      Previously opened file pointer is first rewound to start; if DICOM magic
      number is found, then file pointer is left to the end of magic number,
      and if not, it is rewound to the file start.
  */ 
  FILE *fp
) {
  FILE *lfp;

  if(filename==NULL && fp==NULL) return(0);
  if(fp!=NULL) {
    lfp=fp; rewind(lfp);
  } else {
    lfp=fopen(filename, "rb");
  }
  if(lfp==NULL) return(0);

  /* Skip the first 128 bytes */
  if(fseek(lfp, 128, SEEK_SET)) {if(fp==NULL) fclose(lfp); return(0);}

  /* Read the next 4 bytes as characters */
  char buf[5];
  size_t n=fread(&buf, 1, 4, lfp);
  buf[4]=(char)0;
  //if(fp==NULL) fclose(lfp); else rewind(lfp);
  if(n!=(size_t)4) {rewind(lfp); return(0);}

  /* Check the magic number */
  if(strncmp(buf, "DICM", 4)==0) {
    if(fp==NULL) fclose(lfp);
    return(1);
  } else {
    if(fp==NULL) fclose(lfp); else rewind(lfp);
    return(0);
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Read and identify the DICOM Transfer Syntax UID.
    @return Returns the enumerated UID, or DCM_TRUID_INVALID.
    @sa dcmVerifyMagic
 */
dcmtruid dcmReadTransferSyntaxUID(
  /** Pointer to DICOM file opened in binary format, and positioned right
      after the Magic number. 
      Position will be returned to this position. */
  FILE *fp
) {
  if(fp==NULL || feof(fp)) return(DCM_TRUID_INVALID);
  /* Save original file position */
  fpos_t opos;
  if(fgetpos(fp, &opos)) return(DCM_TRUID_INVALID);
  /* Read file until we find DICOM tag 0x0002,0x0010 */
  DCMTAG tag;
  int tag_found=0;
  while(!tag_found && !feof(fp)) {
    //printf("reading tag\n");
    if(dcmReadFileTag(fp, &tag)) break;
    //printf(" -> (%04x,%04x)\n", tag.group, tag.element);
    if(tag.group==0x0002 && tag.element==0x0010) {tag_found=1; break;}
    /* not the tag that we want, so just go through this item */
    //printf("reading vr etc\n");
    dcmvr vr=dcmReadFileVR(fp, NULL);
    //printf("-> %s\n", dcmVRName(vr));
    unsigned int vl=0;
    if(dcmVRReserved(vr)==0) vl=dcmReadFileVL(fp, 2); 
    else vl=dcmReadFileVL(fp, 4); 
    //printf("reserved=%d\n", dcmVRReserved(vr));
    if(vr==DCM_VR_SQ) break;
    //printf("huuh\n");
    if(vl==0xFFFFFFFF) break;
    char buf[vl+1];
    //printf("read buf length %d\n", vl);
    if(fread(buf, 1, vl, fp)!=vl) break;
  } // get next tag
  //printf("end of loop\n");
  if(!tag_found) {fsetpos(fp, &opos); return(DCM_TRUID_INVALID);}
  /* Read the UID */
  //printf("reading vr\n");
  if(dcmReadFileVR(fp, NULL)!=DCM_VR_UI) {
    fsetpos(fp, &opos); return(DCM_TRUID_INVALID);
  }
  //printf("reading vl\n");
  unsigned int vl=dcmReadFileVL(fp, 2); 
  if(vl==0 || vl==0xFFFFFFFF) {fsetpos(fp, &opos); return(DCM_TRUID_INVALID);}
  char uid[vl+1];
  //printf("reading uid\n");
  if(fread(uid, 1, vl, fp)!=vl) {fsetpos(fp, &opos); return(DCM_TRUID_INVALID);}
  /* Return file position to the original */
  fsetpos(fp, &opos);
  /* Identify the UID */
  return(dcmTrUID(uid));
}
/*****************************************************************************/

/*****************************************************************************/
/** Read DICOM tag from current file position.
    @note Tag validity is not verified here; error may be caused by end-of-file.
     Alternatively, read may seem to be successful, but tag contents are 
     file padding symbols (0xFFFC).
    @sa dcmReadFile, dcmReadFileElement
    @return Returns 0 when successful, otherwise >0.
    @author Vesa Oikonen
 */
int dcmReadFileTag(
  /** File pointer, positioned at the start of the element (and tag). */
  FILE *fp,
  /** Pointer to DICOM tag struct; enter NULL if you just need to move
      file position over the tag. */
  DCMTAG *tag
) {
  if(tag!=NULL) {tag->group=tag->element=0xFFFC;} // padding
  if(fp==NULL || feof(fp)) return(1);
  unsigned short int buf[2];
  size_t n=fread(&buf, 2, 2, fp);
  if(n!=2) return(2+n);
  if(tag!=NULL) {
    tag->group=buf[0];
    tag->element=buf[1];
    if(!endianLittle()) { // tag is by default little endian
      swap(&tag->group, &tag->group, 2);
      swap(&tag->element, &tag->element, 2);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write DICOM tag into current file position.
    @sa dcmWriteFile, dcmReadFileTag
    @return Code tpcerror, TPCERROR_OK (0) when successful.
    @author Vesa Oikonen
 */
int dcmWriteFileTag(
  /** File pointer, at the write position. */
  FILE *fp,
  /** Pointer to DICOM tag struct to write. */
  DCMTAG *tag
) {
  if(fp==NULL || tag==NULL) return(TPCERROR_FAIL);
  unsigned short int buf[2];
  buf[0]=tag->group;
  buf[1]=tag->element;
  if(!endianLittle()) swabip(buf, 4); // tag is by default little endian
  if(fwrite(&buf, 2, 2, fp)!=2) return TPCERROR_CANNOT_WRITE;
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write DICOM Sequence delimitation item into current file position.

    This item consists of four byte sequence delimitation tag (0xFFFE, 0xE0DD)
    and four byte item length (0x00000000), i.e. together 8 bytes.

    @sa dcmWriteFileTag, dcmReadFileTag, dcmWriteFileSQItemDelimTag
    @return Code tpcerror, TPCERROR_OK (0) when successful.
    @author Vesa Oikonen
 */
int dcmWriteFileSQDelimItem(
  /** File pointer, at the write position. */
  FILE *fp
) {
  if(fp==NULL) return(TPCERROR_FAIL);
  int ret;
  DCMTAG tag; 
  tag.group=0xFFFE; tag.element=0xE0DD;
  ret=dcmWriteFileTag(fp, &tag); if(ret!=TPCERROR_OK) return(ret);
  tag.group=0x0000; tag.element=0x0000;
  ret=dcmWriteFileTag(fp, &tag); if(ret!=TPCERROR_OK) return(ret);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write DICOM Sequence Item Delimitation Tag with VL into current file position.

    This tag consists of four bytes, sequence item delimitation tag (0xFFFE, 0xE00D),
    followed by four byte item length (0x00000000), i.e. together 8 bytes.

    @sa dcmWriteFileTag, dcmReadFileTag, dcmWriteFileSQDelimItem
    @return Code tpcerror, TPCERROR_OK (0) when successful.
    @author Vesa Oikonen
 */
int dcmWriteFileSQItemDelimTag(
  /** File pointer, at the write position. */
  FILE *fp
) {
  if(fp==NULL) return(TPCERROR_FAIL);
  int ret;
  DCMTAG tag; 
  tag.group=0xFFFE; tag.element=0xE00D;
  ret=dcmWriteFileTag(fp, &tag); if(ret!=TPCERROR_OK) return(ret);
  tag.group=0x0000; tag.element=0x0000;
  ret=dcmWriteFileTag(fp, &tag); if(ret!=TPCERROR_OK) return(ret);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read DICOM value representation (2 or 4 bytes) from current file position.
    @sa dcmReadFileTag, dcmReadFileElement
    @return Returns the enumerated VR number, DCM_VR_INVALID in case of an error.
    @author Vesa Oikonen
 */
dcmvr dcmReadFileVR(
  /** File pointer, positioned at the VR start. */
  FILE *fp,
  /** Pointer for VR string, allocated for at least 3 characters to have space
      for the trailing null; enter NULL if not needed. */
  char *vrstr 
) {
  if(vrstr!=NULL) vrstr[0]=(char)0;
  if(fp==NULL) return(DCM_VR_INVALID);

  /* Read the first two bytes */
  char buf[3];
  if(fread(&buf, 1, 2, fp)!=2) return(DCM_VR_INVALID);
  buf[2]=(char)0;

  /* Identify the VR */
  dcmvr lvr=dcmVRId(buf);
  if(vrstr!=NULL) {
    if(lvr!=DCM_VR_INVALID) strcpy(vrstr, dcmVRName(lvr));
    else strcpy(vrstr, buf);
  }

  /* If this VR has extra 2 byte reserved space, then
     we need to read but do not use the next 2 bytes. */
  if(dcmVRReserved(lvr)!=0) {
    if(fread(&buf, 1, 2, fp)!=2) return(DCM_VR_INVALID);
  }
  return(lvr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read DICOM value length (2 or 4 bytes, depending on VR) from current file position.
    @sa dcmReadFileTag, dcmReadFileVR, dcmReadFileElement
    @return Returns the value length.
    @author Vesa Oikonen
 */
unsigned int dcmReadFileVL(
  /** File pointer, positioned at the VL start. */
  FILE *fp,
  /** Number of bytes (2 or 4) in the VL representation. */
  unsigned int n
) {
  unsigned int vl=0;
  if(fp==NULL || (n!=2 && n!=4)) return(vl);

  /* Read 2 or 4 bytes */
  if(n==2) {
    unsigned short int si;
    if(fread(&si, 2, 1, fp)!=1) return(vl);
    if(!endianLittle()) swap(&si, &si, 2);
    vl=si;
  } else if(n==4) {
    unsigned int li;
    if(fread(&li, 4, 1, fp)!=1) return(vl);
    if(!endianLittle()) swap(&li, &li, 4);
    vl=li;
  }
  return(vl);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read DICOM Value Representation (VR, 2 or 4 bytes) and Value Length (VL, 2 or 4 bytes)
    from current file position.
   @sa dcmReadFileVR, dcmReadFileVL, dcmReadFileTag, dcmReadFileElement, dcmWriteFileVRVL
   @return Code tpcerror, TPCERROR_OK (0) when successful.
   @author Vesa Oikonen
 */
int dcmReadFileVRVL(
  /** File pointer, positioned at the VR start. */
  FILE *fp,
  /** Pointer for enumerated VR value; enter NULL if not needed (file pointer moved anyway). */
  dcmvr *vr,
  /** Pointer for VL; enter NULL if not needed (file pointer moved anyway). */
  unsigned int *vl,
  /** Pointer for number of bytes read from file; enter NULL if not needed. */
  unsigned int *n
) {
  if(vr!=NULL) *vr=DCM_VR_INVALID;
  if(vl!=NULL) *vl=0;
  if(n!=NULL) *n=0;
  if(fp==NULL) return(TPCERROR_FAIL);

  /* Read the first two bytes */
  char buf[3];
  if(fread(&buf, 1, 2, fp)!=2) return(TPCERROR_CANNOT_READ); else if(n!=NULL) *n+=2;
  buf[2]=(char)0;

  /* Identify the VR */
  dcmvr lvr=dcmVRId(buf);
  if(vr!=NULL) *vr=lvr;
  if(lvr==DCM_VR_INVALID) return(TPCERROR_UNSUPPORTED);

  /* If this VR has extra 2 byte reserved space, then
     we need to read but do not use the next 2 bytes. */
  unsigned int bsize=2+dcmVRReserved(lvr);
  if(bsize==4) {
    if(fread(&buf, 1, 2, fp)!=2) return(TPCERROR_CANNOT_READ);
    if(n!=NULL) *n+=2;
  }

  /* Read VL from the next 2 or 4 bytes */
  unsigned int lvl=0;
  if(bsize==2) {
    unsigned short int si;
    if(fread(&si, 2, 1, fp)!=1) return(TPCERROR_CANNOT_READ);
    if(!endianLittle()) swap(&si, &si, 2);
    lvl=si;
  } else {
    unsigned int li;
    if(fread(&li, 4, 1, fp)!=1) return(TPCERROR_CANNOT_READ);
    if(!endianLittle()) swap(&li, &li, 4);
    lvl=li;
  }
  if(n!=NULL) *n+=bsize;
  if(vl!=NULL) *vl=lvl;

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write DICOM Value Representation (VR, 2 or 4 bytes) and Value Length (VL, 2 or 4 bytes)
    into current file position.
   @sa dcmWriteFileTag, dcmReadFileElement, dcmWriteFileVRVL
   @return Code tpcerror, TPCERROR_OK (0) when successful.
   @author Vesa Oikonen
 */
int dcmWriteFileVRVL(
  /** File pointer, opened fro writing in binary mode. */
  FILE *fp,
  /** Enumerated VR value. */
  dcmvr vr,
  /** VL */
  unsigned int vl,
  /** Pointer for number of bytes written into file; enter NULL if not needed. */
  unsigned int *n
) {
  if(n!=NULL) *n=0;
  if(fp==NULL || vr==DCM_VR_INVALID) return(TPCERROR_FAIL);

  /* If this VR has extra 2 byte reserved space, then
     we need to write VR and VL with 4 bytes each, other wise with 2 bytes each. */
  unsigned int bsize=2+dcmVRReserved(vr);

  char buf[10];

  /* Write VR */
  memcpy(buf, dcmVRName(vr), 2); buf[2]=buf[3]=(char)0;
  if(fwrite(buf, bsize, 1, fp)!=1) return(TPCERROR_CANNOT_WRITE);
  if(n!=NULL) *n+=bsize;

  /* Write VL */
  memcpy(buf, &vl, bsize);
  if(bsize==2) buf[2]=buf[3]=(char)0;
//  if(!endianLittle()) swabip(buf, bsize);
  if(!endianLittle()) swap(buf, buf, bsize);
  if(fwrite(buf, bsize, 1, fp)!=1) return(TPCERROR_CANNOT_WRITE);
  if(n!=NULL) *n+=bsize;

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read an element from DICOM file, and add it to the given linked list.
    This function will be called recursively in case of sequential items.
    @return Code tpcerror, TPCERROR_OK (0) when successful, 
     TPCERROR_NO_KEY when no more elements could be read.
    @sa dcmFileRead
 */
int dcmFileReadNextElement(
  /** Pointer to DCMFILE struct; must be initiated before first call. */
  DCMFILE *dcm,
  /** Pointer to previous element; NULL if none exists (yet). */
  DCMITEM *prev_item,
  /** Pointer to parent element; NULL if none exists (yet). */
  DCMITEM *parent_item,
  /** Add as next element (0) or as child element. */
  const short int sub,
  /** Read only header (1), or read both header and pixel data (0). */
  const short int headerOnly, 
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  int verbose
) {
  if(verbose>0) printf("%s(DCMFILE*, DCMITEM*, DCMITEM*, %d, %d)\n", __func__, sub, headerOnly);
  if(dcm==NULL || dcm->fp==NULL) return(TPCERROR_FAIL);
  if(sub!=0 && parent_item==NULL) return(TPCERROR_FAIL);
  if(feof(dcm->fp)) return(TPCERROR_NO_KEY);

  /* Check whether we currently support the Transfer UID */
  switch(dcm->truid) {
    case DCM_TRUID_LEE:
      break;
    case DCM_TRUID_LEI: // actually not supported yet
      break;
    default:
      return(TPCERROR_UNSUPPORTED);
  }

  if(verbose>10) {
    if(dcm->item==NULL) printf(" will add first element\n");
    else if(sub==0) printf(" will add next element\n");
    else printf(" will add sub-element\n");
  }

  /* Is this a child to a sequence element? */
  int sq_child=0;
  if(parent_item!=NULL && parent_item->vr==DCM_VR_SQ) {sq_child=1;}
  if(verbose>10 && sq_child!=0) printf(" we're a child to a sequence element\n");

  /* Allocate memory for the new element */
  DCMITEM *item=(DCMITEM*)malloc(sizeof(DCMITEM));
  if(item==NULL) return(TPCERROR_OUT_OF_MEMORY);
  item->prev_item=prev_item;
  item->parent_item=parent_item;
  item->next_item=item->child_item=(DCMITEM*)NULL;
  item->fp=dcm->fp; item->truid=dcm->truid;
  item->rd=(char*)NULL;

  /* Save current file position (should be the start of element) */
  if(fgetpos(dcm->fp, &item->pos)) {
    free(item); return(TPCERROR_CANNOT_READ);
  }

  /* Read the tag (2x2 bytes) */
  if(verbose>10) {
    long long int tagpos=ftello(dcm->fp);
    printf(" reading tag at %lld\n", tagpos);
  }
  if(dcmReadFileTag(dcm->fp, &item->tag)) {
    if(verbose>1 && !feof(dcm->fp)) printf(" error in reading the tag.\n");
    free(item);
    if(feof(dcm->fp)) return(TPCERROR_NO_KEY);
    return(TPCERROR_CANNOT_READ);
  }

  if(verbose>2) {
    printf(" tag(%04x,%04x) with %u parents\n", 
           item->tag.group, item->tag.element, dcmitemParentNr(item));
  }

  /* If child, then check for item delimitation tag (although it should not be here) */
  if(dcmitemParentNr(item)>0 && item->tag.group==0xFFFE && item->tag.element==0xE00D) {
    if(verbose>10)
      printf(" item delimitation tag(%04x,%04x) found, reading VL\n", 
             item->tag.group, item->tag.element);
    unsigned long int vl=dcmReadFileVL(dcm->fp, 4); 
    if(verbose>1) printf(" item delimitation tag VL := %lu (0x%08lx)\n", vl, vl);
    if(vl!=0) {
      if(verbose>1) printf(" error: VL should have been 0\n");
      return(TPCERROR_INVALID_VALUE);
    }
    return(TPCERROR_OK);
  }


  /* Read value representation and length (VR and VL, 2x2 or 2x4 bytes) */
  if(item->tag.group==0x0002 || dcm->truid==DCM_TRUID_LEE) {
    // Group 0x0002 is always DCM_TRUID_LEE, the other groups are specified by
    // the Transfer Syntax UID 
    if(verbose>10) printf(" reading VR and VL\n");
    int ret;
    unsigned int n;
    ret=dcmReadFileVRVL(dcm->fp, &item->vr, &item->vl, &n);
    if(ret!=TPCERROR_OK) {
      if(verbose>1) printf(" invalid VR or VL\n");
      free(item); return(ret);
    }
    if(verbose>1) {
      printf(" VR := %s (%s)\n", dcmVRName(item->vr), dcmVRDescr(item->vr));
      printf(" VL := %u (0x%08x) (%d bytes field)\n", item->vl, item->vl, n/2);
      fflush(stdout);
    }
  } else if(dcm->truid==DCM_TRUID_LEI) {
    if(verbose>10) printf(" set VR based on the tag\n");
    unsigned int i=dcmDictFindTag(&item->tag);
    item->vr=dcmVRId(dcmDictIndexVR(i));
    if(item->vr!=DCM_VR_INVALID) {
      if(verbose>1) {
        printf(" VR := %s (%s)\n", dcmVRName(item->vr), dcmVRDescr(item->vr));
        fflush(stdout);
      }
    } else {
      if(verbose>1) printf(" VR not known for tag(%04x,%04x)\n",
                            item->tag.group, item->tag.element);
    }
    if(verbose>10) printf(" reading VL\n");
    item->vl=dcmReadFileVL(dcm->fp, 4); 
    if(verbose>1) {
      printf(" VL := %u (0x%08x)\n", item->vl, item->vl);
      fflush(stdout);
    }
  }

  /* Read value field, and add the current element to the list */
  if(item->vr==DCM_VR_SQ) {
    if(ftello(dcm->fp)<0) return(TPCERROR_INVALID_FORMAT);
    unsigned long long int sqPos=(unsigned long int)ftello(dcm->fp);
    if(verbose>10) {printf(" sequence... at %llu\n", sqPos); fflush(stdout);}
    unsigned long int sqContentLength=item->vl;
    if(verbose>12) printf("    sequence contents length is %lu\n", sqContentLength);
    /* File position is now at the start of first item in the sequence */
    int ret;
    /* If parent has no previous child, then define this as its child */
    if(sq_child!=0 && parent_item->child_item==NULL) {
      parent_item->child_item=item;
    } else {
      /* else, add SQ sequence itself as next element to the list, and later
         add each sequence item as child to it */
      if(prev_item==NULL) {
        if(dcm->item==NULL) { // truly the first item
          dcm->item=item;
        } else { // search for the previous one
          DCMITEM *ip; ip=dcm->item;
          while(ip->next_item!=NULL) ip=ip->next_item;
          ip->next_item=item; item->prev_item=ip;
        }
      } else {
        prev_item->next_item=item; item->prev_item=prev_item;
      }
    }
    /* Read the first item tag and length, but there is no VR to read this time */
    if(verbose>10) {
      long long int tagpos=ftello(dcm->fp);
      printf(" reading first item tag at %lld\n", tagpos);
    }
    DCMTAG itemtag;
    ret=dcmReadFileTag(dcm->fp, &itemtag);
    if(ret!=0) {
      if(verbose>1) printf(" error %d in reading the tag.\n", ret);
      return(TPCERROR_CANNOT_READ);
    }
    if(verbose>1) printf("  item tag(%04x,%04x)\n", itemtag.group, itemtag.element);
    /* It is common that sequence is empty; check it first */
    if(itemtag.group==0xFFFE && (itemtag.element==0xE0DD || itemtag.element==0xE00D)) {
      /* yes; then read also the 4 byte LV, which should be zero */
      if(verbose>10) 
        printf(" sequence delimitation item tag(%04x,%04x) found, reading VL\n",
          itemtag.group, itemtag.element);
      unsigned long int vl=dcmReadFileVL(dcm->fp, 4); 
      if(verbose>1) printf(" item tag VL := %lu (0x%08lx)\n", vl, vl);
      if(vl!=0) {
        if(verbose>1) printf(" error: VL should have been 0\n");
        return(TPCERROR_INVALID_VALUE);
      }
      if(verbose>3) printf(" ending sequence before it really started.\n");
      return(TPCERROR_OK);
    }
    /* If sequence actually contains something, the Item tag must be 0xFFFE,0xE000 */
    if(itemtag.group!=0xFFFE || itemtag.element!=0xE000) {
      if(verbose>1) printf(" invalid sequence item tag(%04x,%04x)\n", itemtag.group, itemtag.element);
      return(TPCERROR_INVALID_FORMAT); //return(TPCERROR_OK);
    }
    /* Read past the VL of this item (always 4 bytes) */
    unsigned long int itemvl=dcmReadFileVL(dcm->fp, 4);
    if(verbose>3) {printf(" item_VL := %lu (0x%08lx)\n", itemvl, itemvl);}
    if(ftello(dcm->fp)<0) return(TPCERROR_INVALID_FORMAT);
    unsigned long int sqItemPos=(unsigned long int)ftell(dcm->fp);
    /* Check if that is all of this sequence (probably Siemens) */
    if((sqItemPos-sqPos)>=sqContentLength) {
      if(verbose>3) printf(" ending sequence since it was found to be empty.\n");
      return(TPCERROR_OK);
    }
    if(verbose>12) printf("  sequence content start position at %ld\n", sqItemPos);
    /* Read the first item value as its own element, adding it as child to SQ */
    ret=dcmFileReadNextElement(dcm, NULL, item, 1, headerOnly, verbose-1);
    if(ret!=TPCERROR_OK) {
      if(verbose>1) printf(" error in reading the first item value dataset\n");
      return(ret);
    }
    /* Now we continue reading more items, until we reach Sequence Delimitation Item */
    while(!feof(dcm->fp)) {
      /* Do not read pass the length of the sequence data */
      if(ftello(dcm->fp)<0) return(TPCERROR_INVALID_FORMAT);
      unsigned long long int cPos=(unsigned long long int)ftello(dcm->fp);
      if(sqContentLength>0 && (cPos-sqPos)>=sqContentLength) {
        if(verbose>3) printf(" we reached the end of sequence VL %lu\n", sqContentLength);
        /* set fake sequence delimitation tag */
        itemtag.group=0xFFFE; itemtag.element=0xE0DD;
        break;
      }
      if(verbose>10) {
        long long int tagpos=ftello(dcm->fp);
        printf(" reading next sequence item tag at %lld, %lld after start\n", tagpos, tagpos-sqItemPos);
      }
      if(dcmReadFileTag(dcm->fp, &itemtag)) return(TPCERROR_CANNOT_READ);
      if(verbose>1) printf(" next item tag(%04x,%04x)\n", itemtag.group, itemtag.element);
      itemvl=dcmReadFileVL(dcm->fp, 4); // delimitation tag has this too
      if(verbose>3) {printf(" item_VL := %lu (0x%08lx)\n", itemvl, itemvl);}
      /* Check if we got sequence delimitation tag */
      if(itemtag.group==0xFFFE && itemtag.element==0xE0DD) 
      {
        if(verbose>3) printf(" we got sequence delimitation tag\n");
        break;
      }
      /* Check if we got item delimitation tag from the previous item */
      if(itemtag.group==0xFFFE && itemtag.element==0xE00D) 
      {
        if(verbose>3) printf(" we got item delimitation tag\n");
        if(itemvl!=0) {
          if(verbose>1) printf(" error: VL should have been 0\n");
          return(TPCERROR_INVALID_VALUE);
        }
        continue;
      }
      /* Otherwise this should be sequence item tag */
      if(itemtag.group!=0xFFFE || itemtag.element!=0xE000) {
        if(verbose>3) printf(" not sequence item tag, move file position back 2x4 bytes\n");
        fseeko(dcm->fp, -8, SEEK_CUR);  //return(TPCERROR_INVALID_VALUE);
      }
      /* Read the item value as its own element, adding it to the SQ child list */
      DCMITEM *child=item->child_item;
      if(child==NULL) {
        if(verbose>1) printf(" error had happened in adding the child element\n");
        return(TPCERROR_UNSUPPORTED);
      }
      while(child->next_item!=NULL) child=child->next_item;
      ret=dcmFileReadNextElement(dcm, child, item, 0, headerOnly, verbose-1);
      if(ret!=TPCERROR_OK) {
        if(verbose>1) printf(" error in reading item value dataset\n");
        return(ret);
      }
    }
    /* Check that loop really stopped at sequence delimitation item */
    /* 0xE00D means the end of item, 0xE0DD the end of sequence. */ 
    if(itemtag.group!=0xFFFE || itemtag.element!=0xE0DD) {
      if(verbose>1)
        printf(" invalid sequence delimitation item tag(%04x,%04x)\n", itemtag.group, itemtag.element);
      return(TPCERROR_UNSUPPORTED);
    }
    /* Done. Do not free item! */
    if(verbose>10) {printf(" end of sequence.\n"); fflush(stdout);} //return(TPCERROR_OK);
  } else if(item->vl!=0xFFFFFFFF) {
    if(verbose>10) {printf(" reading value of %u bytes...\n", item->vl); fflush(stdout);}
    char *buf=NULL;
    if(item->vl>0) {
      buf=(char*)calloc(item->vl+1, sizeof(char));
      if(buf==NULL) {free(item); return(TPCERROR_OUT_OF_MEMORY);}
      if(fread(buf, 1, item->vl, item->fp)!=item->vl) {
        free(item); free(buf); return(TPCERROR_NO_VALUE);
      }
      /* Do not store pixel data, if that was the request */
      if(headerOnly!=0 && 
         ((item->tag.group==0x7FE0 && item->tag.element>0) || item->tag.group==0x7FE1))
      {
        if(verbose>5) {printf(" ...not storing pixel data\n"); fflush(stdout);}
        free(buf); buf=(char*)NULL;
      } else {
        item->rd=buf;
      }
    } else if(verbose>4) {
      printf(" VL=0\n");
    }
    /* Add to list */
    if(sub==0) {
      if(prev_item==NULL) {
        if(dcm->item==NULL) { // truly the first item
          dcm->item=item;
        } else { // search for the previous one
          DCMITEM *ip; ip=dcm->item;
          while(ip->next_item!=NULL) ip=ip->next_item;
          ip->next_item=item; item->prev_item=ip;
        }
      } else {
        prev_item->next_item=item; item->prev_item=prev_item;
      }
    } else { // add as child
      parent_item->child_item=item; item->parent_item=parent_item;
    }
    /* Print tag and its value */
    if(verbose>1) {
      char *tagval=dcmValueString(item);
      printf(" (%04x,%04x) %s := %s\n", item->tag.group, item->tag.element, 
           dcmDictIndexDescr(dcmDictFindTag(&item->tag)), tagval);
      free(tagval);
    } 
    /* Done. Do no free item or buf! */
    return(TPCERROR_OK);

  } else { // VL=0xFFFFFFFF
    size_t s=dcmVRVLength(item->vr);
    if(s==0) {
      if(verbose>0) printf(" Unknown VL!!\n");
      free(item);
      return(TPCERROR_INVALID_VALUE); //return(TPCERROR_OK);
    }
    if(verbose>4) printf(" VR_based_VL=%u\n", (unsigned int)s);
    char *buf=(char*)calloc(s+1, sizeof(char));
    if(buf==NULL) {free(item); return(TPCERROR_OUT_OF_MEMORY);}
    if(fread(buf, 1, s, item->fp)!=s) {
      free(item); free(buf); return(TPCERROR_NO_VALUE);
    }
    item->rd=buf;
    /* Add to list */
    if(sub==0) {
      if(prev_item==NULL) {
        if(dcm->item==NULL) { // truly the first item
          dcm->item=item;
        } else { // search for the previous one
          DCMITEM *ip; ip=dcm->item;
          while(ip->next_item!=NULL) ip=ip->next_item;
          ip->next_item=item; item->prev_item=ip;
        }
      } else {
        prev_item->next_item=item; item->prev_item=prev_item;
      }
    } else { // add as child
      parent_item->child_item=item; item->parent_item=parent_item;
    }
    /* Print tag and its value */
    if(verbose>1) {
      char *tagval=dcmValueString(item);
      printf(" (%04x,%04x) %s := %s\n", item->tag.group, item->tag.element, 
           dcmDictIndexDescr(dcmDictFindTag(&item->tag)), tagval);
      free(tagval);
    } 
    /* Done. Do no free item or buf! */
    return(TPCERROR_OK);
  }

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read a single DICOM file.
    @sa dcmVerifyMagic, dcmfileInit, dcmfileFree, dcmReadTransferSyntaxUID, dcmFileWrite
    @return Code tpcerror, TPCERROR_OK (0) when successful.
 */
int dcmFileRead(
  /** Pointer to filename. */
  const char *filename,
  /** Pointer to initiated data structure. */
  DCMFILE *dcm,
  /** Read only header (1), or read both header and pixel data (0). */
  const short int headerOnly, 
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(filename==NULL || strnlen(filename, 10)<1 || dcm==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>1) printf("%s('%s', %d)\n", __func__, filename, headerOnly);

  /* Delete any previous data */
  dcmfileFree(dcm);

  /* Open the file */
  strlcpy(dcm->filename, filename, FILENAME_MAX);
  dcm->fp=fopen(dcm->filename, "rb");
  if(dcm->fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_OPEN);
    return TPCERROR_CANNOT_OPEN;
  }

  /* Check the magic number and move file pointer to the end of it */
  if(verbose>2) printf("checking DICOM magic number\n");
  if(dcmVerifyMagic(NULL, dcm->fp)!=1) {
    fclose(dcm->fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Get the Transfer Syntax UID */
  if(verbose>2) printf("checking Transfer Syntax UID\n");
  dcm->truid=dcmReadTransferSyntaxUID(dcm->fp);
  if(dcm->truid==DCM_TRUID_INVALID) { // not found
    fclose(dcm->fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  if(verbose>0) { // print the UID
    printf("Transfer Syntax UID := %s\n", dcmTrUIDDescr(dcm->truid));
    fflush(stdout);
  }

  /* Check whether we currently support the Transfer UID */
  switch(dcm->truid) {
    case DCM_TRUID_LEE:
    case DCM_TRUID_LEI: // actually not supported yet
      break;
    default:
      fclose(dcm->fp);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
      return TPCERROR_UNSUPPORTED;
  }

  /* Read DICOM file elements */
  int ret=TPCERROR_OK;
  do {
    // note that the next function may need to call itself,
    // therefore counting loops here would not be useful.
    ret=dcmFileReadNextElement(dcm, NULL, NULL, 0, headerOnly, verbose-10);
  } while(ret==TPCERROR_OK && !feof(dcm->fp));
  fclose(dcm->fp);
  /* TPCERROR_NO_KEY means that no (more) tag was found;
     other codes still mean that something bad happened. */
  if(ret==TPCERROR_NO_KEY) {
    if(verbose>1) printf(" eof\n");
    ret=TPCERROR_OK;
  }
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write a single DICOM file. 
    @sa dcmFileRead, dcmfileInit, dcmfileFree
    @return Code tpcerror, TPCERROR_OK (0) when successful.
 */
int dcmFileWrite(
  /** Pointer to file name. */
  const char *filename,
  /** Pointer to DICOM data to be written. */
  DCMFILE *dcm,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(filename==NULL || strnlen(filename, 10)<1 || dcm==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>1) {printf("%s('%s')\n", __func__, filename); fflush(stdout);}

  /* Check for the data */
  if(dcm->item==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Check whether we currently support the Transfer UID */
  if(dcm->truid!=DCM_TRUID_LEE) { // Little endian explicit
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return TPCERROR_UNSUPPORTED;
  }


  /* Open the file */
  if(verbose>1) printf("opening the file for writing\n");
  FILE *fp;
  fp=fopen(filename, "wb");
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write preamble (just 128 zeroes) and magic number */
  {
    if(verbose>1) printf("writing preamble\n");
    char buf1[128], buf2[5]; 
    for(int i=0; i<128; i++) buf1[i]=(char)0;
    strcpy(buf2, "DICM");
    if(fwrite(buf1, 128, 1, fp)<1 || fwrite(buf2, 4, 1, fp)<1) {
      fclose(fp); statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
  }


  /* Write the contents */
  if(verbose>1) printf("writing DICOM contents\n");
  int ret=0;
  DCMITEM *iptr;
  DCMITEM *d1=dcm->item;
  while(d1!=NULL) {
    if(verbose>2) {dcmitemPrint(d1);}
    /* Write */
    iptr=d1;
    {
      size_t n;
      /* Write tag */
      if(dcmWriteFileTag(fp, &iptr->tag)!=TPCERROR_OK) {ret=1; break;}
      /* Write VR and VL */
      if(dcmWriteFileVRVL(fp, iptr->vr, iptr->vl, NULL)!=TPCERROR_OK) {ret=2; break;}
      /* Write value, unless zero length, or SQ, in which case written later */
      if(iptr->vl>0 && iptr->vr!=DCM_VR_SQ) {
        size_t len;
        if(iptr->vl==0xFFFFFFFF) {
          len=dcmVRVLength(iptr->vr);
          if(verbose>30) printf("  value_len1 := %u\n", (unsigned int)len);
          n=fwrite(iptr->rd, dcmVRVLength(iptr->vr), 1, fp);
        } else {
          len=iptr->vl;
          if(verbose>30) printf("  value_len3 := %u\n", (unsigned int)len);
          n=fwrite(iptr->rd, iptr->vl, 1, fp);
        }
        if(verbose>30) printf("  value_len := %u\n", (unsigned int)len);
        if(n!=1) {ret=4; break;}
      } else if(iptr->vr==DCM_VR_SQ && d1->child_item==NULL) {
        if(verbose>1) printf("SQ, but no contents to write!\n");
        /* Write Sequence Delimitation Item */
        if(dcmWriteFileSQDelimItem(fp)!=TPCERROR_OK) {ret=6; break;}
      }
    }

    /* If this element (SQ) has children, then write those */
    /* Data Elements with a group of 0000, 0002 and 0006 shall not be present within Sequence Items,
       but that is not verified here */
    if(d1->child_item!=NULL) {
      DCMITEM *d2=d1->child_item;
      unsigned int d2counter=0;
      while(d2!=NULL) {
        if(verbose>2) {printf("  "); dcmitemPrint(d2);}

        /* Write */
        iptr=d2;

        /* First, write Item tag (FFFE,E000) */
        if(d2counter==0) {
          DCMTAG tag; tag.group=0xFFFE; tag.element=0xE000;
          if(dcmWriteFileTag(fp, &tag)!=TPCERROR_OK) {ret=11; break;}
        }
        /* Write item length; write 0xFFFFFFFF for now, correct later when known */
        fpos_t d2ilpos; // position for item length
        unsigned int d2il=0; // item length
        if(d2counter==0) {
          if(fgetpos(fp, &d2ilpos)) {ret=12; break;} // save position for writing later
          unsigned int ibuf;
          ibuf=0xFFFFFFFF;
          if(fwrite(&ibuf, 4, 1, fp)!=1) {ret=13; break;}
        }
        d2counter++;

        /* Write item value data set */
        {
          /* Write tag */
          if(dcmWriteFileTag(fp, &iptr->tag)!=TPCERROR_OK) {ret=14; break;}
          d2il+=4;
          /* Write VR and VL */
          unsigned int s;
          if(dcmWriteFileVRVL(fp, iptr->vr, iptr->vl, &s)!=TPCERROR_OK) {ret=15; break;}
          d2il+=s;
          /* Write value, unless zero length, or SQ, in which case written later */
          if(iptr->vl>0 && iptr->vr!=DCM_VR_SQ) {
            size_t len, n;
            if(iptr->vl==0xFFFFFFFF) {
              len=dcmVRVLength(iptr->vr);
              if(verbose>30) printf("  value_len1 := %u\n", (unsigned int)len);
              n=fwrite(iptr->rd, dcmVRVLength(iptr->vr), 1, fp);
              d2il+=len;
            } else {
              len=iptr->vl;
              if(verbose>30) printf("  value_len3 := %u\n", (unsigned int)len);
              n=fwrite(iptr->rd, iptr->vl, 1, fp);
              d2il+=iptr->vl;
            }
            if(verbose>30) printf("  value_len := %u\n", (unsigned int)len);
            if(n!=1) {ret=17; break;}
          } else if(iptr->vr==DCM_VR_SQ && iptr->child_item==NULL) {
            if(verbose>1) printf("SQ, but no contents to write!\n");
            /* Write Sequence Delimitation Item */
            if(dcmWriteFileSQDelimItem(fp)!=TPCERROR_OK) {ret=19; break;}
          }
        }

        /* If this element has children, then write those */
        if(d2->child_item!=NULL) {
          DCMITEM *d3=d2->child_item;
          unsigned int d3counter=0;
          while(d3!=NULL) {
            if(verbose>2) {printf("    "); dcmitemPrint(d3);}

            /* Write */
            iptr=d3;
            if(iptr->vr==DCM_VR_SQ) {d3=d3->next_item; continue;} // for now do not write SQs

            /* First, write Item tag (FFFE,E000) */
            if(d3counter==0) {
              DCMTAG tag; tag.group=0xFFFE; tag.element=0xE000;
              if(dcmWriteFileTag(fp, &tag)!=TPCERROR_OK) {ret=31; break;}
              d2il+=4;
            }
            /* Write item length; write 0xFFFFFFFF for now, correct later when known */
            fpos_t d3ilpos; // position for item length
            unsigned int d3il=0; // item length
            if(d3counter==0) {
              unsigned int ibuf;
              if(fgetpos(fp, &d3ilpos)) {ret=32; break;} // save position for writing later
              ibuf=0xFFFFFFFF;
              if(fwrite(&ibuf, 4, 1, fp)!=1) {ret=33; break;}
              d2il+=4;
            }
            d3counter++;

            /* Write item value data set */
            {
              /* Write tag */
              if(dcmWriteFileTag(fp, &iptr->tag)!=TPCERROR_OK) {ret=34; break;}
              d3il+=4; d2il+=4;
              /* Write VR and VL */
              unsigned int s;
              if(dcmWriteFileVRVL(fp, iptr->vr, iptr->vl, &s)!=TPCERROR_OK) {ret=35; break;}
              d3il+=s; d2il+=s;
              /* Write value, unless zero length, or SQ, in which case written later */
              if(iptr->vl>0 && iptr->vr!=DCM_VR_SQ) {
                size_t len, n;
                if(iptr->vl==0xFFFFFFFF) {
                  len=dcmVRVLength(iptr->vr);
                  if(verbose>30) printf("  value_len1 := %u\n", (unsigned int)len);
                  n=fwrite(iptr->rd, dcmVRVLength(iptr->vr), 1, fp);
                  d3il+=len; d2il+=len;
                } else {
                  len=iptr->vl;
                  if(verbose>30) printf("  value_len3 := %u\n", (unsigned int)len);
                  n=fwrite(iptr->rd, iptr->vl, 1, fp);
                  d3il+=iptr->vl; d2il+=iptr->vl;
                }
                if(verbose>30) printf("  value_len := %u\n", (unsigned int)len);
                if(n!=1) {ret=37; break;}
              } else if(iptr->vr==DCM_VR_SQ && iptr->child_item==NULL) {
                if(verbose>1) printf("SQ, but no contents to write!\n");
                /* Write Sequence Delimitation Item */
                if(dcmWriteFileSQDelimItem(fp)!=TPCERROR_OK) {ret=39; break;}
                d2il+=8;
              }
            }

            /* If this element has children, then write those */
            if(d3->child_item!=NULL) {
              //DCMITEM *d4=d3->child_item;
              if(verbose>0) fprintf(stderr, "Warning: 4th level items not written.\n");
            }

            /* now that we known the length, write it to the saved position */
            if(0) {
              fpos_t opos; // current position to return to
              if(fgetpos(fp, &opos)) {ret=40; break;}
              fsetpos(fp, &d3ilpos); // go to the position for item length
              char buf[4];
              memcpy(buf, &d3il, 4);
              if(!endianLittle()) swabip(buf, 4);
              if(fwrite(&buf, 4, 1, fp)!=1) {ret=41; break;}
              fsetpos(fp, &opos); // back to the position where we were
            }

            d3=d3->next_item;
          }
          if(ret!=0) break;

          /* Write Item Delimitation Tag */
          if(dcmWriteFileSQItemDelimTag(fp)!=0) {ret=21; break;}
          /* End of the sequence - write Sequence Delimitation Item */
          if(dcmWriteFileSQDelimItem(fp)!=TPCERROR_OK) {ret=21; break;}

        }


        /* now that we known the length, write it to the saved position */
        if(0) {
          fpos_t opos; // current position to return to
          if(fgetpos(fp, &opos)) {ret=18; break;}
          fsetpos(fp, &d2ilpos); // go to the position for item length
          char buf[4];
          memcpy(buf, &d2il, 4);
          if(!endianLittle()) swabip(buf, 4);
          if(fwrite(&buf, 4, 1, fp)!=1) {ret=19; break;}
          fsetpos(fp, &opos); // back to the position where we were
        }

        d2=d2->next_item;
      }
      if(ret!=0) break;

      /* Write Item Delimitation Tag */
      if(dcmWriteFileSQItemDelimTag(fp)!=0) {ret=21; break;}
      /* End of the sequence - write Sequence Delimitation Item */
      if(dcmWriteFileSQDelimItem(fp)!=TPCERROR_OK) {ret=21; break;}

    }

    d1=d1->next_item;
  } // next
  if(ret!=0) {
    if(verbose>0) fprintf(stderr, "  ret := %d\n", ret);
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }


  fclose(fp);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
